//
// Copyright 2003 - 2008 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util.regex;

/**
 * This regular expression component matches any character. This
 * pattern element has an associated minimum and maximum range.
 * The component must match at least the minimum count and up
 * to the maximum count (inclusive).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class AnyChar
    extends Component
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a any character component with the specified
     * minimum and maximum range.
     * @param minSize Must match at least this many times.
     * @param maxSize May match up to this many times.
     * @param index Component's index in the pattern.
     */
    /* package */ AnyChar(final int minSize,
                          final int maxSize,
                          final int index)
    {
        super (ANY_CHAR, minSize, maxSize, index);
    } // end of AnyChar(int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Component Abstract Methods Implementation.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false}
     * otherwise. Any character is always less than the specified
     * character <i>if</i> {@code c} is not the minimally
     * allowed character value.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean lessThan(final char c)
    {
        return (c > Character.MIN_VALUE);
    } // end of lessThan(char)

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * Any character is always equal to the specified
     * character.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean equalTo(final char c)
    {
        return (true);
    } // end of equalTo(char)

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * Any character is always greater than the specified
     * character <i>if</i> {@code c} is not the maximally allowed
     * character.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean greaterThan(final char c)
    {
        return (c < Character.MAX_VALUE);
    } // end of greaterThan(char)

    //
    // end of Component Abstract Methods Implementation.
    //-----------------------------------------------------------

    /**
     * Returns a textual representation of a single character
     * match.
     * @return a textual representation of a single character
     * match.
     */
    @Override
    public String toString()
    {
        final StringBuilder buffer = new StringBuilder();

        buffer.append('.');
        appendSize(buffer);

        return (buffer.toString());
    } // end of toString()
} // end of class AnyChar
