//
// Copyright 2009 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.util;

import java.util.Map;
import java.util.WeakHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * When using a thread pool to execute tasks there is no
 * guarantee the tasks will be executed in the same sequential
 * order as offered to the pool. If the tasks are independent of
 * each other, then order is unimportant. But if the tasks are
 * chronologically dependent and must be exected in order, then
 * the lack of guaranteed order is problematic.
 * <p>
 * {@code SequentialTask} offers a solution between a dedicated
 * event thread and a thread pool. Tasks associated <i>with the
 * same object</i> are forced to execute in sequential order.
 * Each task is assigned an index based on its associated object
 * and executes only when its index becomes the current index.
 * Otherwise, the task waits for its index to be reached. When
 * the task completes, the current index is incremented and all
 * waiting sequential tasks are notified of the new index.
 * </p>
 * <p>
 * By associating the task index with an object, it is possible
 * for multiple tasks associated with different objects to
 * execute simultaneously. Only tasks associated with the same
 * object are forced to execute serially. A weak hash map is used
 * to associate an object with its task index values. This allows
 * objects passed to
 * {@link #createTask(Object, Runnable) createTask} to be
 * garbage collected and automatically removed from the task
 * object index map.
 * </p>
 * <h3>Example Use: The Observer Pattern</h3>
 * In the Observer Pattern (GoF pp. 293 - 299) the Subject class'
 * {@code Notify()} method calls back observers:
 * <pre>
 *   <code>
 * public void Notify()
 * {
 *     for (Observers o: observers)
 *     {
 *         o.Update();
 *     }
 * }
 *   </code>
 * </pre>
 * This callback can be converted from updating each observer in
 * turn to a multi-threaded approach using a
 * {@link java.util.concurrent.ExecutorService thread pool} to
 * execute each callback:
 * <pre>
 *   <code>
 * public void Notify()
 * {
 *     Iterator&lt;Observer&gt; oit;
 *
 *     for (oit = observers.iterator(); oit.hasNext();)
 *     {
 *         // The run task requires that the Observer local
 *         // variable be final. Hence using an iterator and
 *         // declaring the variable within the for body.
 *         final Observer o = oit.next();
 *
 *         threadPool.execute(
 *             new Runnable()
 *             {
 *                 &#64;Override
 *                 public void run()
 *                 {
 *                     o.Update();
 *                 }
 *             });
 *     }
 * }
 *   </code>
 * </pre>
 * The above code is acceptable if callback ordering is not a
 * factor. Consider the following code;
 * <pre>
 *   <code>
 * // s0 and s1 are both Subject instances. Assume that their
 * // respective observer lists have some observer instances in
 * // common (that is, a non-null intersection of observers).
 * s0.Notify();
 * s1.Notify()
 *   </code>
 * </pre>
 * If it does not matter if {@code s1} updates an observer before
 * {@code s0}'s update for the same observer, then using the
 * above thread pool solution is acceptable. But if {@code s0}'s
 * update must occur before {@code s1}'s update, then the threads
 * must be serialized. {@code SequentialTask} provides this
 * capability:
 * <pre>
 *   <code>
 * public void Notify()
 * {
 *     Iterator&lt;Observer&gt; oit;
 *
 *     for (oit = observers.iterator(); oit.hasNext();)
 *     {
 *         // The run task requires that the Observer local
 *         // variable be final. Hence using an iterator and
 *         // declaring the variable within the for body.
 *         final Observer o = oit.next();
 *
 *         threadPool.execute(
 *             SequentialTask.createTask(
 *                 o,
 *                 new Runnable()
 *                 {
 *                     &#64;Override
 *                     public void run()
 *                     {
 *                         o.Update();
 *                     }
 *                 }
 *             });
 *     }
 * }
 *   </code>
 * </pre>
 * {@code s0}'s call to {@code o.Update()} is guaranteed to
 * occur before {@code s1}'s {@code o.Update()}. However calls
 * to {@code Update()} for different observers can execute in
 * parallel. Only updates to the same observer are serialized.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class SequentialTask
    implements Runnable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    // Maps objects to an associated task index. A weak
    // referencing map is used to allow the objects garbage
    // collection.
    private static final Map<Object, TaskIndex> sIndexMap =
        new WeakHashMap<>();

    // Provides exclusive access to the index map.
    private static final Lock sIndexLock =
        new ReentrantLock(true);

    //-----------------------------------------------------------
    // Locals.
    //

    // The task's assigned index. Wait for this index to be
    // reached before executing.
    private final int mIndex;

    // Query this object for the current index.
    private final TaskIndex mTaskIndex;

    // When this task's number is called, execute this task.
    private final Runnable mTask;

    // If the task throws an exception, the report the problem
    // to this logger at the specified level.
    private final Logger mLogger;
    private final Level mLogLevel;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    // Creates a new instance of SequentialTask for the given
    // task index and task. The logger and log level are
    // optional.
    private SequentialTask(final TaskIndex taskIndex,
                           final Runnable task,
                           final Logger logger,
                           final Level logLevel)
    {
        mIndex = taskIndex.assignIndex();
        mTaskIndex = taskIndex;
        mTask = task;
        mLogger = logger;
        mLogLevel = logLevel;
    } // end of SequentialTask(TaskIndex,Runnable,Logger,Level)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Runnable Interface Implementation.
    //

    /**
     * Synchronizes on the task index waiting for its turn to
     * run the user task. Once the index is reached, the user
     * task is executed. Once execution completes, the task index
     * is incremented and waiting sequential tasks are notified
     * concerning the new current index.
     * <p>
     * If the user task throws an exception, it is caught and
     * reported via the supplied logger at the specified level.
     */
    @Override
    public void run()
    {
        synchronized (mTaskIndex)
        {
            // Wait for it!
            while (mIndex != mTaskIndex.currentIndex())
            {
                try
                {
                    mTaskIndex.wait();
                }
                catch (InterruptedException interrupt)
                {
                    // Ignore.
                }
            }

            // GO!
            try
            {
                mTask.run();
            }
            catch (Exception jex)
            {
                if (mLogger != null &&
                    mLogger.isLoggable(mLogLevel))
                {
                    mLogger.log(
                        mLogLevel, "task exception", jex);
                }
            }

            // Increment the current task index and then notify
            // all waiting sequential tasks of the fact.
            mTaskIndex.incrementIndex();
            mTaskIndex.notifyAll();
        }
    } // end of run()

    //
    // end of Runnable Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the task's run index.
     * @return the task's run index.
     */
    public int index()
    {
        return (mIndex);
    } // end of index()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Creates a sequential task for the given object and task.
     * If {@code task} throws an exception, it will be caught and
     * quietly ignored.
     * @param o The task is associated with this object.
     * @param task Runs this user task.
     * @return A sequential task associated with {@code o} and
     * executes {@code task}.
     * @exception IllegalArgumentException
     * if either {@code o} or {@code task} is {@code null}.
     */
    public static SequentialTask createTask(final Object o,
                                            final Runnable task)
    {
        return (createTask(o, task, null, Level.OFF));
    } // end of createTask(Object, Runnable)

    /**
     * Creates a sequential task for the given object and task.
     * If {@code task} throws an exception, it will be logged
     * to {@code logger} at {@code logLevel}.
     * @param o The task is associated with this object.
     * @param task Runs this user task.
     * @param logger If {@code task} throws an exception, then
     * report it via this logger.
     * @param logLevel Report {@code task} exception at this
     * log level.
     * @return A sequential task associated with {@code o} and
     * executes {@code task}.
     * @exception IllegalArgumentException
     * if either {@code o} or {@code task} is {@code null}.
     */
    public static SequentialTask createTask(final Object o,
                                            final Runnable task,
                                            final Logger logger,
                                            final Level logLevel)
    {
        SequentialTask retval = null;

        if (o == null)
        {
            throw (new IllegalArgumentException("null o"));
        }
        else if (task == null)
        {
            throw (new IllegalArgumentException("null task"));
        }
        else
        {
            sIndexLock.lock();
            try
            {
                TaskIndex taskIndex;

                if (!sIndexMap.containsKey(o))
                {
                    taskIndex = new TaskIndex();
                    sIndexMap.put(o, taskIndex);
                }
                else
                {
                    taskIndex = sIndexMap.get(o);
                }

                retval =
                    new SequentialTask(taskIndex,
                                       task,
                                       logger,
                                       logLevel);
            }
            finally
            {
                sIndexLock.unlock();
            }
        }

        return (retval);
    } // end of createTask(Object, Runnable, Logger, Level)

//---------------------------------------------------------------
// Inner classes.
//

    // Stores the current and assignment indices for a particular
    // Java object. The task index is from
    // [0, Integer.MAX_VALUE).
    private static final class TaskIndex
    {
    //-----------------------------------------------------------
    // Member data.
    //

        // The task with this index runs next.
        private int mCurrentIndex;

        // Assign a new task this index.
        private int mAssignIndex;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private TaskIndex()
        {
            mCurrentIndex = 0;
            mAssignIndex = 0;
        } // end of TaskIndex()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        // Returns the currently active task index.
        private int currentIndex()
        {
            return (mCurrentIndex);
        } // end of currentIndex()

        // Returns the next available assigned index for a new
        // task. This method is synchronized using the static
        // index lock rather than synchronizing this task index
        // instance.
        private int assignIndex()
        {
            final int retval = mAssignIndex;

            ++mAssignIndex;
            mAssignIndex %= Integer.MAX_VALUE;

            return (retval);
        } // end of assignIndex()

        //
        // end of Get methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set methods.
        //

        private void incrementIndex()
        {
            ++mCurrentIndex;
            mCurrentIndex %= Integer.MAX_VALUE;
        } // end of incrementIndex()

        //
        // end of Set methods.
        //-------------------------------------------------------
    } // end of class TaskIndex
} // end of class SequentialTask
