//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2003 - 2010. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util.regex;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.Queue;
import java.util.regex.PatternSyntaxException;

/**
 * A compiled regular expression representation.
 * <p>
 * Patterns are created indirectly by calling
 * {@link #compile(java.lang.String)} and passing in the
 * regular expression text. If the regular expression is
 * successfully compiled, then a {@code Pattern} instance is
 * returned. Use the {@link #matches(java.lang.CharSequence)}
 * method to test if a character sequence matches the regular
 * expression.
 * <p>
 * Pattern instance are immutable and safe for concurrent thread
 * usage.
 * </p>
 * <h3>Summary of regular-expression constructs</h3>
 * <table border="0" cellpadding="1" cellspacing="0" summary="Regular expression constructs, and what they match">
 *   <tr align="left">
 *     <th align="left" id="construct">
 *       Construct
 *     </th>
 *     <th align="center" id="matches">
 *       Matches
 *     </th>
 *   </tr>
 *   <tr><th colspan="2">&nbsp;</th></tr>
 *   <tr align="left">
 *     <th colspan="2" id="characters">
 *       Characters
 *     </th>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>x</i>
 *     </td>
 *     <td headers="matches">
 *       The character <i>x</i>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\\</tt>
 *     </td>
 *     <td headers="matches">
 *       The backslash character
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\0</tt><i>n</i>
 *     </td>
 *     <td headers="matches">
 *       The character with octal value 0<i>n</i>
 *       (0 &le; <i>n</i> &le; 7)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\0</tt><i>nn</i>
 *     </td>
 *     <td headers="matches">
 *       The character with octal value 0<i>nn</i>
 *       (0 &le; <i>n</i> &le; 7)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\0</tt><i>mnn</i>
 *     </td>
 *     <td headers="matches">
 *       The character with octal value 0<i>mnn</i>
 *       (0 &le; <i>m</i> &le; 3, 0 &le; <i>n</i> &le; 7)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\x</tt><i>hh</i>
 *     </td>
 *     <td headers="matches">
 *       The character with hexadecimal value 0x<i>hh</i>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\u005Cu</tt><i>hhhh</i>
 *     </td>
 *     <td headers="matches">
 *       The character with hexadecimal value 0x<i>hhhh</i>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\t</tt>
 *     </td>
 *     <td headers="matches">
 *       The tab character (<tt>'&#92;u0009'</tt>)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\n</tt>
 *     </td>
 *     <td headers="matches">
 *       The newline (line feed) character (<tt>'&#92;u000A'</tt>)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\r</tt>
 *     </td>
 *     <td headers="matches">
 *       The carriage-return character (<tt>'&#92;u000D'</tt>)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\f</tt>
 *     </td>
 *     <td headers="matches">
 *       The form-feed character (<tt>'&#92;u000C'</tt>)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\a</tt>
 *     </td>
 *     <td headers="matches">
 *       The alert (bell) character (<tt>'&#92;u0007'</tt>)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\e</tt>
 *     </td>
 *     <td headers="matches">
 *       The escape character (<tt>'&#92;u001B'</tt>)
 *     </td>
 *   </tr>
 *   <tr><th>&nbsp;</th></tr>
 *   <tr align="left">
 *     <th colspan="2" id="classes">
 *       Character classes
 *     </th>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>[abc]</tt>
 *     </td>
 *     <td headers="matches">
 *       <tt>a</tt>, <tt>b</tt>, or <tt>c</tt> (simple class)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>[^abc]</tt>
 *     </td>
 *     <td headers="matches">
 *       Any character except <tt>a</tt>, <tt>b</tt>, or
 *       <tt>c</tt> (negation)
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>[a-zA-Z]</tt>
 *     </td>
 *     <td headers="matches">
 *       <tt>a</tt> through <tt>z</tt> or <tt>A</tt> through
 *       <tt>Z</tt>, inclusive (range)
 *     </td>
 *   </tr>
 *   <tr><th>&nbsp;</th></tr>
 *   <tr align="left">
 *     <th colspan="2">
 *       Predefined character classes
 *     </th>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>.</tt>
 *     </td>
 *     <td headers="matches">
 *       Any character.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\d</tt>
 *     </td>
 *     <td headers="matches">
 *       A digit: <tt>[0-9]</tt>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\D</tt>
 *     </td>
 *     <td headers="matches">
 *       A non-digit: <tt>[^0-9]</tt>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\s</tt>
 *     </td>
 *     <td headers="matches">
 *       A whitespace character: <tt>[ \t\n\0x0B\f\r]</tt>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\S</tt>
 *     </td>
 *     <td headers="matches">
 *       A non-whitespace character: <tt>[^\s]</tt>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\w</tt>
 *     </td>
 *     <td headers="matches">
 *       A word character: <tt>[a-zA-Z_0-9]</tt>
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <tt>\W</tt>
 *     </td>
 *     <td headers="matches">
 *       A non-word character: <tt>[^\w]</tt>
 *     </td>
 *   </tr>
 *   <tr><th>&nbsp;</th></tr>
 *   <tr align="left">
 *     <th colspan="2">
 *       Greedy quantifiers
 *     </th>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>?
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, once or not at all.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>*
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, zero or more times.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>+
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, one or more times.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>{<i>n</i>}
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, exactly <i>n</i> times.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>{<i>n</i>,}
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, at least <i>n</i> times.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>{, <i>n</i>}
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, zero to at most <i>n</i> times.
 *     </td>
 *   </tr>
 *   <tr>
 *     <td valign="top" headers="construct">
 *       <i>X</i>{<i>n</i>, <i>m</i>}
 *     </td>
 *     <td headers="matches">
 *       <i>X</i>, at least <i>n</i> times but no more than
 *       <i>m</i> times where <i>n</i> &le; <i>m</i>.
 *     </td>
 *   </tr>
 * </table>
 * <hr>
 * <p>
 * This regular expression class is specifically designed to
 * efficiently search
 * {@link net.sf.eBus.util.TernarySearchTree} keys. Hence the
 * reason for developing a separate regular expression package
 * from {@code java.util.regex}. If you
 * need to perform regular expression pattern matching over
 * generic character sequences, then use the
 * {@code java.util.regex.Pattern} class which supports
 * more extensive regular expression constructs.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 * @see net.sf.eBus.util.TernarySearchTree#entrySet(Pattern)
 * @see net.sf.eBus.util.TernarySearchTree#entrySet(Pattern, int)
 * @see net.sf.eBus.util.TernarySearchTree#keySet(Pattern)
 * @see net.sf.eBus.util.TernarySearchTree#keySet(Pattern, int)
 * @see net.sf.eBus.util.TernarySearchTree#values(Pattern)
 * @see net.sf.eBus.util.TernarySearchTree#values(Pattern, int)
 */

public final class Pattern
    implements Comparable<Pattern>,
               Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    // The original regular expression pattern string.
    private final String mPattern;

    // The subordinate regular expression components.
    private final Component[] mComponents;

    // To match, strings must be at least the minimum size
    // and at most the maximum size.
    private final int mMinimumSize;
    private final int mMaximumSize;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a regular expression pattern from the components
     * list.
     * @param pattern the original regular expression pattern.
     * @param components regular expression components.
     * @param minSize the minimal accepted string size.
     * @param maxSize the maximal accept string size.
     */
    private Pattern(final String pattern,
                    final Component[] components,
                    final int minSize,
                    final int maxSize)
    {
        mPattern = pattern;
        mComponents = components;
        mMinimumSize = minSize;
        mMaximumSize = maxSize;
    } // end of Pattern(String, Component[], int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Comparable Interface Implementation.
    //

    /**
     * Returns an integer value &lt; zero if this pattern is
     * less than the argument; zero if this pattern is equal
     * to the argument; &gt; zero if this pattern is greater
     * than the argument.
     * @param pattern Compare agains this pattern
     * @return an integer value &lt; zero if this pattern is
     * less than the argument; zero if this pattern is equal
     * to the argument; &gt; zero if this pattern is greater
     * than the argument.
     */
    @Override
    public int compareTo(final Pattern pattern)
    {
        return (mPattern.compareTo(pattern.mPattern));
    } // end of compareTo(Pattern)

    //
    // end of Comparable Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the original regular expression pattern string.
     * @return the original regular expression pattern string.
     */
    public String pattern()
    {
        return (mPattern);
    } // end of pattern()

    /**
     * Returns the pattern's components.
     * @return the pattern's components.
     */
    public Component[] components()
    {
        return (Arrays.copyOf(mComponents, mComponents.length));
    } // end of components()

    /**
     * Returns a matching string's minimum possible size.
     * @return a matching string's minimum possible size.
     */
    public int minimumSize()
    {
        return (mMinimumSize);
    } // end of minimumSize()

    /**
     * Returns a matching string's maximum possible size.
     * {@code -1} means there is no limit.
     * @return a matching string's maximum possible size.
     */
    public int maximumSize()
    {
        return (mMaximumSize);
    } // end of maximumSize()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns {@code true} if this regular expression
     * pattern accepts {@code input} and {@code false} otherwise.
     * @param input match this character sequence against this
     * regular expression pattern.
     * @return {@code true} if this regular expression
     * pattern accepts {@code input} and {@code false} otherwise.
     */
    public boolean matches(final CharSequence input)
    {
        final int length = input.length();
        boolean retcode =
            (length >= mMinimumSize &&
             (length <= mMaximumSize ||
              mMaximumSize == Component.NO_MAX_MATCH_LIMIT));

        if (retcode == true)
        {
            final Queue<MatchPly> queue = new LinkedList<>();

            queue.add(new MatchPly(0, 0, 0));
            retcode =
                matches(
                    input, length, mComponents.length, queue);
        }

        return (retcode);
    } // end of matches(final CharSequence)

    /**
     * Returns {@code true} if this object and the argument
     * {@code object} are both regular expression patterns
     * with the same underlying pattern string and
     * {@code false} otherwise.
     * @param object test equality with this object.
     * @return {@code true} if this object and the argument
     * {@code object} are both regular expression patterns
     * with the same underlying pattern string and
     * {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object object)
    {
        boolean retcode = (this == object);

        if (retcode == false &&
            object != null &&
            object instanceof Pattern)
        {
            retcode =
                mPattern.equals(((Pattern) object).mPattern);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns this pattern's textual representation.
     * @return this pattern's textual representation.
     */
    @Override
    public String toString()
    {
        final StringBuilder buffer = new StringBuilder();
        int index;

        for (index = 0; index < mComponents.length; ++index)
        {
            buffer.append(mComponents[index]);
        }
        buffer.append(" (min size=");
        buffer.append(mMinimumSize);
        buffer.append(", max size=");
        if (mMaximumSize == Component.NO_MAX_MATCH_LIMIT)
        {
            buffer.append("unlimited");
        }
        else
        {
            buffer.append(mMaximumSize);
        }
        buffer.append(")");

        return (buffer.toString());
    } // end of toString()

    /**
     * Returns a hosh code based on the regular expression
     * pattern string.
     * @return a hosh code based on the regular expression
     * pattern string.
     */
    @Override
    public int hashCode()
    {
        return (mPattern.hashCode());
    } // end of hashCode()

    /**
     * Returns the eBus regular expression pattern compiled from
     * the given string.
     * @param regex compile this string into an eBus pattern.
     * @return the eBus regular expression pattern compiled from
     * the given string.
     * @exception IllegalArgumentException
     * if {@code regex} is {@code null}.
     * @exception PatternSyntaxException
     * if {@code regex}'s syntax is invalid.
     */
    public static Pattern compile(final String regex)
        throws IllegalArgumentException,
               PatternSyntaxException
    {
        final Collection<Component> components =
            new ArrayList<>();
        Component[] componentsArray;
        RegexLexer lexer;
        RegexLexer.Token token;
        int type;
        int index = 0;
        int size;
        int minSize = 0;
        int maxSize = 0;

        // A null string is not a valid pattern.
        if (regex == null)
        {
            throw (new IllegalArgumentException("null regex"));
        }
        // An empty string is not a valid pattern.
        else if (regex.length() == 0)
        {
            throw (
                new PatternSyntaxException(
                    "regex empty",
                    regex,
                    0));
        }

        lexer = new RegexLexer(regex);
        for  (token = lexer.nextToken(), type = token.type();
              type == RegexLexer.REGEX_COMPONENT;
              token = lexer.nextToken(), type = token.type())
        {
            components.add(token.regexComponent());
        }

        if (type == RegexLexer.DONE_FAILED)
        {
            final PatternSyntaxException t =
                new PatternSyntaxException(
                    token.errorMessage(),
                    regex,
                    token.index());

            t.initCause(token.cause());

            throw (t);
        }
        // Else the parse succeeded.

        // A pattern which accepts nothing is *not*
        // acceptable. Add up the components' minimum and
        // maximum sizes.
        componentsArray = new Component[components.size()];
        for (Component component: components)
        {
            componentsArray[index++] = component;

            minSize += component.minimumSize();
            if ((size = component.maximumSize()) < 0)
            {
                maxSize = Component.NO_MAX_MATCH_LIMIT;
            }
            // Once maximum size is set to no limit, there is
            // no point in continuing to add the new size.
            else if (maxSize >= 0)
            {
                maxSize += size;
            }
        }

        components.clear();

        return (
            new Pattern(
                regex, componentsArray, minSize, maxSize));
    } // end of compile(String)

    /**
     * Performs the actual pattern matching using a search queue.
     * <ul>
     *   <li>
     *     ii: input index.
     *   </li>
     *   <li>
     *     iLength: input length (so input.length() called once).
     *   </li>
     *   <li>
     *     ci: component index.
     *   </li>
     *   <li>
     *     cLength: component length.
     *   </li>
     *   <li>
     *     mc: match count (current component matched this many
     *     times).
     *   </li>
     * </ul>
     * @param input the character sequence matched against the
     * pattern.
     * @param iLength the input length.
     * @param cLength the number of components in the pattern.
     * @param queue the search queue.
     * @return {@code true} if the input matches this pattern;
     * {@code false} otherwise.
     */
    private boolean matches(final CharSequence input,
                            final int iLength,
                            final int cLength,
                            final Queue<MatchPly> queue)
    {
        MatchPly ply;
        int ii;
        int ci;
        int mc;
        int ii2;
        int ci2;
        int mc2;
        int minSize;
        int maxSize;
        boolean retcode = false;

        while (retcode == false && (ply = queue.poll()) != null)
        {
            ii = ply.inputIndex();
            ci = ply.componentIndex();
            mc = ply.matchCount();

            // Are we at the input's end?
            if (ii == iLength)
            {
                // Yes. Are we at the component's end?
                if (ci == cLength)
                {
                    // Yes. We have a winner!
                    retcode = true;
                }
                // If we have satisfied the current component,
                // then go an to the next even if we are at the
                // input's end. Why?
                // Because the next component may accept
                // zero-sized input.
                else if (mc >= mComponents[ci].minimumSize())
                {
                    queue.add(new MatchPly(ii, (ci + 1), 0));
                }
                // else we reached the input's end but not the
                // component's end: no match.
            }
            // If there is at least one more component and that
            // component either accepts the current input or has
            // zero minimum size, then keep matching.
            else if (ci < cLength &&
                     (mComponents[ci].minimumSize() == 0 ||
                      mComponents[ci].equalTo(
                          input.charAt(ii)) == true))
            {
                ii2 = (ii + 1);
                ci2 = (ci + 1);
                mc2 = (mc + 1);
                minSize = mComponents[ci].minimumSize();
                maxSize = mComponents[ci].maximumSize();

                // There are three cases to check:
                // 1. Same component, next character index.
                //    If the component is not fully satisfied by
                //    the character, then remain at this
                //    component and move to the next character to
                //    see if that satisfies component.
                //
                // 2. Next component, next character index.
                //    The component is satisfied by this
                //    character. Move on to the next.
                //
                // 3. Next component, same character index.
                //    The component is more than satisfied by
                //    this character. So move on to the next
                //    component and see if it can also consume
                //    this character.
                //
                // Same component, same character index is not
                // a case because that may cause an infinite
                // loop.

                // Have we maximally matched the current
                // component?
                if (mc2 < maxSize ||
                    maxSize == Component.NO_MAX_MATCH_LIMIT)
                {
                    // No. Continue with the current component.
                    queue.add(new MatchPly(ii2, ci, mc2));
                }

                // Have we minimally matched the current
                // component?
                if (mc2 >= minSize)
                {
                    // Yes. Move on to the next component and
                    // input index.
                    queue.add(new MatchPly(ii2, ci2, 0));
                }

                // Have we matched more than the minimum size?
                if (mc2 > minSize)
                {
                    // Yes. Present the same input to the
                    // next component.
                    queue.add(new MatchPly(ii, ci2, 0));
                }
            }
            // else we are at the regex's end but not at the
            // input's end: no match.
            // OR
            // the current input simply doesn't match the current
            // component.
        }

        return (retcode);
    } // end of matches(...)

//---------------------------------------------------------------
// Inner classes.
//

    // Used to store the next pattern match ply.
    private static final class MatchPly
    {
    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        public MatchPly(final int inputIndex,
                        final int componentIndex,
                        final int matchCount)
        {
            _inputIndex = inputIndex;
            _componentIndex = componentIndex;
            _matchCount = matchCount;
        } // end of MatchPly(int, int, int)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        public int inputIndex()
        {
            return (_inputIndex);
        } // end of inputIndex()

        public int componentIndex()
        {
            return (_componentIndex);
        } // end of componentIndex()

        public int matchCount()
        {
            return (_matchCount);
        } // end of matchCount()

        //
        // end of Get methods.
        //-------------------------------------------------------

    //-----------------------------------------------------------
    // Member data.
    //

        private final int _inputIndex;
        private final int _componentIndex;
        private final int _matchCount;
    } // end of class MatchPly
} // end of class Pattern

//
// CHANGE LOG
// $Log: Pattern.java,v $
// Revision 1.13  2007/11/27 19:20:17  charlesr
// Implemented Comparable<Pattern> interface.
//
// Revision 1.12  2007/03/10 02:41:45  charlesr
// Corrected matches() algorithm.
//
// Revision 1.11  2007/03/08 21:01:05  charlesr
// Replaced recursive pattern matching with iterative
// matching using a queue.
//
// Revision 1.10  2007/02/23 13:37:51  charlesr
// Corrected javadoc comments.
//
// Revision 1.9  2006/10/22 17:10:35  charlesr
// Replaced Component.matches() call with Component.equalTo().
//
// Revision 1.8  2006/10/20 19:42:27  charlesr
// Made all data members "final".
//
// Revision 1.7  2006/10/20 19:37:26  charlesr
// Added components() method.
//
// Revision 1.6  2006/10/20 19:23:10  charlesr
// Modified to match new regular expression syntax.
//
// Revision 1.5  2006/09/02 14:40:28  charlesr
// Javadoc corrections.
//
// Revision 1.4  2005/06/21 11:43:02  charlesr
// Check-in prior to Java 5 update.
//
// Revision 1.3  2004/09/29 16:02:19  charlesr
// Added hashCode method to override the
// java.lang.Object.hashCode.
//
// Revision 1.2  2004/08/15 21:18:41  charlesr
// Corrected javadocs.
//
// Revision 1.1  2004/07/25 16:03:13  charlesr
// Corrected javadoc comments.
//
// Revision 1.0  2004/07/19 16:14:16  charlesr
// Initial revision
//
