//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2003 - 2010. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util.regex;

import java.util.Iterator;
import java.util.SortedSet;

/**
 * This regular expression component matches only those
 * characters listed in its set. This pattern element has an
 * associated minimum and maximum range. The component must match
 * at least the minimum count and up to the maximum count
 * (inclusive).
 * <p>
 * This set may also be negated which means that it will match
 * any character <i>except</i> those listed in its set.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class CharacterSet
    extends Component
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The character set is stored as a bit set for quick
     * matching.
     */
    private final SortedSet<Character> mSet;

    /**
     * Set to {@code true} if this set is negated.
     */
    private final boolean mNegationFlag;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates character set (either postive or negative) with
     * the minimum and maximum match count.
     * @param set The character bit set.
     * @param flag {@code true} if this is a negated set.
     * @param minSize Must match at least this many times.
     * @param maxSize May match up to this many times.
     * @param index The component's index within the pattern.
     */
    /* package */ CharacterSet(final SortedSet<Character> set,
                               final boolean flag,
                               final int minSize,
                               final int maxSize,
                               final int index)
    {
        super (CHARACTER_SET, minSize, maxSize, index);

        mSet = set;
        mNegationFlag = flag;
    } // end of CharacterSet(Set<>, boolean, int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Component Abstract Methods Implementation.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     * A character set is less than a specific character if the
     * set's smallest character is less than {@code c}.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean lessThan(final char c)
    {
        return (((mSet.first()) < c) || mNegationFlag);
    } // end of lessThan(char)

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * A character set is equal to a specific characer if
     * {@code c} is in the set.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean equalTo(final char c)
    {
        return (mSet.contains(c) != mNegationFlag);
    } // end of equalTo(char)

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * A character set is greater than a specified character if
     * the set's maximum character is greater than {@code c}.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean greaterThan(final char c)
    {
        return (((mSet.last()) > c) || mNegationFlag);
    } // end of greaterThan(char)

    //
    // end of Component Abstract Methods Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns {@code true} if this character is negated (^)
     * and {@code false} otherwise.
     * @return {@code true} if this character is negated (^)
     * and {@code false} otherwise.
     */
    public boolean isNegated()
    {
        return (mNegationFlag);
    } // end of isNegated()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns a textual representation of a character set.
     * @return a textual representation of a character set.
     */
    @Override
    public String toString()
    {
        Iterator<Character>  it;
        int index;
        final int[] range = new int[2];
        int rangeIndex = 0;
        final StringBuilder buffer = new StringBuilder();

        buffer.append('[');

        if (mNegationFlag == true)
        {
            buffer.append('^');
        }

        for (it = mSet.iterator(); it.hasNext() == true;)
        {
            index = it.next();

            // Is a range being collected?
            if (rangeIndex > 0)
            {
                // Is this character a part of this range?
                if (index == (range[1] + 1))
                {
                    // Yes. Add this character to the range.
                    range[1] = index;
                }
                else
                {
                    buffer.append((char) range[0]);

                    // No. Output the range. If the range is
                    // two consecutive characters, then don't
                    // use a dash.
                    if (range[1] > (range[0] + 1))
                    {
                        buffer.append('-');
                    }

                    buffer.append((char) range[1]);

                    // The range is now empty.
                    rangeIndex = 0;
                }
            }

            // If the potential range is empty, then place this
            // character into the range.
            if (rangeIndex == 0)
            {
                range[0] = index;
                rangeIndex = 1;
            }
        }

        // Output any and all waiting characters.
        // Is there a range being collected?
        if (rangeIndex == 0)
        {
            // No. Output the singleton character.
            buffer.append((char) range[0]);
        }
        else
        {
            buffer.append((char) range[0]);

            // Again it is not a range if the start and end
            // are next to each other.
            if (range[1] > (range[0] + 1))
            {
                buffer.append('-');
            }

            buffer.append((char) range[1]);
        }

        buffer.append(']');
        appendSize(buffer);

        return (buffer.toString());
    } // end of toString()
} // end of class CharacterSet

//
// CHANGE LOG
// $Log: CharacterSet.java,v $
// Revision 1.3  2006/10/22 17:09:37  charlesr
// Removed matches() method - redundant with equalTo().
// Move MAX_CHAR_SIZE to Component.
//
// Revision 1.2  2006/10/20 19:21:58  charlesr
// Modified to use new regular expression syntax.
//
// Revision 1.1  2005/06/21 11:30:53  charlesr
// *** empty log message ***
//
// Revision 1.0  2004/07/19 16:12:33  charlesr
// Initial revision
//
