/*
 * Decompiled with CFR 0.152.
 */
package net.sf.eBus.util;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Date;
import java.util.Formatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

public final class IndexCache {
    private final Map<String, Index> _indices;
    private final String _cacheFileName;
    private final RandomAccessFile _cacheFile;
    private final Date _lastModified;
    private static final Map<String, IndexCache> _cacheMap = new HashMap<String, IndexCache>();
    private static final Lock _cacheMutex = new ReentrantLock();
    public static final long DEFAULT_INITIAL_INDEX = 0L;
    public static final long DEFAULT_FINAL_INDEX = Long.MAX_VALUE;
    public static final long DEFAULT_INCREMENT = 1L;
    public static final boolean DEFAULT_AUTO_RESET = false;
    public static final int MAXIMUM_INDEX_NAME_LENGTH = 20;
    private static final String ACCESS_MODE = "rwd";
    private static final long MINIMUM_FILE_LENGTH = 4L;
    private static final long INDEX_OFFSET = 4L;
    private static final long INDEX_SIZE = 54L;
    private static final byte PADDING_VALUE = 0;
    private static final byte[] NAME_PADDING = new byte[]{0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    private IndexCache(Map<String, Index> indices, String cacheFileName, RandomAccessFile cacheFile, Date lastModified) {
        this._indices = indices;
        this._cacheFileName = cacheFileName;
        this._cacheFile = cacheFile;
        this._lastModified = lastModified;
    }

    public String cacheFileName() {
        return this._cacheFileName;
    }

    public Date lastModified() {
        return new Date(this._lastModified.getTime());
    }

    public boolean isEmpty() {
        return this._indices.isEmpty();
    }

    public int indexCount() {
        return this._indices.size();
    }

    public List<String> indexNames() {
        return new ArrayList<String>(this._indices.keySet());
    }

    public boolean containsIndex(String indexName) {
        return this._indices.containsKey(indexName);
    }

    public Index getIndex(String indexName) {
        return this._indices.get(indexName);
    }

    public static boolean isCacheOpen(String cacheName) {
        boolean retcode = false;
        _cacheMutex.lock();
        try {
            retcode = _cacheMap.containsKey(cacheName);
        }
        finally {
            _cacheMutex.unlock();
        }
        return retcode;
    }

    public Index addIndex(String indexName, long initialIndex, long finalIndex, long increment, boolean resetFlag) throws IllegalArgumentException {
        Index retval = null;
        if (indexName == null) {
            throw new IllegalArgumentException("null indexName");
        }
        if (indexName.length() == 0) {
            throw new IllegalArgumentException("empty indexName");
        }
        if (indexName.length() > 20) {
            throw new IllegalArgumentException("indexName too long");
        }
        if (this._indices.containsKey(indexName)) {
            throw new IllegalArgumentException("indexName not unique");
        }
        if (increment == 0L) {
            throw new IllegalArgumentException("zero increment");
        }
        if (increment > 0L && initialIndex > finalIndex) {
            throw new IllegalArgumentException("increment > 0 and initialIndex > finalIndex");
        }
        if (increment < 0L && initialIndex < finalIndex) {
            throw new IllegalArgumentException("increment < 0 and initialIndex < finalIndex");
        }
        retval = new Index(indexName, initialIndex, finalIndex, increment, resetFlag, initialIndex);
        this._indices.put(indexName, retval);
        return retval;
    }

    public boolean removeIndex(String indexName) {
        boolean retcode = this._indices.containsKey(indexName);
        if (retcode) {
            this._indices.remove(indexName);
        }
        return retcode;
    }

    public void resetAllIndices() {
        this._indices.values().stream().forEach(index -> index.resetIndex());
    }

    public void clearIndices() {
        this._indices.clear();
    }

    public void close() {
        this.doClose();
        _cacheMutex.lock();
        try {
            _cacheMap.remove(this._cacheFileName);
        }
        finally {
            _cacheMutex.unlock();
        }
    }

    public String toString() {
        Formatter retval = new Formatter();
        retval.format("cache file = %s%n", this._cacheFileName);
        retval.format("   indices =", new Object[0]);
        if (this._indices.isEmpty()) {
            retval.format(" (none)", new Object[0]);
        } else {
            this._indices.values().stream().forEach(index -> retval.format("%n%s", index));
        }
        return retval.toString();
    }

    public static IndexCache open(String cacheFileName) throws IllegalArgumentException, IOException {
        IndexCache retval = null;
        if (cacheFileName == null) {
            throw new IllegalArgumentException("null cacheFileName");
        }
        if (cacheFileName.length() == 0) {
            throw new IllegalArgumentException("empty cacheFileName");
        }
        _cacheMutex.lock();
        try {
            File fInfo = new File(cacheFileName);
            if (_cacheMap.containsKey(cacheFileName)) {
                retval = _cacheMap.get(cacheFileName);
            } else if (!fInfo.exists() || fInfo.length() == 0L) {
                retval = IndexCache.newCache(cacheFileName);
                _cacheMap.put(cacheFileName, retval);
            } else {
                if (fInfo.length() < 4L) {
                    throw new IOException(String.format("%s invalid, %,d bytes", cacheFileName, 4L));
                }
                retval = IndexCache.openCache(cacheFileName, fInfo);
                _cacheMap.put(cacheFileName, retval);
            }
        }
        finally {
            _cacheMutex.unlock();
        }
        return retval;
    }

    private static IndexCache newCache(String fn) throws IOException {
        RandomAccessFile fs = new RandomAccessFile(fn, ACCESS_MODE);
        HashMap<String, Index> indices = new HashMap<String, Index>();
        return new IndexCache(indices, fn, fs, new Date());
    }

    private static IndexCache openCache(String fn, File fInfo) throws IOException {
        RandomAccessFile fs = new RandomAccessFile(fn, ACCESS_MODE);
        HashMap<String, Index> indices = new HashMap<String, Index>();
        int indexCount = fs.readInt();
        long fileSize = 4L + (long)indexCount * 54L;
        if (indexCount < 0 || fileSize > fInfo.length()) {
            throw new IOException("corrupt index cache");
        }
        for (int i = 0; i < indexCount; ++i) {
            Index index = IndexCache.readIndex(fs);
            indices.put(index.name(), index);
        }
        return new IndexCache(indices, fn, fs, new Date(fInfo.lastModified()));
    }

    private static Index readIndex(RandomAccessFile fs) throws IOException {
        String indexName = IndexCache.readName(fs);
        long initialIndex = fs.readLong();
        long finalIndex = fs.readLong();
        long increment = fs.readLong();
        boolean resetFlag = fs.readBoolean();
        long currentIndex = fs.readLong();
        if (increment == 0L || increment > 0L && initialIndex > finalIndex || increment < 0L && initialIndex < finalIndex || increment > 0L && currentIndex > finalIndex || increment < 0L && currentIndex < finalIndex) {
            throw new IOException("corrupt index cache");
        }
        return new Index(indexName, initialIndex, finalIndex, increment, resetFlag, currentIndex);
    }

    private void doClose() {
        try {
            this._cacheFile.seek(0L);
            this._cacheFile.writeInt(this._indices.size());
            for (Index index : this._indices.values()) {
                index.store(this._cacheFile);
            }
            this._indices.clear();
        }
        catch (IOException iOException) {
        }
        finally {
            try {
                this._cacheFile.close();
            }
            catch (IOException iOException) {}
        }
    }

    private static String readName(RandomAccessFile fs) throws IOException {
        int length = fs.read();
        String retval = null;
        if (length <= 0 || length > 20) {
            throw new IOException("corrupt index cache");
        }
        byte[] data = new byte[20];
        if (fs.read(data) < 0) {
            throw new IOException("error reading index name");
        }
        retval = new String(data, StandardCharsets.US_ASCII);
        return retval;
    }

    private static void writeName(String name, RandomAccessFile fs) throws IOException {
        int length = name.length();
        int paddingLength = 20 - length;
        fs.write(length);
        fs.write(name.getBytes(StandardCharsets.US_ASCII));
        if (paddingLength > 0) {
            fs.write(NAME_PADDING, 0, paddingLength);
        }
    }

    static {
        Runtime.getRuntime().addShutdownHook(new CacheShutdown());
    }

    private static final class CacheShutdown
    extends Thread {
        private CacheShutdown() {
        }

        @Override
        public void run() {
            ArrayList caches = new ArrayList();
            _cacheMutex.lock();
            try {
                caches.addAll(_cacheMap.values());
                _cacheMap.clear();
            }
            finally {
                _cacheMutex.unlock();
            }
            caches.stream().forEach(cache -> ((IndexCache)cache).doClose());
        }
    }

    public static final class Index {
        private final String _name;
        private final long _initialIndex;
        private final long _finalIndex;
        private final long _increment;
        private final boolean _resetUponFinalFlag;
        private long _currentIndex;

        private Index(String name, long initialIndex, long finalIndex, long increment, boolean resetFlag, long currentIndex) {
            this._name = name;
            this._initialIndex = initialIndex;
            this._finalIndex = finalIndex;
            this._increment = increment;
            this._resetUponFinalFlag = resetFlag;
            this._currentIndex = currentIndex;
        }

        public String name() {
            return this._name;
        }

        public long initialIndex() {
            return this._initialIndex;
        }

        public long increment() {
            return this._increment;
        }

        public long finalIndex() {
            return this._finalIndex;
        }

        public long currentIndex() {
            return this._currentIndex;
        }

        public boolean doesResetUponFinalIndex() {
            return this._resetUponFinalFlag;
        }

        public long nextIndex() throws IndexOutOfBoundsException {
            long retval = this._currentIndex;
            this._currentIndex += this._increment;
            if (retval > this._finalIndex && this._increment > 0L || retval < this._finalIndex && this._increment < 0L) {
                if (this._resetUponFinalFlag) {
                    retval = this._initialIndex;
                } else {
                    throw new IndexOutOfBoundsException(String.format("cannot increment beyond final index (%,d)", this._finalIndex));
                }
            }
            this._currentIndex = retval + this._increment;
            return retval;
        }

        public void currentIndex(long index) throws IllegalArgumentException {
            if (this._increment > 0L && (index < this._initialIndex || index > this._finalIndex) || this._increment < 0L && (index > this._initialIndex || index < this._finalIndex)) {
                throw new IllegalArgumentException("index outside initial, final interval");
            }
            this._currentIndex = index;
        }

        public long resetIndex() {
            this._currentIndex = this._initialIndex;
            return this._currentIndex;
        }

        public String toString() {
            Formatter retval = new Formatter();
            retval.format("[%s]%n", this._name);
            retval.format("   initial index = %,d%n", this._initialIndex);
            retval.format("     final index = %,d%n", this._finalIndex);
            retval.format("       increment = %,d%n", this._increment);
            retval.format("   current index = %,d%n", this._currentIndex);
            retval.format("reset upon final = %b", this._resetUponFinalFlag);
            return retval.toString();
        }

        private void store(RandomAccessFile fs) throws IOException {
            IndexCache.writeName(this._name, fs);
            fs.writeLong(this._initialIndex);
            fs.writeLong(this._finalIndex);
            fs.writeLong(this._increment);
            fs.writeBoolean(this._resetUponFinalFlag);
            fs.writeLong(this._currentIndex);
        }
    }
}

