//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2003 - 2008. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util.regex;

/**
 * This regular expression component matches any character. This
 * pattern element has an associated minimum and maximum range.
 * The component must match at least the minimum count and up
 * to the maximum count (inclusive).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class AnyChar
    extends Component
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x050200L;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a any character component with the specified
     * minimum and maximum range.
     * @param minSize Must match at least this many times.
     * @param maxSize May match up to this many times.
     * @param index Component's index in the pattern.
     */
    /* package */ AnyChar(final int minSize,
                          final int maxSize,
                          final int index)
    {
        super (ANY_CHAR, minSize, maxSize, index);
    } // end of AnyChar(int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Component Abstract Methods Implementation.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false}
     * otherwise. Any character is always less than the specified
     * character <i>if</i> {@code c} is not the minimally
     * allowed character value.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean lessThan(final char c)
    {
        return (c > Character.MIN_VALUE);
    } // end of lessThan(char)

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * Any character is always equal to the specified
     * character.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean equalTo(final char c)
    {
        return (true);
    } // end of equalTo(char)

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * Any character is always greater than the specified
     * character <i>if</i> {@code c} is not the maximally allowed
     * character.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean greaterThan(final char c)
    {
        return (c < Character.MAX_VALUE);
    } // end of greaterThan(char)

    //
    // end of Component Abstract Methods Implementation.
    //-----------------------------------------------------------

    /**
     * Returns a textual representation of a single character
     * match.
     * @return a textual representation of a single character
     * match.
     */
    @Override
    public String toString()
    {
        final StringBuilder buffer = new StringBuilder();

        buffer.append('.');
        appendSize(buffer);

        return (buffer.toString());
    } // end of toString()
} // end of class AnyChar

//
// CHANGE LOG
// $Log: AnyChar.java,v $
// Revision 1.3  2006/10/22 17:09:11  charlesr
// Removed matches() method - redundant with equalTo().
// Corrected lessThan() and greaterThan() methods.
//
// Revision 1.2  2006/10/20 19:21:51  charlesr
// Modified to use new regular expression syntax.
//
// Revision 1.1  2005/06/21 11:43:53  charlesr
// Check-in prior to Java 5 update.
//
// Revision 1.0  2004/07/19 16:14:26  charlesr
// Initial revision
//
