//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2008, 2016. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.text;

/**
 * Contains the token's type, corresponding Java object, raw
 * text and the input line number where the token was found. This
 * class is immutable.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class Token
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a token with the given attributes.
     * @param type The token type. See {@link TokenLexer} for the
     * token type list.
     * @param value The token value created from raw token
     * string.
     * @param token The raw token string.
     * @param line The input line where the token was found.
     */
    public Token(final int type,
                 final Object value,
                 final String token,
                 final int line)
    {
        _type = type;
        _value = value;
        _token = token;
        _lineNumber = line;
    } // end of Token(int, Object, String, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the token type.
     * @return the token type.
     */
    public int type()
    {
        return (_type);
    } // end of type()

    /**
     * Returns the token value.
     * @return the token value.
     */
    public Object value()
    {
        return (_value);
    } // end of value()

    /**
     * Returns the raw token string.
     * @return the raw token string.
     */
    public String token()
    {
        return (_token);
    } // end of token()

    /**
     * Returns the input line number where the token was found.
     * @return the input line number where the token was found.
     */
    public int lineNumber()
    {
        return (_lineNumber);
    } // end of lineNumber()

    /**
     * Returns {@code true} if {@code value} equals the
     * token value and {@code false} otherwise.
     * @param value Test equality between this object and the
     * token value.
     * @return {@code true} if {@code value} equals the
     * token value and {@code false} otherwise.
     */
    public boolean isValue(final Object value)
    {
        return (_value == null ?
                (value == null) :
                _value.equals(value));
    } // end of isValue(Object)

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns {@code true} if {@code o} is a non-{@code null}
     * {@code Token} instance of the same type and raw token
     * text; {@code false} otherwise.
     * @param o Test equality with this object.
     * @return {@code true} if {@code o} is a non-{@code null}
     * {@code Token} instance of the same type and raw token
     * text; {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof Token)
        {
            final Token token = (Token) o;

            retcode =
                (_type == token._type &&
                 _token.equals(token._token) == true);
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns the token hashcode based on the raw token string.
     * @return the token hashcode based on the raw token string.
     */
    @Override
    public int hashCode()
    {
        return (_token.hashCode());
    } // end of hashCode()

    /**
     * Returns this token's textual representation.
     * @return this token's textual representation.
     */
    @Override
    public String toString()
    {
        return (
            String.format(
                "{line=%,d;type=%d; value=\"%s\"; token=\"%s\"}",
                _lineNumber,
                _type,
                _value,
                _token));
    } // end of toString()

//---------------------------------------------------------------
// Member data.
//

    private final int _type;
    private final Object _value;
    private final String _token;
    private final int _lineNumber;
} // end of class Token
