/*
 *  Copyright 2003-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * CHANGE LOG
 * (See the bottom of this file.)
 */

package net.sf.eBus.util;

import java.io.Serializable;
import java.util.Formatter;

/**
 * Allows multi-valued keys to be used in Java's single key
 * {@code Map} collections. If an {@code Order} object is
 * associated with a tuple
 * (String customer, Date timestamp, OrderId id), Java cannot
 * handle multi-object keys but only a single object key. The
 * solutions to this mismatch is:
 * <ol>
 *   <li>
 *     Use nested {@code Map} objects:
 *     <code>
 * Map&lt;String, Map&lt;Date, Map&lt;OrderId, Order&gt;&gt;&gt;.
 *     </code>
 *     This solution is both cumbersome to implement, uses much
 *     memory and is not recommended.
 *   </li>
 *   <li>
 *     Write a class to store the tuple objects and provides
 *     the appropriate hash code for the key values. This is
 *     the only solution if {@code MultiKey}'s hash code is
 *     insufficient.
 *   </li>
 * </ol>
 * {@code MultiMap} stores multiple key values in a general
 * class obviating the need to write a proprietary key class.
 * The hash code algorithm is:
 * <pre>
 *   <code>
 *     Object[] keys;
 *     int i;
 *     int hashCode;
 *
 *     for (i = 0, hashCode = 0; i &lt; keys.length; ++i)
 *     {
 *         if (keys[i] != null)
 *         {
 *             hashCode ^= keys[i].hashCode();
 *         }
 *     }
 *   </code>
 * </pre>
 * <p>
 * Unlike a proprietary key class, {@code MultiKey} cannot
 * enforce key value number and type requirements. It is possible
 * to use keys of differing lengths and type within the same map.
 * If {@code MultiKey}'s size and type ordering enforcement is
 * required, use the subclasses {@link MultiKey2},
 * {@link MultiKey3} or {@link MultiKey4}.
 * <p>
 * It is recommended that only immutable objects be used in the
 * key but if mutable objects are used and are modified
 * <i>after</i> creating the {@code MultiKey}, the
 * {@code MultiKey} hash code is unaffected because the hash code
 * is calculated once in the constructor. This results in a
 * discrepency between the key's values and hash code.
 *
 * @since Commons Collections 3.0
 * @version $Revision: 1.1 $ $Date: 2005/10/28 14:37:52 $
 *
 * @author Howard Lewis Ship
 * @author Stephen Colebourne
 * @author Charles W. Rapp
 */

public class MultiKey
    implements Comparable<MultiKey>,
               Serializable
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a multiple key container for the given objects.
     * @param keys One or more key objects. May contain
     * {@code null} references.
     * @exception IndexOutOfBoundsException
     * if no keys are specified.
     */
    public MultiKey(final Object... keys)
        throws IndexOutOfBoundsException
    {
        if (keys.length == 0)
        {
            throw (new IndexOutOfBoundsException("no keys"));
        }
        else
        {
            int i;
            int hashCode;

            // Make a shallow copy of the keys array.
            _keys = new Object[keys.length];
            System.arraycopy(keys, 0, _keys, 0, keys.length);

            // Since _hashCode is final, we can only assign
            // it once. So use a local variable to calculate
            // the hash code and then put the result into
            // _hashCode.
            for (i = 0, hashCode = 0; i < keys.length; ++i)
            {
                if (keys[i] != null)
                {
                    hashCode ^= keys[i].hashCode();
                }
            }

            _hashCode = hashCode;
        }
    } // end of MultiKey(Object...)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Comparable Interface Implementation.
    //

    /**
     * Returns a negative integer, zero or a positive integer if
     * this object is less than, equal to or greater than the
     * argument, respectively.
     * <p>
     * The comparison is done as follows:
     * <ol>
     *   <li>
     *     If {@code this} key's size does not equal
     *     {@code key}'s size, then returns
     *     {@code this.size() - key.size()}.
     *   </li>
     *   <li>
     *     Otherwise iterates over the keys, comparing each
     *     key in turn until a non-zero value is found or
     *     all the keys are compared.
     *   </li>
     * </ol>
     * @param key compare against this multi-valued key.
     * @return a negative integer, zero or a positive integer if
     * this object is less than, equal to or greater than the
     * argument, respectively.
     * @exception ClassCastException
     * if the keys contain incompatible or incomparable types.
     */
    @Override
    @SuppressWarnings("unchecked")
    public int compareTo(final MultiKey key)
        throws ClassCastException
    {
        int retval = 0;

        if (this != key)
        {
            int i;

            retval = (_keys.length - key._keys.length);
            for (i = 0; i < _keys.length && retval == 0; ++i)
            {
                if (_keys[i] == null)
                {
                    if (key._keys[i] != null)
                    {
                        retval = -1;
                    }
                    // else retval is already zero.
                }
                else
                {
                    retval =
                        ((Comparable<Object>) _keys[i]).compareTo(
                            key._keys[i]);
                }
            }
        }

        return (retval);
    } // end of compareTo(MultiKey)

    //
    // end of ComparableInterface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the number of key values.
     * @return the number of key values.
     */
    public int size()
    {
        return (_keys.length);
    } // end of size()

    /**
     * Returns the key value at the specified index.
     * <p>
     * If the returned object is mutable, modifying it will
     * result in unspecified behavior by the map.
     * @param index Index to desired key value.
     * @return the key value at the specified index.
     * @exception IndexOutOfBoundsException
     * if {@code index} is either &lt; zero or &gt;
     * key size.
     */
    public Object key(final int index)
        throws IndexOutOfBoundsException
    {
        if (index < 0 || index >= _keys.length)
        {
            throw (
                new IndexOutOfBoundsException(
                    Integer.toString(index) +
                    " invalid index"));
        }

        return (_keys[index]);
    } // end of key(int)

    /**
     * Returns a copy of the key array.
     * <p>
     * If any of the returned key objects is mutable, modifying a
     * key will result in unspecified behavior by the map.
     * @return the key array copy.
     */
    public Object[] keys()
    {
        final Object[] keys = new Object[_keys.length];

        System.arraycopy(_keys, 0, keys, 0, _keys.length);

        return (keys);
    } // end of keys()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns {@code true} if {@code key} is a non-null
     * MultiKey instance with the same number of key values and
     * the key values are equal and in the same order. Returns
     * {@code false} otherwise.
     * @param o Test equality with this object.
     * @return {@code true} if {@code key} is a non-null
     * MultiKey instance with the same number of key values and
     * the key values are equal and in the same order;
     * {@code false} otherwise.
     */
    @Override
    public boolean equals(final Object o)
    {
        boolean retcode = (this == o);

        if (retcode == false && o instanceof MultiKey)
        {
            final MultiKey mkey = (MultiKey) o;

            if (_keys.length == mkey._keys.length)
            {
                int i;

                for (retcode = true, i = 0;
                     retcode == true && i < _keys.length;
                     ++i)
                {
                    retcode =
                        (_keys[i] == null ?
                         (mkey._keys[i] == null) :
                         _keys[i].equals(mkey._keys[i]));
                }
            }
        }

        return (retcode);
    } // end of equals(Object)

    /**
     * Returns a hash code value for this multi-valued key.
     * <p>
     * The hash code is calculated once in the constructor and
     * cached. If any key objects is modified after construction,
     * its new hash code will <i>not</i> be reflected in this
     * {@code MultiKey} object's hash code.
     * <p>
     * The hash code calculation is:
     * <pre>
     *   <code>
     *     Object[] keys;
     *     int i;
     *     int hashCode;
     *
     *     for (i = 0, hashCode = 0; i &lt; keys.length; ++i)
     *     {
     *         if (keys[i] != null)
     *         {
     *             hashCode ^= keys[i].hashCode();
     *         }
     *     }
     *   </code>
     * </pre>
     * @return a hash code value for this multi-valued key.
     */
    @Override
    public int hashCode()
    {
        return (_hashCode);
    } // end of hashCode()

    /**
     * Returns a textual representation of this multi-valued
     * key.
     * @return a textual representation of this multi-valued
     * key.
     */
    @Override
    public String toString()
    {
        final Formatter buffer = new Formatter();
        int i;
        String sep;

        buffer.format("{");
        for (i = 0, sep = ""; i < _keys.length; ++i, sep = ", ")
        {
            buffer.format("%s%s", sep, _keys[i]);
        }
        buffer.format("}");

        return (buffer.toString());
    } // end of toString()

//---------------------------------------------------------------
// Member data.
//

    /**
     * The key objects.
     */
    private final Object[] _keys;

    /**
     * The overall hash code for the keys. Calculated once in the
     * constructor.
     */
    private final int _hashCode;

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x020100L;
} // end of class MultiKey

//
// CHANGE LOG
// $Log: MultiKey.java,v $
// Revision 1.1  2005/10/28 14:37:52  charlesr
// Added java.lang.Comparable interface.
//
// Revision 1.0  2005/10/27 15:07:41  charlesr
// Initial revision
//
