//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2003 - 2008. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util.regex;

/**
 * This regular expression component matches one character. This
 * pattern element has an associated minimum and maximum range.
 * The component must match at least the minimum count and up
 * to the maximum count (inclusive).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class Literal
    extends Component
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a component for the given literal character,
     * minimum and maximum match sizes.
     * @param c The literal character.
     * @param minSize Must match at least this many times.
     * @param maxSize May match up to this many times.
     * @param index The component's index within the pattern.
     */
    /* package */ Literal(final char c,
                          final int minSize,
                          final int maxSize,
                          final int index)
    {
        super (LITERAL, minSize, maxSize, index);

        _c = c;
    } // end of Literal(char, int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Component Abstract Methods Implementation.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean lessThan(final char c)
    {
        return (_c < c);
    } // end of lessThan(char)

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean equalTo(final char c)
    {
        return (_c == c);
    } // end of equalTo(char)

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    @Override
    public boolean greaterThan(final char c)
    {
        return (_c > c);
    } // end of greaterThan(char)

    //
    // end of Component Abstract Methods Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the literal character.
     * @return the literal character.
     */
    public int literal()
    {
        return (_c);
    } // end of literal()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    /**
     * Returns a textual representation of a literal.
     * @return a textual representation of a literal.
     */
    @Override
    public String toString()
    {
        final StringBuilder buffer = new StringBuilder();

        buffer.append(_c);
        appendSize(buffer);

        return (buffer.toString());
    } // end of toString()

//---------------------------------------------------------------
// Member data.
//

    /**
     * The literal component's character.
     */
    private final char _c;

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     *  This is eBus version 2.1.0.
     */
    private static final long serialVersionUID = 0x020100L;
} // end of class Literal

//
// CHANGE LOG
// $Log: Literal.java,v $
// Revision 1.3  2006/10/22 17:10:08  charlesr
// Removed matches() method - redundant with equalTo().
//
// Revision 1.2  2006/10/20 19:22:24  charlesr
// Modified to use new regular expression syntax.
//
// Revision 1.1  2005/06/21 11:37:50  charlesr
// Check-in prior to Java 5 update.
//
// Revision 1.0  2004/07/19 16:13:22  charlesr
// Initial revision
//
