//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2001 - 2008, 2016. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util;

import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * An {@code eBus.util.TimerTask} associates a timer
 * task with a {@link TimerTaskListener}. When the timer expires,
 * {@link TimerTaskListener#handleTimeout(TimerEvent)} where
 * {@link TimerEvent#getSource()} returns the value passed to
 * the {@link #TimerTask(TimerTaskListener, Object)} constructor.
 * <p>
 * In eBus v. 4.2.0, {@code TimerTaskListener} is marked as a
 * {@code @FunctionalInterface}. This allows the listener to be
 * defined using a lambda expression.
 * </p>
 * <p>
 * If {@link TimerTaskListener#handleTimeout(TimerEvent)} throws
 * an exception, the exception stack trace is logged at the
 * warning level.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class TimerTask
    extends java.util.TimerTask
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code TimerTask}. This task
     * is sent to the {@link TimerTaskListener} when the
     * timer expires. The associated value is {@code null}.
     * @param listener Inform this listener of the timeout.
     * @exception IllegalArgumentException
     * if {@code listener} is {@code null}.
     */
    public TimerTask(final TimerTaskListener listener)
        throws IllegalArgumentException
    {
        this (listener, null);
    } // end of TimerTask(TimerTaskListener)

    /**
     * Creates a new {@code TimerTask}. This task
     * is sent to
     * {@link TimerTaskListener#handleTimeout(TimerEvent)}
     * when the timer expires. The associated value is
     * {@code value}.
     * @param listener Inform this listener of the timeout.
     * @param value The associated value passed to the listener.
     * May be {@code null}.
     * @exception IllegalArgumentException
     * if {@code listener} is {@code null}.
     */
    public TimerTask(final TimerTaskListener listener,
                     final Object value)
        throws IllegalArgumentException
    {
        super ();

        if (listener == null)
        {
            throw (
                new IllegalArgumentException("null listener"));
        }

        _callback = listener;
        _value = value;
    } // end of TimerTask(TimerTaskListener)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // java.util.TimerTask Method Overrides.
    //

    /**
     * Cancels this timer task. See
     * {@code java.util.TimerTask.cancel} for a
     * detailed description of cancel.
     * @return {@code true} if this timer task is scheduled to
     * run once and never ran or is scheduled for repeated
     * execution. Returns {@code false} if this task was
     * scheduled to run once and was executed, if this task
     * was never scheduled or if this task is already canceled.
     */
    @Override
    public boolean cancel()
    {
        _callback = null;

        return (super.cancel());
    } // end of cancel()

    /**
     * Tells the listener that this timer has expired.
     * <p>
     * <b>Note: Do not call this method!</b> This method is part
     * of {@code java.util.TimerTask} and should be called
     * by {@code java.util.Timer} only.
     */
    @Override
    public void run()
    {
        if (_callback != null)
        {
            try
            {
                _callback.handleTimeout(
                    new TimerEvent(this, _value));
            }
            catch (Exception jex)
            {
                _logger.log(Level.WARNING,
                            "TimerTaskListener exception",
                            jex);
            }
        }

        return;
    } // end of run()

    //
    // end of java.util.TimerTask Method Overrides.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Member data.
//

    /**
     * Callback to this method when the timer expires.
     */
    private volatile TimerTaskListener _callback;

    /**
     * The associated value. Placed in the {@link TimerEvent}
     * passed to the listener.
     */
    private final Object _value;

    //-----------------------------------------------------------
    // Statics.
    //

    private static final Logger _logger =
        Logger.getLogger(TimerTask.class.getName());
} // end of class TimerTask

//
// CHANGE LOG
// $Log: TimerTask.java,v $
// Revision 1.4  2006/01/07 14:24:22  charlesr
// Corrected javadoc code tags.
//
// Revision 1.3  2005/07/22 01:39:40  charlesr
// Move to Java 5.
//
// Revision 1.2  2004/05/24 15:50:15  charlesr
// Corrected incorrect javadoc @link tag.
//
// Revision 1.1  2004/04/06 00:08:49  charlesr
// Protect java.util.Timer from TimerTask.handleTimeout()-thrown
// exceptions.
//
// Revision 1.0  2003/11/20 01:47:26  charlesr
// Initial revision
//
