//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2003 - 2008. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util.regex;

import java.io.Serializable;

/**
 * Abstract base class for all regular expression components.
 * Every component has a minimum and maximum match length. The
 * component must match at least the minimum length and at most
 * the maximum length (inclusive).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class Component
    implements Serializable
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Sets the component's type, minimum and maximum length.
     * @param type The regular expression component type.
     * @param minLength The component's minimum length.
     * @param maxLength The component's maximum length.
     * @param index The component's position in the pattern.
     */
    protected Component(final int type,
                        final int minLength,
                        final int maxLength,
                        final int index)
    {
        _type = type;
        _minimumSize = minLength;
        _maximumSize = maxLength;
        _index = index;
    } // end of Component(int, int, int, int)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns this component's type. Return value will be one of
     * <ul>
     *   <li>
     *     {@code LITERAL}
     *   </li>
     *   <li>
     *     {@code CHARACTER_SET}
     *   </li>
     *   <li>
     *     {@code ANY_CHAR}
     *   </li>
     * </ul>
     * @return this component's type.
     */
    public int type()
    {
        return (_type);
    } // end of type()

    /**
     * Returns the component's minimum size. This component must
     * match at least this many times.
     * @return the component's minimum size.
     */
    public int minimumSize()
    {
        return (_minimumSize);
    } // end of minimumSize()

    /**
     * Returns the component's maximum size. This component must
     * match at most this many times (inclusive).
     * @return the component's maximum size.
     */
    public int maximumSize()
    {
        return (_maximumSize);
    } // end of maximumSize()

    /**
     * Returns the component's pattern index.
     * @return the component's pattern index.
     */
    public int index()
    {
        return (_index);
    } // end of index()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Returns {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is less than
     * the character {@code c}; returns {@code false} otherwise.
     */
    public abstract boolean lessThan(char c);

    /**
     * Returns {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is equal to
     * the character {@code c}; returns {@code false} otherwise.
     */
    public abstract boolean equalTo(char c);

    /**
     * Returns {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     * @param c Test against this character.
     * @return {@code true} if this component is greater than
     * the character {@code c}; returns {@code false} otherwise.
     */
    public abstract boolean greaterThan(char c);

    //
    // Abstract Method Declarations.
    //-----------------------------------------------------------

    /**
     * Adds the size to the output only if not {1, 1}.
     * @param buffer Append the size to this output buffer.
     */
    protected void appendSize(final StringBuilder buffer)
    {
        if (_minimumSize != 1 || _maximumSize != 1)
        {
            // * is {0,} (Kleene closure).
            if (_minimumSize == 0 &&
                _maximumSize == NO_MAX_MATCH_LIMIT)
            {
                buffer.append('*');
            }
            // + is {1,} (Transitive closure).
            else if (_minimumSize == 1 &&
                     _maximumSize == NO_MAX_MATCH_LIMIT)
            {
                buffer.append('+');
            }
            else
            {
                buffer.append('{');
                if (_minimumSize > 0)
                {
                    buffer.append(_minimumSize);
                }
                buffer.append(',');
                if (_maximumSize > 0)
                {
                    buffer.append(_maximumSize);
                }
                buffer.append('}');
            }
        }

        return;
    } // end of appendSize(StringBuilder)

//---------------------------------------------------------------
// Member data.
//

    /**
     * The component's type. Must be one of
     * {@code LITERAL}, {@code CHARACTER_SET} or
     * {@code ANY_CHAR}.
     */
    protected final int _type;

    /**
     * The component's minimum possible length.
     */
    protected final int _minimumSize;

    /**
     * The component's maximum possible length.
     */
    protected final int _maximumSize;

    /**
     * The component's position within the regular expression
     * pattern. Indexing begins at zero.
     */
    protected final int _index;

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The maximum range limit is set to this value when the
     * the limit is infinite.
     */
    public static final int NO_MAX_MATCH_LIMIT = -1;

    // The valid component types.

    /**
     * A literal component matches exactly.
     */
    public static final int LITERAL       = 1;

    /**
     * A character set component matches only those characters
     * in the set (positive) or not in the set (negative).
     */
    public static final int CHARACTER_SET = 2;

    /**
     * This component matches any character.
     */
    public static final int ANY_CHAR      = 3;

    /**
     * The serialization version.
     */
    private static final long serialVersionUID = 1L;
} // end of class Component

//
// CHANGE LOG
// $Log: Component.java,v $
// Revision 1.4  2007/02/23 13:37:58  charlesr
// Corrected javadoc comments.
//
// Revision 1.3  2006/10/22 17:10:00  charlesr
// Removed matches() method - redundant with equalTo().
// Moved MAX_CHAR_SIZE here from CharacterSet.
//
// Revision 1.2  2006/10/20 19:22:05  charlesr
// Modified to use new regular expression syntax.
//
// Revision 1.1  2005/06/21 11:33:27  charlesr
// Check-in prior to Java 5 update.
//
// Revision 1.0  2004/07/19 16:12:51  charlesr
// Initial revision
//
