//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2001 - 2005. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.util.logging;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.Formatter;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Timer;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.eBus.util.TimerEvent;
import net.sf.eBus.util.TimerTask;
import net.sf.eBus.util.TimerTaskListener;

/**
 * Writes a status report to a log at a specified
 * interval. The default interval is every 15 minutes on the
 * quarter hour. The default
 * {@code java.util.logging.Logger} is the application's
 * root log. The default logging level is
 * {@code java.util.logging.Level.INFO}.
 * <p>
 * When the timer expires it generates a report containing:
 * <ul>
 *   <li>
 *     The application name, version and build date.
 *   </li>
 *   <li>
 *     The JVM's statistics: version and memory usage.
 *   </li>
 *   <li>
 *     Each registered
 *     {@link net.sf.eBus.util.logging.StatusReporter}'s
 *     statistics.
 *   </li>
 * </ul>
 * @see net.sf.eBus.util.logging.StatusReporter
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class StatusReport
    implements TimerTaskListener
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    // Create the status report object and start the timer.
    private StatusReport()
    {
        _initFlag = false;
        _appName = "<app name not set>";
        _version = "<version not set>";
        _buildDate = null;
        _boilerPlate = null;
        _level = Level.INFO;
        _frequency = ReportFrequency.FIFTEEN_MINUTE;

        // Do *not* dump the active threads by default.
        _dumpThreadsFlag = false;

        // Use the root logger by default
        _logger = Logger.getLogger("");

        _reporters = new LinkedList<>();
    } // end of StatusReport()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // TimerTaskListener Interface Implementation
    //

    /**
     * Time to output another status report.
     * <p>
     * <b>DO NOT CALL THIS METHOD!</b>
     * This method is called when the report timer expires.
     * @param event the expired timer event.
     */
    @Override
    public void handleTimeout(final TimerEvent event)
    {
        final StringWriter sw = new StringWriter();
        final PrintWriter pw = new PrintWriter(sw);
        final Runtime sys = Runtime.getRuntime();
        final long now = System.currentTimeMillis();
        final long upTime = now - _startTime.getTime();
        final long totalMem = sys.totalMemory();
        final long freeMem = sys.freeMemory();
        final long usedMem = totalMem - freeMem;
        List<StatusReporter> reporters;

        pw.println("STATUS REPORT");
        pw.println();

        // Report system wide data first.
        logHeader(pw);
        pw.println();
        pw.print("          JRE: ");
        pw.print(System.getProperty("java.version"));
        pw.print(" (");
        pw.print(System.getProperty("java.vendor"));
        pw.println(")");
        pw.print("          JVM: ");
        pw.print(System.getProperty("java.vm.name"));
        pw.print(", ");
        pw.print(System.getProperty("java.vm.version"));
        pw.print(" (");
        pw.print(System.getProperty("java.vm.vendor"));
        pw.println(")");
        pw.println();
        pw.print("   Start date: ");
        pw.format("%1$tm/%1$td/%1$tY at %1$tH:%1$tM:S%n",
                  _startTime);
        pw.print("      Up time: ");
        pw.println(formatTime(upTime));
        pw.format("  Used memory: %,d bytes%n", usedMem);
        pw.format("  Free memory: %,d bytes%n", freeMem);
        pw.format(" Total memory: %,d bytes%n", totalMem);
        pw.format(" Thread count: %,d%n%n", countThreads());

        // Print out the active threads if so requested.
        if (_dumpThreadsFlag == true)
        {
            outputThreads(pw);
            pw.println();
        }

        // Copy the reporters list and iterate over that.
        // This allows reports to deregister while the status
        // report is generated.
        synchronized (_reporters)
        {
            reporters = new LinkedList<>(_reporters);
        }

        reporters.stream().
            map((reporter) ->
        {
            reporter.reportStatus(pw);
            return reporter;
        }).
            forEach((_item) ->
        {
            pw.println();
        });

        // Log the report and then clear the buffer.
        _logger.log(_level, sw.toString());

        return;
    } // end of handleTimeout(TimerEvent)

    //
    // end of TimerTaskListener Interface Implementation
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get methods.
    //

    /**
     * Returns the {@code java.util.logging.Logger} to which
     * the status report is written.
     * @return the {@code java.util.logging.Logger} to which
     * the status report is written.
     */
    public synchronized Logger getLogger()
    {
        return (_logger);
    } // end of getLogger()

    /**
     * Returns the {@code java.util.logging.Level} at which
     * the status report is logged.
     * @return the {@code java.util.logging.Level} at which
     * the status report is logged.
     */
    public synchronized Level getLevel()
    {
        return (_level);
    } // end of getLevel()

    /**
     * Returns the current status report frequency in
     * milliseconds.
     * @return the current status report frequency in
     * milliseconds.
     */
    public synchronized ReportFrequency getReportFrequency()
    {
        return (_frequency);
    } // end of getReportFrequency()

    /**
     * Returns the "dump threads" flag. If this flag is
     * {@code true}, then the active threds are
     * listed in the report. This flag is {@code false} by
     * default.
     * @return the "dump threads" flag.
     */
    public synchronized boolean getDumpThreads()
    {
        return (_dumpThreadsFlag);
    } // end of getDumpThreads()

    //
    // end of Get methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set methods.
    //

    /**
     * Sets the {@code java.util.logging.Logger}. The
     * status report is written to this logger.
     * @param logger write the status report to this log.
     */
    public synchronized void setLogger(final Logger logger)
    {
        _logger = logger;
        return;
    } // end of setLogger(Logger)

    /**
     * Sets the {@code java.util.logging.Level} at which
     * the status report is logged.
     * @param level log the status report at this level.
     * @throws IllegalArgumentException
     * if {@code level} is {@code null}.
     */
    public synchronized void setLevel(final Level level)
        throws IllegalArgumentException
    {
        if (level == null)
        {
            throw (new IllegalArgumentException("null level"));
        }

        _level = level;
        return;
    } // end of setLevel(Level)

    /**
     * Sets the status report frequency. The allowed frequencies
     * are: 5 minute, 10 minute, 15 minute, 20 minute, 30 minute
     * and hourly. The default is 15 minute. If the frequency is
     * none of the above, then an
     * {@code java.lang.IllegalArgumentException} is thrown.
     * @param frequency the status report frequency.
     */
    public synchronized void setReportFrequency(
        final ReportFrequency frequency)
    {
        // If the frequency has changed, then stop the timer and
        // set it to the new rate.
        if (_frequency != frequency)
        {
            _frequency = frequency;

            if (_reportTimer != null)
            {
                // Once a TimerTask object is cancelled, you
                // have to throw it away and create a new one.
                // TimerTasks can't be reused.
                _reportTimer.cancel();
                _reportTimer = null;
            }

            // Now start the periodic report timer.
            startReportTimer();
        }

        return;
    } // end of setReportFrequency(ReportFrequency)

    /**
     * Sets the "dump threads" flag to the given value.
     * If this flag is {@code true}, then the active threads
     * are listed in the report.
     * @param flag set the "dump threads" flag to this value.
     */
    public synchronized void setDumpThreads(final boolean flag)
    {
        _dumpThreadsFlag = flag;
        return;
    } // end of setDumpThreads(boolean)

    //
    // end of Set methods.
    //-----------------------------------------------------------

    /**
     * Sets the application's name, version and build date. This
     * information is placed at the start of each status report
     * log message.
     * @param name the application's name.
     * @param version the application's version.
     * @param buildDate the application's build date.
     * @param boilerPlate usually a copyright statement.
     * @exception IllegalStateException
     * if the application information is already set.
     */
    public synchronized void setApplicationInfo(
        final String name,
        final String version,
        final Date buildDate,
        final String boilerPlate)
        throws IllegalStateException
    {
        if (_initFlag == true)
        {
            throw (
                new IllegalStateException(
                    "app info already set"));
        }
        else
        {
            final StringWriter sw = new StringWriter();
            final PrintWriter pw = new PrintWriter(sw);

            _initFlag = true;
            _appName = name;
            _version = version;
            _buildDate = new Date(buildDate.getTime());
            _boilerPlate = boilerPlate;

            pw.println(
                "=============================================" +
                "====================");
            logHeader(pw);

            _logger.info(sw.toString());
        }
    } // end of setApplicationInfo(String, String, Date, String)

    /**
     * Registers a status reporter. If the reporter is already
     * registered, then it is not added again.
     * <p>
     * <b>Note:</b> When the status report is generated,
     * reporters will be called in the order they
     * register.
     * @param reporter register this status reporter.
     */
    public void register(final StatusReporter reporter)
    {
        synchronized (_reporters)
        {
            if (_reporters.contains(reporter) == false)
            {
                _reporters.add(reporter);
            }
        }

        return;
    } // end of register(StatusReporter)

    /**
     * Deregisters a status reporter. If the reporter is not
     * registered, then nothing is done.
     * @param reporter deregister this reporter.
     */
    public void deregister(final StatusReporter reporter)
    {
        synchronized (_reporters)
        {
            _reporters.remove(reporter);
        }

        return;
    } // end of deregister(StatusReporter)

    /**
     * Returns the singleton {@link StatusReport} instance.
     * @return the singleton {@link StatusReport} instance.
     */
    public static StatusReport getInstance()
    {
        _ctorMutex.lock();
        try
        {
            if (_instance == null)
            {
                // Store away this single instance.
                _instance = new StatusReport();
                _instance.startReportTimer();
            }
        }
        finally
        {
            _ctorMutex.unlock();
        }

        return (_instance);
    } // end of getInstance()

    /**
     * Given a millisecond duration, generate a string that
     * reports the duration in human-readable form.
     * @param delta The millisecond duration.
     * @return the textual representation of a duration.
     */
    public static String formatTime(final long delta)
    {
        Iterator<StatusReport.TimeUnit> it;
        boolean startFlag;
        TimeUnit timeUnit;
        long unitTime;
        long scalar;
        long delta2 = delta;
        final Formatter retval = new Formatter();

        for (it = _timeUnitList.iterator(), startFlag = false;
             it.hasNext() == true;)
        {
            timeUnit = it.next();
            unitTime = timeUnit.getUnitTime();

            // Figure out the number of these time units
            // in the delta.
            scalar = (delta2 / unitTime);
            delta2 -= (scalar * unitTime);

            // Print out this time unit but only if it is not
            // zero or a non-zero unit has already been printed.
            if (scalar > 0 || startFlag == true)
            {
                // If the buffer already constains output,
                // then make sure to put a ", " before this
                // new unit.
                if (startFlag == false)
                {
                    startFlag = true;
                }
                else
                {
                    retval.format(", ");
                }

                retval.format(
                    "%d %s", scalar, timeUnit.getName());

                // Make the unit name plural for either 0
                // or > 1 units.
                if (scalar == 0 || scalar > 1)
                {
                    retval.format("s");
                }
            }
        }

        // If nothing was printed out, then print out
        // 0 millisecs.
        if (startFlag == false)
        {
            retval.format("0 millisecs");
        }

        return (retval.toString());
    } // end of FormatTime(long)

    // Place the application information in the log.
    private void logHeader(final PrintWriter pw)
    {
        pw.print(_appName);
        pw.print(" v. ");
        pw.print(_version);

        if (_buildDate != null)
        {
            pw.format(
                " (build %1$tm/%1$td/%1$tY at %1$tH:%1$tM:%1$tS)%n",
                _buildDate);
            pw.println(")");
        }

        if (_boilerPlate != null)
        {
            pw.println();
            pw.println(_boilerPlate);
        }

        pw.println();
        pw.print("Current working directory is ");
        pw.print(System.getProperty("user.dir"));
        pw.println(".");
        pw.print("Class path is ");
        pw.print(System.getProperty("java.class.path"));
        pw.println(".");

        return;
    } // end of logHeader(PrintWriter)
    // Create the report timer task and start it running.

    private void startReportTimer()
    {
        final long frequency = _frequency.getFrequency();
        final long reportDelay =
            (frequency -
                 (System.currentTimeMillis() % frequency));

        _reportTimer = new TimerTask(this);
        _timer.scheduleAtFixedRate(_reportTimer,
                                   reportDelay,
                                   frequency);

        return;
    } // end of startReportTimer()
    // Count up all the active threads in this JVM.

    private int countThreads()
    {
        ThreadGroup group =
            Thread.currentThread().getThreadGroup();
        ThreadGroup parent;

        // First, find the root thread group. This is the group
        // with no parent.
        while ((parent = group.getParent()) != null)
        {
            group = parent;
        }

        // We are now king of the hill. Now have the root thread
        // group return the number of child threads.
        // The last thread is always null so decrement the
        // thread count by one.
        return (group.activeCount() - 1);
    } // end of countThreads()
    // Print out the active threads in this JVM.

    private void outputThreads(final PrintWriter pw)
    {
        ThreadGroup group =
            Thread.currentThread().getThreadGroup();
        ThreadGroup parent;
        int threadCount;

        // First, find the root thread group. This is the group
        // with no parent.
        while ((parent = group.getParent()) != null)
        {
            group = parent;
        }

        // We are now king of the hill. Now have the root thread
        // group return all the child threads.
        threadCount = group.activeCount();

        if (threadCount == 0)
        {
            pw.println("Active threads: None");
        }
        else
        {
            final Thread[] threads = new Thread[threadCount];
            int index;
            Thread thread;

            group.enumerate(threads);

            // The last thread is always null so decrement the
            // thread count by one.
            --threadCount;

            for (index = 0; index < threadCount; ++index)
            {
                thread = threads[index];

                if (thread != null)
                {
                    pw.print("[");
                    pw.print(thread.getName());
                    pw.print("] ");
                    pw.print(thread.getClass().getName());
                    pw.print(" (");

                    if (thread.isAlive() == true)
                    {
                        pw.print("alive, ");
                    }
                    else
                    {
                        pw.print("dead, ");
                    }

                    if (thread.isDaemon() == true)
                    {
                        pw.print("daemon, ");
                    }
                    else
                    {
                        pw.print("non-daemon, ");
                    }

                    if (thread.isInterrupted() == true)
                    {
                        pw.print("interrupted, ");
                    }
                    else
                    {
                        pw.print("not interrupted, ");
                    }

                    pw.print(thread.getState());

                    pw.println(")");
                }
            }
        }

        return;
    } // end of outputThreads(PrintWriter)

//---------------------------------------------------------------
// Member data.
//

    // Set this flag to true when the application information
    // has been set. The application information may only
    // be set once.
    private boolean _initFlag;

    // The application's name, version, build date and copyright
    // boilerplate.
    private String _appName;
    private String _version;
    private Date _buildDate;
    private String _boilerPlate;

    // Write the status report to this log.
    private Logger _logger;

    // Write the status report to this log level.
    private Level _level;

    // Generate the status report at this frequency.
    private ReportFrequency _frequency;

    // The registered status reporters list.
    private final List<StatusReporter> _reporters;

    // When this timer expires, generate the status report.
    private TimerTask _reportTimer;

    // If this flag is true, then print out the status of all
    // active threads. This flag is false by default.
    private boolean _dumpThreadsFlag;

    //-----------------------------------------------------------
    // Statics.
    //

    // Make one and only one StatusReport instance.
    private static StatusReport _instance;
    private static final Lock _ctorMutex = new ReentrantLock();

    // Remember when this application was started.
    private static final Date _startTime = new Date();

    // Used by FormatTime() to print out elasped time in
    // human-readable form.
    private static final List<TimeUnit> _timeUnitList;

    private static final Timer _timer =
        new Timer("ReportTimer", true);

    //-----------------------------------------------------------
    // Constants
    //
    private static final int MILLIS_PER_DAY = 86400000;
    private static final int MILLIS_PER_HOUR = 3600000;
    private static final int MILLIS_PER_MINUTE = 60000;
    private static final int MILLIS_PER_SECOND = 1000;

    static
    {
        _timeUnitList = new ArrayList<>();

        _timeUnitList.add(
            new TimeUnit("day", MILLIS_PER_DAY));
        _timeUnitList.add(
            new TimeUnit("hour", MILLIS_PER_HOUR));
        _timeUnitList.add(
            new TimeUnit("minute", MILLIS_PER_MINUTE));
        _timeUnitList.add(
            new TimeUnit("second",  MILLIS_PER_SECOND));
        _timeUnitList.add(
            new TimeUnit("millisec",   1));
    } // end of static

    //-----------------------------------------------------------
    // Enums.
    //

    /**
     * Enumerates the allowed status report frequencies:
     * <ol>
     *   <li>
     *     FIVE_MINUTE: generate a report every five minutes on
     *     the five minute.
     *   </li>
     *   <li>
     *     TEN_MINUTE: generate a report every ten minutes on the
     *     ten minute;
     *   </li>
     *   <li>
     *     FIFTEEN_MINUTE: generate a report every fifteen
     *     minutes on the quarter hour.
     *   </li>
     *   <li>
     *     TWENTY_MINUTE: generate a report every twenty minutes
     *     on the twenty minute.
     *   </li>
     *   <li>
     *     THIRTY_MINUTE: generate a report every thirty minutes
     *     on the half hour.
     *   </li>
     *   <li>
     *     HOURLY: generate a report every hour on the hour.
     *   </li>
     * </ol>
     */
    public enum ReportFrequency
    {
        /**
         * Report every five minutes.
         */
        FIVE_MINUTE(300000),

        /**
         * Report every ten minutes.
         */
        TEN_MINUTE(600000),

        /**
         * Report every fifteen minutes.
         */
        FIFTEEN_MINUTE(900000),

        /**
         * Report every twenty minutes.
         */
        TWENTY_MINUTE(1200000),

        /**
         * Report every thirty minutes.
         */
        THIRTY_MINUTE(1800000),

        /**
         * Report every sixty minutes.
         */
        HOURLY(3600000);

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        // Creates a report frequecy instance.
        private ReportFrequency(final long frequency)
        {
            _frequency = frequency;
        } // end of ReportFrequency(long)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        /**
         * Returns the report frequency in milliseconds.
         * @return the report frequency in milliseconds.
         */
        public long getFrequency()
        {
            return (_frequency);
        } // end of getFrequency()

        //
        // end of Get methods.
        //-------------------------------------------------------

    //-----------------------------------------------------------
    // Member data.
    //

        // The reporting frequency in milliseconds.
        private final long _frequency;
    } // end of enum ReportFrequency

//---------------------------------------------------------------
// Inner classes.
//

    // A time unit has a name and a duration in milliseconds.
    private static final class TimeUnit
    {
    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private TimeUnit(final String name, final long unitTime)
        {
            _name = name;
            _unitTime = unitTime;
        } // end of TimeUnit(String, long)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get methods.
        //

        private String getName()
        {
            return (_name);
        } // end of getName()

        private long getUnitTime()
        {
            return (_unitTime);
        } // end of getUnitTime()

        //
        // end of Get methods.
        //-------------------------------------------------------

    //-----------------------------------------------------------
    // Member data.
    //

        private final String _name;
        private final long _unitTime;
    } // end of class TimeUnit
} // end of class StatusReport

//
// CHANGE LOG
// $Log: StatusReport.java,v $
// Revision 1.11  2006/09/30 16:18:12  charlesr
// Corrected comments.
//
// Revision 1.10  2005/12/28 00:18:15  charlesr
// Add thread state to thread output.
//
// Revision 1.9  2005/12/20 01:23:38  charlesr
// Replaced notifyAll with notify.
//
// Revision 1.8  2005/11/05 15:57:11  charlesr
// Gave _timer a name: "ReportTimer".
//
// Revision 1.7  2005/07/22 01:52:07  charlesr
// Moved to Java 5:
// + Added generics.
// + Using Java enums for report frequency.
// + Replaced StringBuffer with StringBuilder.
//
// Revision 1.6  2005/01/09 20:04:40  charlesr
// Replace net.sf.eBus.util.TimerKeeper with java.util.Timer.
//
// Revision 1.5  2004/12/13 16:04:31  charlesr
// Correct countThreads(), thread count is one too many.
//
// Revision 1.4  2004/09/29 00:39:11  charlesr
// + Added appropriate synchronization to critical sections.
// + Added an optional "boilerplate" to application which is
//   output both at start time and in every report. Generally
//   used for copyright notification.
// + Added a special header output at startup containing
//   application information.
// + Added a "thread dump" flag. If true, then information on all
//   active threads is added to report.
//
// Revision 1.3  2004/07/28 13:33:41  charlesr
// Exception occurs when a status reporter deregisters when
// status report is generated. This is corrected by copying the
// reporters list and iterating over the copy.
//
// Revision 1.2  2004/04/27 14:05:29  charlesr
// Using net.sf.eBus.util.TimerKeeper's singleton
// java.util.Timer.
//
// Revision 1.1  2004/01/28 02:06:58  charlesr
// Use StringWriter and PrintWriter to collect status reports
// instead of a StringBuffer.
//
// Revision 1.0  2003/11/20 01:49:51  charlesr
// Initial revision
//
