//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.test;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import org.junit.runners.BlockJUnit4ClassRunner;
import org.junit.runners.model.FrameworkMethod;
import org.junit.runners.model.InitializationError;

/**
 * This class extends
 * {@code org.junit.runners.BlockJUnit4ClassRunner} and overrides
 * the {@code computeTestMethods()} method to guarantee that
 * test methods are run in a user-defined order using the
 * {@link TestIndex} annotation. While JUnit discourages test
 * method dependencies because increases test maintenance
 * difficultly, breaking a complex test down to a series of
 * simple, dependent test methods is necessary at times.
 * <p>
 * An example using this class runner:
 * <pre><code>
 * &#64;RunWith(OrderedRunner.class)
 * public final class MyTest
 * {
 *     &#64;Test
 *     &#64;TestIndex(index=0)
 *     public void test0()
 *     {
 *         ...
 *     }
 *
 *     &#64;Test
 *     &#64;TestIndex(index=1)
 *     public void test1
 *     {
 *         ...
 *     }
 * }</code></pre>
 * <p>
 * <strong>Note:</strong> this is not meant to replace
 * {@code org.junit.runner.manipulation.Sortable} interface since
 * that interface should not be used to cope with test method
 * dependencies (as per the JUnit documentation).
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public class OrderedRunner
    extends BlockJUnit4ClassRunner
{
//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new OrderedRunner instance for the given test
     * class.
     * @param clazz the Java class containing the test methods.
     * @throws InitializationError
     * if this constructor fails.
     */
    public OrderedRunner(final Class<?> clazz)
        throws InitializationError
    {
        super (clazz);
    } // end of OrderedRunner(Class)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // BlockJUnit4ClassRunner Override Methods.
    //

    /**
     * Returns the test methods list sorted according to their
     * {@link TestIndex test index} values.
     * @return the sorted test method list.
     */
    @Override
    protected List<FrameworkMethod> computeTestMethods()
    {
        final List<FrameworkMethod> retval =
            new ArrayList<>(super.computeTestMethods());

        Collections.sort(retval, FM_COMPARATOR);

        return (retval);
    } // end of computeTestMethod()

    //
    // end of BlockJUnit4ClassRunner Override Methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * The FrameworkMethod comparator singleton. Used to sort
     * the default test method list by test index values.
     */
    private static final Comparator<FrameworkMethod>
        FM_COMPARATOR = new FMComparator();

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Used to compare two test methods based on their
     * {@link TestIndex @TestIndex} annotation values.
     */
    private static final class FMComparator
        implements Comparator<FrameworkMethod>,
                   Serializable
    {
    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Default constructor.
         */
        public FMComparator()
        {}

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparator Interface Implementation.
        //

        /**
         * Returns a value &lt;, equal to, or &gt; zero if
         * {@code t0} test index is &lt;, equal to, or &gt;
         * {@code t1} test index.
         * @param t0 the first test method.
         * @param t1 the second test method.
         * @return an integer value &lt;, equal to, or &gt; zero.
         */
        @Override
        public int compare(final FrameworkMethod t0,
                           final FrameworkMethod t1)
        {
            final TestIndex i0 =
                t0.getAnnotation(TestIndex.class);
            final TestIndex i1 =
                t1.getAnnotation(TestIndex.class);
            int retval = -1;

            if (i0 != null && i1 != null)
            {
                retval = (i0.index() - i1.index());
            }

            return (retval);
        } // end of compare(FrameworkMethd, FrameworkMethod)

        //
        // end of Comparator Interface Implementation.
        //-------------------------------------------------------

    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Constants.
        //

        private static final long serialVersionUID = 0x020204L;
    } // end of class FMComparator
} // end of class OrderedRunner
