//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import java.util.Collection;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBusx.time.EInterval;
import net.sf.eBusx.time.EInterval.TimeLocation;

/**
 * This interface provides {@link EHistoricPublishFeed} the
 * ability to persist notification messages and to retrieve
 * those message for a given historic time interval. Notification
 * messages are persisted when an
 * {@link net.sf.eBus.client.EPublisher historic publisher}
 * {@link EHistoricPublishFeed#publish(ENotificationMessage) publishes}
 * a notification message. Notification messages are retrieved
 * when {@code EHistoricPublishFeed} receives a
 * {@link HistoricRequest request for historic messages}.
 * <p>
 * Please note that this message store is not only used to
 * persist application notification messages but also
 * {@link PublishStatusEvent} messages.
 * </p>
 * <p>
 * An {@code IEMessageStore} instance may be shared among
 * multiple {@code EHistoricPublishFeed}s but the implementation
 * is responsible for maintaining thread-safety.
 * </p>
 * <p>
 * When building an {@code EHistoricPublishFeed}, a
 * non-{@code null}, open message store must be passed to
 * {@link EHistoricPublishFeed.Builder#messageStore(IEMessageStore)}.
 * Failure to do so results in an exception. The appropriate
 * opening and closing of the {@code IEMessageStore} is outside
 * the scope of the historic publish feed and left up to the
 * application. The restriction is that the message store must
 * be opened prior to opening the historic publish feed and
 * closed after closing the historic publish feed.
 * </p>
 * <p>
 * It is left up to the implementation to decide how messages are
 * persisted or retrieved.
 * </p>
 * <h3>Publishing multiple updates to one subject.</h3>
 * Consider the following: the status for five different
 * connections are published and stored for a single subject.
 * Subscribers need only subscribe to a single subject to receive
 * updates on all five connections. The problem is retrieving
 * the latest update notification for these five connections.
 * If {@link #retrieve(EInterval)} for a {now, now} interval is
 * handled literally, then only the very latest notification is
 * returned which means the subscriber is missing status for the
 * four other notifications.
 * <p>
 * A better solution is to interpret this interval as the latest
 * notification for each of the five connections - which is
 * what the subscriber expects. It is up to the
 * {@code IEMessageStore} implementation to decide how to
 * achieve this.
 * </p>
 *
 * @see EHistoricPublishFeed
 * @see EInterval
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IEMessageStore
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Returns {@code true} if this message store is open and
     * ready to store or retrieve notification messages.
     * @return {@code true} if message store is open.
     */
    boolean isOpen();

    /**
     * Returns message key associated with message store.
     * @return message store's associated key.
     */
    EMessageKey key();

    /**
     * Persists given notification message to store.
     * @param message persist this message.
     * @throws NullPointerException
     * if {@code message} is {@code null}.
     *
     * @see #retrieve(EInterval)
     */
    void store(ENotificationMessage message);

    /**
     * Retrieves historic messages as defined by given date/time
     * interval. Returned collection may be empty but never
     * {@code null}.
     * <p>
     * There are three interval types:
     * </p>
     * <ol>
     *   <li>
     *     {{@link TimeLocation#PAST past},
     *     {@link TimeLocation#PAST past}}: retrieved
     *     notifications are entirely in the past.
     *   </li>
     *   <li>
     *     {{@link TimeLocation#PAST past},
     *     {@link TimeLocation#NOW now}}: retrieve notifications
     *     are from the past to the most recently stored
     *     notification.
     *   </li>
     *   <li>
     *     {{@link TimeLocation#NOW now},
     *     {@link TimeLocation#NOW now}}: retrieve only the
     *     latest notifications. See {@link IEMessageStore}
     *     class documentation on this specific retrieval.
     *   </li>
     * </ol>
     * <p>
     * Please note that implementation are <em>not</em> required
     * to validate {@code interval} since this is performed by
     * {@link net.sf.eBus.feed.historic.EHistoricPublishFeed}.
     * That said, such validation is encourage if store is
     * used outside of eBus historic feed framework.
     * </p>
     * @param interval date/time interval.
     * @return iterator over historic messages. Does <em>not</em>
     * return {@code null} but may return iterator where
     * {@code Iterator.hasNext()} returns {@code false}.
     *
     * @see #store(ENotificationMessage)
     */
    Collection<ENotificationMessage> retrieve(EInterval interval);
} // end of interface IEMessageStore

