//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import net.sf.eBus.client.EObject;
import net.sf.eBus.feed.historic.EHistoricSubscribeFeed.HistoricFeedState;
import net.sf.eBus.messages.ENotificationMessage;

/**
 * Classes wanting to receive historic notifications from an
 * {@link EHistoricSubscribeFeed historic subscribe feed} need to
 * implement this interface.
 * <p>
 * An {@code EHistoricSubscribeFeed} is a hybrid objects which
 * sits between an {@code IEHistoricSubscriber} and historic
 * feed publishers. This historic subscribe feed collects both
 * historic (previously published) and newly published
 * notifications and feeds them back to the historic subscriber
 * ordered by time. This playback also includes changes in a
 * publisher's feed status. Method
 * {@link #notify(ENotificationMessage, EHistoricSubscribeFeed)}
 * is used to deliver both historic and live notifications to
 * the historic subscriber.
 * </p>
 * <p>
 * Method {@link #feedStatus(PublishStatusEvent, EHistoricSubscribeFeed)}
 * reports a feed status for a particular historic publisher
 * identified by {@link PublishStatusEvent#publisherId} and
 * {@link PublishStatusEvent#feedState} the publisher's feed
 * state. Unlike {@code IESubscribeFeed}, feed state is tracked
 * per publisher and not for all publishers collectively.
 * </p>
 * <p>
 * If an historic subscribe feed has a fixed ending time, then
 * when the historic subscribe feed has reached that end time
 * <em>or</em> if the historic feed fails to retrieve past
 * notification messages, then the feed calls
 * {@link #feedDone(EHistoricSubscribeFeed.HistoricFeedState, EHistoricSubscribeFeed)}
 * to let the historic subscriber know that no more notifications
 * or feed state changes will be delivered.
 * </p>
 * <p>
 * <strong>Note:</strong> if the historic feed end time is marked
 * as on-going {@link EHistoricSubscribeFeed.Builder#toForever()},
 * then {@code historicFeedStatus} is never called with a down
 * feed state because the feed ends only when the historic
 * subscribe feed is unsubscribed or closed.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IEHistoricSubscriber
    extends EObject
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Reports historic subscribe feed is completed and no more
     * events will be posted from the feed. Feed state will be
     * either {@code DONE_SUCCESS} or {@code DONE_ERROR}. If feed
     * experienced an error, the exception associated with the
     * error can be retrieve by calling
     * {@link EHistoricSubscribeFeed#errorCause()}.
     * <p>
     * This method is <em>not</em> called if historic subscribe
     * feed end time is set to
     * {@link net.sf.eBusx.time.EInterval.TimeLocation#ON_GOING}.
     * </p>
     * @param feedState historic subscribe feed status. Will be
     * either {@link HistoricFeedState#DONE_SUCCESS} or
     * {@link HistoricFeedState#DONE_ERROR}.
     * @param feed historic subscribe feed.
     */
    default void feedDone(HistoricFeedState feedState,
                          EHistoricSubscribeFeed feed)
    {
        throw (
            new UnsupportedOperationException(
                "historicFeedStatus not implemented"));
    } // end of feedDone(...)

    /**
     * Reports a publisher feed state change for the underlying
     * notification message feed.
     * @param event contains latest publisher feed state for a
     * given message key.
     * @param feed historic subscribe feed.
     */
    default void feedStatus(PublishStatusEvent event,
                            EHistoricSubscribeFeed feed)
    {
        throw (
            new UnsupportedOperationException(
                "feedStatus not implemented"));
    } // end of feedStatus(...)

    /**
     * Reports latest notification message.
     * @param msg latest notification message.
     * @param feed historic subscribe feed.
     */
    default void notify(ENotificationMessage msg,
                        EHistoricSubscribeFeed feed)
    {
        throw (
            new UnsupportedOperationException(
                "notify not implemented"));
    } // end of notify(...)
} // end of interface IEHistoricSubscriber

