//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import net.sf.eBus.client.EFeed;
import net.sf.eBus.client.EFeed.FeedScope;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.EObject;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.util.ValidationException;
import net.sf.eBus.util.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * Base class for {@link EHistoricPublishFeed} and
 * {@link EHistoricSubscribeFeed} containing the data members
 * common to each. This includes:
 * <ul>
 *   <li>
 *     feed name (used for logging purposes),
 *   </li>
 *   <li>
 *     {@code EObject} containing (or owning) this feed hybrid
 *     object,
 *   </li>
 *   <li>
 *     notification and historic request message keys, and
 *   </li>
 *   <li>
 *     feed scope (applies to notification, request, and reply
 *     feeds).
 *   </li>
 * </ul>
 * <p>
 * This class also contains data members which track whether the
 * feed is open, in place (that is, advertised or subscribed),
 * and the current historic feed state.
 * </p>
 *
 * @param <T> historic feed owner type.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public abstract class EAbstractHistoricFeed<T extends EObject>
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        LoggerFactory.getLogger(EAbstractHistoricFeed.class);

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * eBus object name.
     */
    protected final String mName;

    /**
     * This historic feed works on behalf of this eBus object.
     */
    protected final T mOwner;

    /**
     * Notifications published on this message key.
     */
    protected final EMessageKey mKey;

    /**
     * {@link PublishStatusEvent} published on this message key.
     */
    protected final EMessageKey mStatusKey;

    /**
     * Historic message request feed key.
     */
    protected final EMessageKey mRequestKey;

    /**
     * Notification and request feed scope.
     */
    protected final EFeed.FeedScope mScope;

    /**
     * Set to {@code true} when this feed is open and started
     * and {@code false} when stopped.
     */
    protected boolean mIsOpen;

    /**
     * Set to {@code true} when historic feed is up.
     */
    protected boolean mInPlace;

    /**
     * Current historic feed state.
     */
    protected EFeedState mFeedState;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new historic feed instance based on builder
     * settings.
     * @param builder contains historic feed settings.
     */
    protected EAbstractHistoricFeed(final Builder<T, ?, ?> builder)
    {
        mName = builder.mName;
        mOwner = builder.mOwner;
        mKey = builder.mKey;
        mStatusKey = builder.mStatusKey;
        mRequestKey = builder.mRequestKey;
        mScope = builder.mScope;

        mIsOpen = true;
        mInPlace = false;
        mFeedState = EFeedState.UNKNOWN;
    } // end of EAbstractHistoricFeed(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Performs actual work to close historic feed.
     */
    protected abstract void doClose();

    //
    // end of Abstract Method Declarations.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if this feed is open and
     * {@code false} if closed. Once an historic feed is closed,
     * it cannot be opened again.
     * @return {@code true} if feed is open.
     *
     * @see #inPlace()
     */
    public final boolean isOpen()
    {
        return (mIsOpen);
    } // end of isOpen()

    /**
     * Returns {@code true} if this feed is "in place" (that is,
     * subscribed or advertised) and {@code false} if not.
     * @return {@code true} if feed is in place.
     */
    public final boolean inPlace()
    {
        return (mInPlace);
    } // end of inPlace()

    /**
     * Returns notification message key.
     * @return notification message key.
     */
    public final EMessageKey key()
    {
        return (mKey);
    } // end of key()

    /**
     * Returns notification message key used to publish
     * {@link PublishStatusEvent} messages.
     * @return {@link PublishStatusEvent} message key.
     */
    public final EMessageKey publisherStatusKey()
    {
        return (mStatusKey);
    } // end of publisherStatusKey()

    /**
     * Returns historic request message key.
     * @return historic request message key.
     */
    public final EMessageKey requestKey()
    {
        return (mRequestKey);
    } // end of requestKey()

    /**
     * Returns historic notification and request feed scope.
     * @return feed scope.
     */
    public final FeedScope scope()
    {
        return (mScope);
    } // end of scope()

    /**
     * Returns current historic subscribe feed state.
     * @return current feed state.
     */
    public final EFeedState feedState()
    {
        return (mFeedState);
    } // end of feedState()

    /**
     * Returns eBus object owning this historic feed.
     * @return historic subscribe feed owner.
     */
    @VisibleForTesting
    /* package */ final T owner()
    {
        return (mOwner);
    } // end of owner()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Permanently closes historic feed. This historic feed
     * cannot be re-opened or used to publish/retrieve historic
     * notifications after this.
     */
    public final void close()
    {
        sLogger.debug("{}: closing historic feed.", mName);

        mIsOpen = false;
        mInPlace = false;
        mFeedState = EFeedState.UNKNOWN;

        try
        {
            doClose();
        }
        catch(Exception jex)
        {
            sLogger.warn("{}: failure closing historic feed:",
                         mName,
                         jex);
        }

        sLogger.info("{}: historic feed closed.", mName);
    } // end of close()

    /**
     * Closes feed if not {@code null}.
     * @param feed close this feed.
     */
    protected final void closeFeed(final EFeed feed)
    {
        if (feed != null)
        {
            feed.close();
        }
    } // end of closeFeed(EFeed)

    //
    // end of Set Methods.
    //-----------------------------------------------------------

//---------------------------------------------------------------
// Member data.
//

    /**
     * Base class for {@link EHistoricPublishFeed.Builder} and
     * {@link EHistoricSubscribeFeed.Builder} subclasses.
     * Provides methods for setting feed name, feed owner, and
     * feed scope. Also defines method for building target feed
     * type and validating common data member settings.
     *
     * @param <T> historic feed owner type.
     * @param <F> subclass feed type.
     * @param <B> subclass builder type.
     */
    protected static abstract class Builder<T extends EObject,
                                            F extends EAbstractHistoricFeed<T>,
                                            B extends Builder<T, F, ?>>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Statics.
        //

        /**
         * Incremented when a new historic feed is created.
         */
        protected static final AtomicInteger sFeedIndex =
            new AtomicInteger();

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * This historic notification hybrid object feed works
         * for this owner.
         */
        protected final T mOwner;

        /**
         * Historic feed class being built. Used only when
         * throwing a validation exception.
         */
        protected final Class<F> mFeedClass;

        /**
         * Notification message key.
         */
        protected final EMessageKey mKey;

        /**
         * {@code PublishStatusEvent} message key.
         */
        protected final EMessageKey mStatusKey;

        /**
         * Historic notification request key.
         */
        protected final EMessageKey mRequestKey;

        /**
         * eBus object name. If not defined, then set to a
         * default value based on owner name and message key.
         */
        protected String mName;

        /**
         * Notification and reply feed scope.
         */
        protected FeedScope mScope;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Initializes historic feed abstract elements with the
         * given parameters. Sets message store based on the
         * message key.
         * @param owner eBus object owning this historic
         * notification feed.
         * @param feedClass
         * @param key stores and retrieves messages for this
         * class.
         */
        @SuppressWarnings ("unchecked")
        protected Builder(final T owner,
                          final Class<F> feedClass,
                          final EMessageKey key)
        {
            mOwner = owner;
            mFeedClass = feedClass;
            mKey = key;

            // Status and request keys use same subject as the
            // notification key.
            mStatusKey =
                new EMessageKey(
                    PublishStatusEvent.class, mKey.subject());
            mRequestKey =
                new EMessageKey(
                    HistoricRequest.class, mKey.subject());
        } // end of AbstractBuilder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Declarations.
        //

        /**
         * Returns subclass builder {@code this} reference.
         * @return subclass builder {@code this} reference.
         */
        protected abstract B self();

        /**
         * Returns default historic feed name. Used when feed
         * name is not explicitly set.
         * @return default historic feed name.
         */
        protected abstract String generateName();

        /**
         * Returns a new historic feed instance based on builder
         * settings.
         * @return new historic feed instance.
         */
        protected abstract F buildImpl();

        //
        // end of Abstract Method Declarations.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets historic feed name. May be {@code null} or
         * an empty string. If so, then name will be set to
         * default value.
         * @param name historic feed name.
         * @return {@code this Builder} instance.
         */
        public final B name(final @Nullable String name)
        {
            mName = name;

            return (self());
        } // end of name(String)

        /**
         * Sets historic feed scope. The same scope is
         * used for both notification and request feeds. This
         * means notifications are sent to or received from
         * underlying feeds within this scope.
         * @param scope historic feed scope.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code scope} is {@code null}.
         */
        public final B scope(final FeedScope scope)
        {
            mScope =
                Objects.requireNonNull(scope, "scope is null");

            return (self());
        } // end of scope(FeedScope)

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Returns a new historic feed instance created from
         * builder settings. Validates builder settings prior to
         * building target historic feed instance.
         * <p>
         * If historic feed name is not set, then generates a
         * default feed name.
         * </p>
         * @return new historic feed instance.
         * @throws ValidationException
         * if this builder's settings are not valid.
         */
        public final F build()
        {
            final Validator problems = new Validator();

            // Validate
            validate(problems).throwException(mFeedClass);

            if (Strings.isNullOrEmpty(mName))
            {
                mName = generateName();
            }

            return (buildImpl());
        } // end of build()

        /**
         * Validates that historic feed owner and scope are
         * set. Returns {@code problems} parameter to allow for
         * {@code validate} method chaining.
         * @param problems place validation failures into this
         * list.
         * @return @code problems} to allow {@code validate}
         * method chaining.
         */
        protected Validator validate(final Validator problems)
        {
            // An historic feed is valid if:
            // + EObject owner is not null,
            // + feed scode is not null,
            return (problems.requireNotNull(mOwner, "owner")
                            .requireNotNull(mScope, "scope"));
        } // end of validate(Validator)
    } // end of class Builder<T extends EObject>
} // end of class EAbstractHistoricFeed
