//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import java.util.Collection;
import java.util.Objects;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.util.Validator;


/**
 * This reply message contains one or more historic notification
 * message in response to an
 * {@link HistoricRequest historic notification request}.
 * These notifications are in publish order for a given publisher
 * identifier. If there are further notifications to be returned
 * then {@link EReplyMessage#replyStatus reply status} is set
 * to {@code EReplyMessage.ReplyStatus.OK_CONTINUING}; if the
 * notification playback is completed, then reply status is set
 * to {@code EReplyMessage.ReplyStatus.OK_FINAL}. If an error
 * occurs processing the historic notification request, then
 * {@link EReplyMessage} is sent in reply with
 * {@code EReplyMessage.ReplyStatus.ERROR} and
 * {@link EReplyMessage#replyReason} explaining the error.
 * <p>
 * An example historic notification reply creation (note that
 * replies may only be created using a {@link Builder builder}
 * instance.
 * </p>
 * <pre><code>final HistoricReply.Builder builder = HistoricReply.builder();
final Collection&lt;ENotificationMessage&gt; notifications = ...; // Notifications sent in reply.
final HistroricReply reply = builder.subject(mReplySubject)
                                    .timestamp(Instant.now())
                                    .notifications(notifications)
                                    .build()</code></pre>
 * <p>
 * This reply is generally used by {@link EHistoricPublishFeed}
 * to send historic notification messages back to requester.
 * </p>
 * <p>
 * Note: the notification messages may contain
 * {@link PublishStatusEvent} along with the published
 * notification messages. This allows the requester to determine
 * when a publisher's feed is down, meaning that the notification
 * stream <em>may</em> be missing messages.
 * </p>
 *
 * @see HistoricRequest
 * @see EHistoricPublishFeed
 * @see EHistoricSubscribeFeed
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class HistoricReply
    extends EReplyMessage
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x060300L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * One of more historic notification messages. Will not be
     * an empty array.
     */
    public final ENotificationMessage[] notifications;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new historic reply instance based on builder's
     * settings.
     * @param builder contains reply message contents.
     */
    private HistoricReply(final Builder builder)
    {
        super (builder);

        this.notifications = builder.mNotifications;
    } // end of HistoricReply(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns textual representation of historic notification
     * reply. Reports number of notifications in reply but not
     * the notification messages themselves.
     * @return historic notification reply as text.
     */
    @Override
    public String toString()
    {
        final StringBuilder retval = new StringBuilder();

        return (retval.append(super.toString())
                      .append("\n        # notifications: ")
                      .append(notifications.length)
                      .toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns new historic reply message builder instance. It is
     * not recommended to re-use a builder instance to create
     * more than one historic reply.
     * @return new historic reply message builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Builder class for {@code HistoricReply} message instances.
     * A {@code Builder} is acquired using the
     * {@link HistoricReply#builder()} method. Each builder
     * instance should be used to build only one
     * {@code HistoricReply} instance.
     * <p>
     * {@code Builder} require notification array to be
     * non-{@code null} and non empty.
     * </p>
     *
     * @see HistoricReply
     */
    public static final class Builder
        extends EReplyMessage.Builder<HistoricReply, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Constants.
        //

        /**
         * Empty array used to transform notification message
         * collection to any array.
         */
        private static final ENotificationMessage[] COPY_ARRAY =
            new ENotificationMessage[0];

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Notification messages returned in the reply. May
         * not be {@code null} or empty.
         */
        private ENotificationMessage[] mNotifications;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new historic reply builder. Constructor is
         * private since builder instances are created by
         * {@link HistoricReply#builder()} method only.
         */
        private Builder()
        {
            super (HistoricReply.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Checks if this builder is correctly configured before
         * building target message instance. A valid
         * {@code HistoricReply} requires that notifications
         * are set to a non-{@code null}, non-empty array.
         * @param problems place invalid configuration settings
         * into this problems list.
         * @return {@code problems} is returned to all method
         * chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            return (super.validate(problems)
                         .requireNotNull(mNotifications,
                                         "notifications"));
        } // end of validate(Validator)

        /**
         * Returns a new historic notification reply based on
         * {@code this Builder}'s settings.
         * @return new historic notification reply.
         */
        @Override
        protected HistoricReply buildImpl()
        {
            return (new HistoricReply(this));
        } // end of buildImpl()

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets historic messages returned in this reply.
         * Messages array may not be {@code null} or empty.
         * @param messages reply contains these messages.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code messages} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code messages} array is empty.
         * @throws IllegalStateException
         * if messages array was previously set.
         */
        public Builder notifications(final ENotificationMessage[] messages)
        {
            Objects.requireNonNull(messages, "messages is null");

            final int size = messages.length;

            if (size == 0)
            {
                throw (
                    new IllegalArgumentException(
                        "messages is empty"));
            }

            if (mNotifications != null)
            {
                throw (
                    new IllegalStateException(
                        "messages already set"));
            }

            mNotifications = new ENotificationMessage[size];
            System.arraycopy(
                messages, 0, mNotifications, 0, size);

            return (this);
        } // end of notification(ENotificationMessage[])

        /**
         * Sets historic messages returned in this reply.
         * Messages collection may not be {@code null} or empty.
         * @param messages reply contains these messages.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code messages} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code messages} is empty.
         * @throws IllegalStateException
         * if messages array was previously set.
         */
        public Builder notifications(final Collection<ENotificationMessage> messages)
        {
            Objects.requireNonNull(messages, "messages is null");

            if (messages.isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "messages is empty"));
            }

            if (mNotifications != null)
            {
                throw (
                    new IllegalStateException(
                        "messages already set"));
            }

            mNotifications = messages.toArray(COPY_ARRAY);

            return (this);
        } // end of notifications(Collection<>)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class HistoricReply
