//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Objects;
import javax.annotation.Nullable;
import static net.sf.eBus.client.EFeed.FEED_NOT_ADVERTISED;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.EPublishFeed;
import net.sf.eBus.client.EPublisher;
import net.sf.eBus.client.EReplier;
import net.sf.eBus.client.EReplyFeed;
import net.sf.eBus.client.IEPublishFeed;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.messages.EReplyMessage.ReplyStatus;
import net.sf.eBus.util.Validator;
import net.sf.eBusx.time.EInterval;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The historic publisher feed extends {@link EPublishFeed}'s
 * ability to publish notification messages to subscribers with
 * the ability to:
 * <ul>
 *   <li>
 *     persist notification messages to a {@link IEMessageStore}
 *     implementation and
 *   </li>
 *   <li>
 *     handle
 *     {@link HistoricRequest historic notification requests}
 *     retrieving previously published messages from message
 *     store.
 *   </li>
 * </ul>
 * <p>
 * Please note that {@link net.sf.eBus.client.ESubscriber}s are
 * able to seamlessly access notification messages posted to an
 * historic publish feed. There is no distinction between
 * live-only and historic feeds. Of course an {@code ESubscriber}
 * cannot access previously published notification messages.
 * </p>
 * <p>
 * <strong>Note:</strong> an {@code IEMessageStore} instance may
 * be shared between multiple {@code EHistoricPublishFeed}s but
 * the implementation is responsible for maintaining thread
 * safety.
 * </p>
 * <p>
 * {@code EHistoricPublishFeed} is similar to using an
 * {@code EPublishFeed}:
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 1:</strong> Implement
 * {@link IEHistoricPublisher} interface.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 2:</strong>
 * Open an historic publish feed using an
 * {@link EHistoricPublishFeed.Builder} instance (obtained using
 * {@link EHistoricPublishFeed#builder(EMessageKey, IEHistoricPublisher)})
 * for a given {@code IEHistoricPublisher} instance,
 * {@link EMessageKey type+topic message key},
 * {@link net.sf.eBus.client.EFeed.FeedScope feed scope}, and
 * {@code IEMessageStore} instance.
 * </p>
 * <p>
 * <strong>Note:</strong> historic notification feeds are
 * dependent on setting a unique publisher identifier. This
 * allows historic feed subscribers to differentiate between
 * publishers.
 * </p>
 * <p>
 * Note that when opening an historic publish feed, the
 * configured message store <em>must</em> be open and ready to
 * persist and retrieve historic notification messages.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 3:</strong>
 * {@link #startup() Start up} historic publish feed. Call this
 * method directly rather than using
 * {@link net.sf.eBus.client.EFeed#register(net.sf.eBus.client.EObject)}
 * and
 * {@link net.sf.eBus.client.EFeed#startup(net.sf.eBus.client.EObject)}
 * because {@code EHistoricPublishFeed} is an eBus hybrid object
 * and runs in the {@code IEHistoricPublisher}'s dispatcher. The
 * underlying historic notification <strong>reply</strong> feed
 * is opened, advertised, and feed state set to up at this time.
 * This means the historic publish feed is able to process
 * historic notification requests even when the publish feed is
 * unadvertised.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 4:</strong>
 * {@link #advertise() Advertise} this publisher to eBus.
 * This allows eBus to match publishers with subscribers.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 5:</strong>
 * {@link #updateFeedState(EFeedState) Set feed state} to
 * {@link EFeedState#UP up}. This step <em>must</em> be done
 * prior to publishing any notifications. Put another way,
 * notifications cannot be published when the feed state is not
 * up.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 6:</strong>
 * {@link #publish(ENotificationMessage) Start publishing}
 * notifications. Note that this differs from
 * {@code EPublishFeed} where the publisher must wait for
 * subscribers prior to posting a notification. This is because
 * notifications need to be persisted despite the lack of
 * subscribers. This allows an {@link IEHistoricSubscriber} to
 * access previously published messages when opening its
 * {@link EHistoricSubscribeFeed historic subscribe feed}.
 * </p>
 * <p>
 * It is <em>highly</em> recommended that publishers set
 * {@link ENotificationMessage#position} values especially if
 * more than one notification is published per millisecond.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 7:</strong> When
 * shutting down the publisher {@link #unadvertise() retract}
 * notification advertisement and {@link #close() close} the
 * historic feed.
 * </p>
 * <h2>Example use of <code>EHistoricPublishFeed</code></h2>
 * <pre><code>import net.sf.eBus.client.EFeed.FeedScope;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.feed.historic.IEHistoricPublisher;
import net.sf.eBus.feed.historic.EHistoriicPublishFeed;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;

<strong style="color:ForestGreen">// Step 1: Implement IEHistoricPublisher interface.</strong>
public class CatalogPublisher implements IEHistoricPublisher {
    // Roll over notification position when this value is reached.
    private static final int MAX_POSITION = 1_000;

    // HistoricReply messages contain up to 10 notifications per reply message.
    private static final int MAX_NOTIFICATIONS_PER_REPLY = 10;

    // Publishes this notification message class/subject key.
    private final EMessageKey mKey;

    // Unique publisher identifier.
    private final long mPublisherId;

    // Catalog publisher name used for logging purposes.
    private final String mName;

    // Published messages remain within this scope.
    private final FeedScope mScope;

    // Message store for persisting and retrieving notification messages.
    private final IEMessageStore mStore;

    // Advertise and publish on this feed.
    private EHistoricPublishFeed mFeed;

    // Latest message position. Incremented on each notification publish.
    // Value is reset to zero when MAX_POSITION is reached.
    private int mPosition;

    public CatalogPublisher(final String subject,
                            final long pubId,
                            final FeedScope scope,
                            final IEMessageStore store) {
        mKey = new EMessageKey(CatalogUpdate.class, subject);
        mPublisherId = pubId;
        mName = (this.getClass()).getSimpleName + "-" + pubId;
        mScope = scope;
        mStore = store;
        mFeed = null;
        mPosition = 0;
    }

    &#64;Override
    public void startup() {
        try {
            <strong style="color:ForestGreen">// Step 2: Open publish feed. Place IEHistoricPublisher interface method overrides here.</strong>
            // This publisher overrides IEHistoricPublisher interface method.
            final EHistoricPublishFeed.Builder builder = EHistoricPublishFeed.builder(this, mKey);

            mFeed = builder.name(mName)
                           .scope(mScope)
                           .messageStore(mStore)
                           .notificationsPerReply(MAX_NOTIFICATIONS_PER_REPLY)
                           .build();

            <strong style="color:ForestGreen">// Step 3: Start up this publisher.</strong>
            mFeed.startup();

            <strong style="color:ForestGreen">// Step 4: Advertise this publisher to eBus.</strong>
            mFeed.advertise();

            <strong style="color:ForestGreen">// Step 5: Inform the world that this publisher's feed state is up.</strong>
            mFeed.updateFeedState(EFeedState.UP);
        } catch (IllegalArgumentException argex) {
            // Advertisement failed. Place recovery code here.
        }
    }

    // Note: this method is <em>never</em> called by EHistoricPublishFeed since the publishStatus is always up once
    // historic publish feed is opened, advertised, and publish feed state is set to up.
    &#64;Override
    public void publishStatus(final EFeedState feedState, final EPublishFeed feed) {
        ...
    }

    <strong style="color:ForestGreen">// Step 6: Start publishing notifications.</strong>
    public void updateProduct(final String productName, final Money price, final int stockQty) {
        if (mFeed != null &amp;&amp; mFeed.inPlace()) {
            mFeed.publish((CatalogUpdate.builder()).subject(mKey.subject)
                                                   .timestamp(Instant.now())
                                                   .publisherId(mPublisherId)
                                                   .position(mPosition)
                                                   .productName(productName)
                                                   .price(price)
                                                   .inStockQuantity(stockQty)
                                                   .build());

            ++mPosition;

            // Reset position?
            if (mPosition == MAX_POSITION) {
                mPosition = 0;
            }
        }
    }

    // Retract the notification feed.
    &#64;Override
    public void shutdown() {
        <strong style="color:ForestGreen">Step 7: On shutdown either unadvertise or close publish feed.</strong>
        if (mFeed != null) {
            // unadvertise() unnecessary since close() retracts an in-place advertisement.
            mFeed.close();
            mFeed = null;
        }
    }
}</code></pre>
 * <h1>Historic Publisher and Message Exhaust</h1>
 * eBus release 6.2.0 introduced
 * {@link net.sf.eBus.client.IMessageExhaust message exhaust interface}
 * which allows all messages (notification, request, reply)
 * flowing through eBus to be exhausted to persistent store. The
 * difference between this and an historic publisher is that
 * the historic publisher persists a notification message even if
 * it is not published to eBus which message exhaust persists
 * only those messages successfully published to eBus. The reason
 * for historic feeds is to allow subscribers to retrieve
 * historic and live notification messages in a seamless manner.
 * The reason for message exhaust is to track what messages
 * flowed through eBus. This allows the user to recreate what
 * happened when a system error occurred.
 *
 * @see EPublishFeed
 * @see IEHistoricPublisher
 * @see EHistoricSubscribeFeed
 * @see IEHistoricSubscriber
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class EHistoricPublishFeed
    extends EAbstractHistoricFeed<IEHistoricPublisher>
    implements EPublisher,
               EReplier
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Default maximum number of notifications per
     * {@link HistoricReply} is {@value}.
     */
    public static final int DEFAULT_NOTIFICATIONS_PER_REPLY = 5;

    /**
     * If historic publish feed name is not set, then defaults to
     * {@value} appended with feed index.
     */
    public static final String DEFAULT_PUBLISH_FEED_NAME =
        "EHistoricPublishFeed-";

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        LoggerFactory.getLogger(EHistoricPublishFeed.class);

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Publish status callback. If not explicitly set by client,
     * then defaults to
     * {@link IEHistoricPublisher#publishStatus(EFeedState, EHistoricPublishFeed)}.
     */
    private final HistoricPublishStatusCallback mStatusCallback;

    /**
     * Publisher identifier.
     */
    private final long mPublisherId;

    /**
     * Message store used to persist or retrieve historic
     * notifications.
     */
    private final IEMessageStore mMessageStore;

    /**
     * Post this many notifications in each reply.
     */
    private final int mNotificationsPerReply;

    /**
     * Post live notification messages to this feed.
     */
    private EPublishFeed mPublishFeed;

    /**
     * Post live {@link PublishStatusEvent} messages to this
     * feed.
     */
    private EPublishFeed mStatusFeed;

    /**
     * Historic notification message request feed.
     */
    private EReplyFeed mReplyFeed;

    /**
     * Used to set publish status event position. Initialized to
     * zero and incremented after posting a publish status event.
     */
    private int mStatusPosition;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of EHistoricPublishFeed.
     */
    private EHistoricPublishFeed(final Builder builder)
    {
        super (builder);

        mStatusCallback = builder.getStatusCallback();
        mPublisherId = (builder.mOwner).publisherId();
        mMessageStore = builder.mMessageStore;
        mNotificationsPerReply = builder.mNotificationsPerReply;

        mStatusPosition = 0;
    } // end of EHistoricPublishFeed(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Implementation.
    //

    /**
     * Closes subordinate notification and request feeds, if
     * open.
     */
    @Override
    protected void doClose()
    {
        closeFeed(mPublishFeed);
        mPublishFeed = null;

        closeFeed(mStatusFeed);
        mStatusFeed = null;

        closeFeed(mReplyFeed);
        mReplyFeed = null;
    } // end of doClose()

    //
    // end of Abstract Method Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // EObject Interface Implementation.
    //

    /**
     * Returns historic publish feed's eBus object name. Used for
     * logging purposes only.
     * @return eBus object name.
     */
    @Override
    public String name()
    {
        return (mName);
    } // end of name()

    /**
     * Opens notification and {@link PublishStatusEvent} publish
     * feeds, and historic request reply feeds <em>but</em>
     * leaves them un-advertised. These feeds are advertised when
     * {@link #advertise()} is called.
     *
     * @see #advertise()
     * @see #unadvertise()
     */
    @Override
    public void startup()
    {
        final EPublishFeed.Builder pubBuilder =
            EPublishFeed.builder();
        final EPublishFeed.Builder statBuilder =
            EPublishFeed.builder();
        final EReplyFeed.Builder repBuilder =
            EReplyFeed.builder();

        sLogger.debug("{}: starting.", mName);

        // 1. Open notification publish feed.
        mPublishFeed =
            pubBuilder.target(this)
                      .messageKey(mKey)
                      .scope(mScope)
                      .statusCallback(this::logFeedState)
                      .build();

        // 2. Open, advertise, and mark as up publisher status
        //    publish feed.
        mStatusFeed =
            statBuilder.target(this)
                       .messageKey(mStatusKey)
                       .scope(mScope)
                       .statusCallback(this::logFeedState)
                       .build();
        mStatusFeed .advertise();
        mStatusFeed.updateFeedState(EFeedState.UP);

        // 3. Open, advertise, and mark as up reply feed.
        mReplyFeed =
            repBuilder.target(this)
                      .messageKey(mRequestKey)
                      .scope(mScope)
                      .requestCallback(this::historicRequest)
                      .cancelRequestCallback(this::historicCancel)
                      .build();
        mReplyFeed.advertise();
        mReplyFeed.updateFeedState(EFeedState.UP);

        // 4. Mark this historic publish feed as open.
        mIsOpen = true;

        sLogger.info("{}: started.", mName);
    } // end of startup()

    /**
     * Closes all extand publish and reply feeds.
     */
    @Override
    public void shutdown()
    {
        sLogger.debug("{}: shutting down.", mName);

        // Close subordinate feeds if open.
        close();

        sLogger.info("{}: shut down.", mName);
    } // end of shutdown()

    //
    // end of EObject Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // EPublisher Interface Implementation.
    //

    /**
     * Logs change in subscriber feed state. This change in
     * state is <em>not</em> forwarded to owner.
     * @param feedState new feed state.
     * @param feed underlying notification publish feed.
     */
    private void logFeedState(final EFeedState feedState,
                              final IEPublishFeed feed)
    {
        sLogger.debug("{} publisher {}, feed {}: {} is {}.",
                      mPublishFeed.location(),
                      mPublishFeed.clientId(),
                      mPublishFeed.feedId(),
                      feed.key(),
                      feedState);

        forwardPublishStatus(feedState);
    } // end of logFeedState(EFeedStatus, IEPublishFeed)

    //
    // end of EPublisher Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // EReplier Interface Implementation.
    //

    /**
     * Sends a reply in response to the historic notification
     * request. Historic notifications are sent via
     * {@link HistoricReply} messages which contains one or
     * more notification messages. If there are no notifications
     * matching {@code request}, then an {@link EReplyMessage} is
     * sent back with {@link ReplyStatus#OK_FINAL} status and
     * a reason explaining there are no notifications in the
     * given time interval.
     * @param request historic notification request.
     */
    private void historicRequest(final EReplyFeed.ERequest request)
    {
        final HistoricRequest hReq =
            (HistoricRequest) request.request();
        final EInterval interval = hReq.interval;

        sLogger.debug(
            "{}: received {} historic request for interval {}.",
            mName,
            mRequestKey,
            interval);

        try
        {
            final Collection<ENotificationMessage> history =
                mMessageStore.retrieve(interval);

            // Were any historic notifications retrieve?
            if (history.isEmpty())
            {
                // No. Send an ordinary EReply explaining that
                // there were no notifications retrieved.
                postEmptyReply(hReq.interval, request);
            }
            else
            {
                // Yes. Forward historic notifications to
                // requestor.
                sendHistoricReplies(history, request);
            }
        }
        catch(Exception jex)
        {
            final String reason =
                (Strings.isNullOrEmpty(jex.getMessage()) ?
                 String.format(
                     "failed to retrieve historic messages for interval %s; no reason given",
                     interval) :
                 jex.getMessage());

            // Log this error & send back a failure reply.
            sLogger.warn(
                "{}: failed to retrieve historic notifications for interval {}.",
                mName,
                interval);

            postErrorReply(reason, request);
        }
    } // end of historicRequest(ERequest)

    /**
     * Does nothing since historic requests are processed to
     * completion when received.
     * @param request cancel this historic feed request.
     * @param mayRespond {@code true} if a response is expected.
     */
    @SuppressWarnings ({"java:S1186"})
    private void historicCancel(final EReplyFeed.ERequest request,
                                final boolean mayRespond)
    {
        sLogger.debug(
            "Attempt to cancel {} historic request (may respond {}); ignored.",
            request.messageSubject(),
            mayRespond);
    } // end of historicCancel(ERequest, boolean)

    //
    // end of EReplier Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns historic publisher identifier.
     * @return historic publisher identifier.
     */
    public long publisherId()
    {
        return (mPublisherId);
    } // end of publisherId()

    /**
     * Returns maximum number of historic notifications placed in
     * a historic message reply.
     * @return maximum number of historic messages per reply.
     */
    public int notificationsPerReply()
    {
        return (mNotificationsPerReply);
    } // end of notificationsPerReply()

    /**
     * Returns {@code true} if this historic publish feed is
     * 1) open, 2) advertised, and 3) feed state is up; otherwise
     * returns {@code true}
     * @return {@code true} if historic publisher is clear to
     * publish notification messages.
     */
    public boolean isFeedUp()
    {
        return (mInPlace && mFeedState == EFeedState.UP);
    } // end of isFeedUp()

    /**
     * Returns eBus historic message store associated with
     * publisher.
     * @return historic message store.
     */
    @VisibleForTesting
    /* package */ IEMessageStore messageStore()
    {
        return (mMessageStore);
    } // end of messageStore()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Advertises publisher feed and historic reply feed
     * associated with message key. If this historic feed is
     * currently advertised, then does nothing. Historic feed
     * publisher may publish notifications only after both
     * advertising the feed and setting the publish status to
     * {@link EFeedState#UP up}. <strong>Note:</strong> once the
     * publisher has advertised and marked this historic feed as
     * up, the publisher is free to publish notifications even if
     * there are no subscribers to this feed since all
     * notification messages are persisted.
     * @throws IllegalStateException
     * if this feed is closed.
     *
     * @see #unadvertise
     * @see #updateFeedState(EFeedState)
     * @see #shutdown()
     */
    public void advertise()
    {
        // Is this feed open?
        if (!mIsOpen)
        {
            throw (
                new IllegalStateException("feed is closed"));
        }

        // Is this feed already advertised?
        if (!mInPlace)
        {
            // No. Put the advertisements in place.
            sLogger.debug(
                "{} publisher {}, feed {}: advertised.",
                mPublishFeed.location(),
                mPublishFeed.clientId(),
                mPublishFeed.feedId());

            // Advertise both the notification publisher and
            // historic replier as well setting the reply feed
            // state as up.
            mPublishFeed.advertise();

            // Feed is now advertised.
            mInPlace = true;
        }
    } // end of advertise()

    /**
     * Retracts both the notification publish and historic reply
     * feeds. Does nothing if this historic feed is not currently
     * advertised.
     *
     * @see #advertise()
     * @see #shutdown()
     */
    public void unadvertise()
    {
        // Are the feed advertisements in place?
        if (mInPlace)
        {
            // Yes. Is the publish feed state up?
            if (mFeedState == EFeedState.UP)
            {
                // Yes. Well, it is unknown now by the fact that
                // this feed is now unadvertised.
                updateFeedState(EFeedState.UNKNOWN);
            }

            // Retract the advertisements.
            mPublishFeed.unadvertise();

            mInPlace = false;
        }
    } // end of unadvertise()

    /**
     * Updates publish feed state to the given value. If
     * {@code update} equals the current state, nothing is done.
     * Otherwise this new publish feed state is forwarded to the
     * underlying {@code EPublishFeed} and persisted to this
     * historic message store.
     * @param update new publish feed state.
     * @throws NullPointerException
     * if {@code update} is {@code null}.
     * @throws IllegalStateException
     * if this feed is not advertised.
     */
    public void updateFeedState(final EFeedState update)
    {
        Objects.requireNonNull(update, "update is null");

        // Is the advertisement in place.
        if (!mInPlace)
        {
            // No and that is a problem.
            throw (
                new IllegalStateException(FEED_NOT_ADVERTISED));
        }

        // Does this update actually change anything?
        if (update != mFeedState)
        {
            final PublishStatusEvent.Builder builder =
                PublishStatusEvent.builder();
            final PublishStatusEvent pse =
                builder.subject(mKey.subject())
                       .publisherId(mPublisherId)
                       .position(mStatusPosition)
                       .key(mKey)
                       .feedState(update)
                       .build();

            ++mStatusPosition;

            // Yes. Apply the update.
            mFeedState = update;

            sLogger.debug(
                "{} publisher {}, feed {}: setting {} feed state to {} ({}).",
                mPublishFeed.location(),
                mPublishFeed.clientId(),
                mPublishFeed.feedId(),
                mKey,
                update,
                mScope);

            // Update the notification publish feed.
            mPublishFeed.updateFeedState(update);

            // Persist this change and then publish it.
            try
            {
                mMessageStore.store(pse);
            }
            catch(Exception jex)
            {
                sLogger.warn("{}: attempt to persist {} failed:",
                             mName,
                             mStatusKey,
                             jex);
            }

            if (mStatusFeed.isFeedUp())
            {
                mStatusFeed.publish(pse);
            }
        }
        // Nope, no change to publisher feed state.
    } // end of updateFeedState(EFeedState)

    /**
     * First persists this notification to message store and
     * then, if the notification feed is up. Historic publish
     * feeds differ from {@link EPublishFeed} in that
     * notifications may be published even when the feed is
     * down.
     * <p>
     * If any exception is thrown, this means the message was
     * <em>not</em> persisted.
     * </p>
     * @param msg post this notification message to subscribers.
     * @throws NullPointerException
     * if {@code msg} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code msg} message key or publisher identifier does
     * not match the feed's values.
     * @throws IllegalStateException
     * if this feed is not advertised or the publisher has not
     * declared the feed to be up.
     */
    public void publish(final ENotificationMessage msg)
    {
        Objects.requireNonNull(msg, "msg is null");

        if (!mKey.equals(msg.key()))
        {
            throw (
                new IllegalArgumentException(
                    String.format(
                        "received msg key %s, expected %s",
                        msg.key(),
                        mKey)));
        }

        // Is the publisher identifier set correctly?
        if (msg.publisherId != mPublisherId)
        {
            // No. Notification is not acceptable.
            throw (
                new IllegalArgumentException(
                    String.format(
                        "received publisher ID %d, expected %d",
                        msg.publisherId,
                        mPublisherId)));
        }

        // Is the advertisement in place?
        if (!mInPlace)
        {
            // No. Gotta do that first.
            throw (
                new IllegalStateException(FEED_NOT_ADVERTISED));
        }

        // Is the publisher state up?
        if (mFeedState != EFeedState.UP)
        {
            // No. Gotta do that second.
            throw (
                new IllegalStateException(
                    "publish state is down"));
        }

        // Firstly, persist the message. Catch and log any
        // thrown exception and continue to the next step.
        try
        {
            mMessageStore.store(msg);
        }
        catch(Exception jex)
        {
            sLogger.warn("{}: attempt to persist {} failed:",
                         mName,
                         mKey,
                         jex);
        }

        // Secondly, publish the message *if* the notification
        // publish feed is up.
        if (mPublishFeed.isFeedUp())
        {
            mPublishFeed.publish(msg);
        }
    } // end of publish(T)

    /**
     * Returns a new historic publish feed builder for the
     * specified notification message key and publisher
     * identifier. It is recommended that a new builder be
     * obtained for each historic publish feed rather than
     * re-using an existing builder.
     * @param key published notification message key.
     * @param publisher historic publisher associated with this
     * feed.
     * @return new historic publish feed builder.
     * @throws NullPointerException
     * if {@code key} or {@code publisher} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code key} is not a notification message.
     */
    public static Builder builder(final EMessageKey key,
                                  final IEHistoricPublisher publisher)
    {
        Objects.requireNonNull(key, "message key is null");
        Objects.requireNonNull(publisher, "publisher is null");

        if (!key.isNotification())
        {
            throw (
                new IllegalArgumentException(
                    "not a notification message key"));
        }

        return (new Builder(publisher, key));
    } // end of builder(EMessageKey, IEHistoricPublisher)

    /**
     * Posts given historic notification messages back to
     * requester in {@link HistoricReply} messages with up to
     * {@link #mNotificationsPerReply} notification messages per
     * reply.
     * @param history historic notifications collection.
     * @param request historic notification request.
     */
    private void sendHistoricReplies(final Collection<ENotificationMessage> history,
                                     final EReplyFeed.ERequest request)
    {
        final Iterator<ENotificationMessage> nIt =
            history.iterator();
        final Collection<ENotificationMessage> notifications =
            new ArrayList<>(mNotificationsPerReply);
        ReplyStatus status;

        // Reached the end of the historic notifications?
        while (nIt.hasNext())
        {
            // No. Add the next notification to the collection.
            notifications.add(nIt.next());

            // Has the notification collection reached its
            // maximum limit?
            if (notifications.size() == mNotificationsPerReply)
            {
                // Yes. Post this latest collection before
                // retrieving more.
                // Firstly, is this the final reply?
                // It is possible that the number of
                // notifications is an exact multiple of the
                // maximum number of notifications per reply.
                status = (nIt.hasNext() ?
                          ReplyStatus.OK_CONTINUING :
                          ReplyStatus.OK_FINAL);

                postReply(status, notifications, request);

                // Clear out the collection before proceeding.
                notifications.clear();
            }
            // No, limit not yet reached. Continue collecting.
        }

        // Are there any left over notifications to transmit?
        if (!notifications.isEmpty())
        {
            // Yes. Send the remaining notifications now.
            postReply(
                ReplyStatus.OK_FINAL, notifications, request);
        }
    } // end of sendHistoricReplies(Collection<>, ERequest)

    /**
     * Posts historic notification reply for the given
     * notification collection to requester.
     * @param status either {@code OK_CONTINUING} or
     * {@code OK_FINAL}.
     * @param notifications send these historic notifications
     * to requester.
     * @param request reply is for this historic notification
     * request.
     */
    private void postReply(final ReplyStatus status,
                           final Collection<ENotificationMessage> notifications,
                           final EReplyFeed.ERequest request)
    {
        final HistoricReply.Builder builder =
            HistoricReply.builder();

        request.reply(builder.subject(mRequestKey.subject())
                             .timestamp(Instant.now())
                             .replyStatus(status)
                             .notifications(notifications)
                             .build());
    } // end of postReply(/ReplyStatus, Collection<>, ERequest)

    /**
     * Posts an {@code EReplyMessage} in response explaining that
     * there are no notifications for the given request.
     * @param interval requested historic notification interval.
     * @param request historic notification request.
     */
    private void postEmptyReply(final EInterval interval,
                                final EReplyFeed.ERequest request)
    {
        final String reason =
            String.format(
                "no notification messages within %s", interval);
        final EReplyMessage.Builder<?, ?> builder =
            EReplyMessage.builder();

        request.reply(builder.subject(mRequestKey.subject())
                             .timestamp(Instant.now())
                             .replyStatus(ReplyStatus.OK_FINAL)
                             .replyReason(reason)
                             .build());
    } // end of postEmptyReply(EInterval, ERequest)

    /**
     * Posts an {@code EReplyMessage} in response explaining that
     * an error occurred attempting to retrieve historic
     * notifications for the given request.
     * @param reason reason explaining why retrieval failed.
     * @param request historic notification request.
     */
    private void postErrorReply(final String reason,
                                final EReplyFeed.ERequest request)
    {
        final EReplyMessage.Builder<?, ?> builder =
            EReplyMessage.builder();

        request.reply(builder.subject(mRequestKey.subject())
                             .timestamp(Instant.now())
                             .replyStatus(ReplyStatus.ERROR)
                             .replyReason(reason)
                             .build());
    } // end of postErrorReply(String, ERequest)

    /**
     * Forwards publish status event to historic publisher.
     * Catches and logs any exception thrown by publisher status
     * callback.
     * @param feedState latest publish feed state.
     */
    private void forwardPublishStatus(final EFeedState feedState)
    {
        try
        {
            mStatusCallback.call(feedState, this);
        }
        catch(Exception jex)
        {
            sLogger.warn(
                "{}: publisher publish status callback exception:",
                mName,
                jex);
        }
    } // end of forwardPublishStatus(EFeedState)

//---------------------------------------------------------------
// Member data.
//

    /**
     * Builder is used to instantiate
     * {@code EHistoricPublishFeed} instances. A builder instance
     * is acquired via
     * {@link #builder(EMessageKey, IEHistoricPublisher)} which
     * requires that a non-{@code null} notification message key
     * and historic publisher instance be provided. This message
     * key defines the notification message class and subject
     * published and persisted by the historic publish feed.
     * The publisher is the historic publish feed owner. The
     * historic publish feed is an eBus hybrid object and runs
     * in the owner's dispatcher. This historic publisher
     * provides a unique publisher identifier (note that eBus
     * cannot determine identifier is unique),
     * <p>
     * This builder requires the following properties be
     * defined (beside those properties required by
     * {@link EAbstractHistoricFeed.Builder}):
     * </p>
     * <ul>
     *   <li>
     *     non-{@code null} {@link IEMessageStore message store},
     *     and
     *   </li>
     *   <li>
     *     maximum number of historic notification messages
     *    placed into each {@link HistoricReply} message.
     *   </li>
     * </ul>
     * <p>
     * It is recommended that each builder instance be used to
     * create only one historic publish feed.
     * </p>
     */
    public static final class Builder
        extends EAbstractHistoricFeed.Builder<IEHistoricPublisher,
                                              EHistoricPublishFeed,
                                              Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Publish status callback. If not explicitly set by
         * client, then defaults to
         * {@link IEHistoricPublisher#publishStatus(EFeedState, EHistoricPublishFeed)}.
         */
        private HistoricPublishStatusCallback mStatusCallback;

        /**
         * Message store used to persist or retrieve historic
         * notifications.
         */
        private IEMessageStore mMessageStore;

        /**
         * Post this many notifications in each reply.
         */
        private int mNotificationsPerReply;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a historic publish feed builder for the given
         * notification message key.
         * @param key notification message key.
         * @param publisher historic publisher.
         */
        private Builder(final IEHistoricPublisher publisher,
                        final EMessageKey key)
        {
            super (publisher, EHistoricPublishFeed.class, key);

            mNotificationsPerReply =
                DEFAULT_NOTIFICATIONS_PER_REPLY;
        } // end of Builder(EMessageKey, IEHistoricPublisher)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Implementation.
        //

        /**
         * Returns {@code this Builder} self reference.
         * @return {@code this Builder} self reference.
         */
        @Override
        protected Builder self()
        {
            return (this);
        } // end of self()

        /**
         * Validates historic publish feed builder settings.
         * Extends {@code EAbstractHistoricFeed} validation with
         * a check that message store is set. Returns
         * {@code problems} parameter to allow for
         * {@code validate} method chaining.
         * @param problems
         * @return @code problems} to allow {@code validate}
         * method chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            // An historic publish feed is valid if:
            // + abstract historic validation passes, and
            // + message store is set.
            return (
                super.validate(problems)
                    .requireNotNull(mMessageStore,
                                    "messageStore"));
        } // end of validate(Validator)

        /**
         * Returns default historic public feed name. Used
         * when feed name is not explicitly set.
         * @return default historic public feed name.
         */
        @Override
        protected String generateName()
        {
            return (DEFAULT_PUBLISH_FEED_NAME +
                    sFeedIndex.getAndIncrement());
        } // end of generateName()

        /**
         * Returns new historic publish feed instance created
         * from builder settings.
         * @return new historic publish feed instance.
         */
        @Override
        protected EHistoricPublishFeed buildImpl()
        {
            return (new EHistoricPublishFeed(this));
        } // end of buildImpl()

        //
        // end of Abstract Method Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns historic publisher status callback method.
         * @return historic publisher status callback method.
         */
        private HistoricPublishStatusCallback getStatusCallback()
        {
            // Was publish status callback explicitly defined?
            if (mStatusCallback == null)
            {
                // No. Create a callback to
                // IEHistoricPublisher.publishStaus() method.
                mStatusCallback = mOwner::publishStatus;
            }

            return (mStatusCallback);
        } // end of getStatusCallback()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Puts historic publish status callback in place. If
         * {@code cb} is not {@code null}, publish status updates
         * will be passed to {@code cb} rather than
         * {@link IEHistoricPublisher#publishStatus(EFeedState, EHistoricPublishFeed)}.
         * The reverse is true: if {@code cb} is {@code null},
         * then updates are post to
         * {@code IEHistoricPublisher#publishStatus} override.
         * <p>
         * An example using this method is:
         * </p>
         * <pre><code>statusCallback(
    (fs, f) &rarr; {
        if (fs == EFeedState.DOWN) {
            <strong style="color:ForestGreen">// Respond to down live notification feed.</strong>
        }
    })</code></pre>
         * @param cb historic publisher publish status update
         * callback.
         * @return {@code this Builder} instance.
         */
        public Builder statusCallback(@Nullable final HistoricPublishStatusCallback cb)
        {
            mStatusCallback = cb;

            return (this);
        } // end of statusCallback(HistoricPublishStatusCallback)

        /**
         * Sets the message store used to persist and retrieve
         * notification messages.
         * @param store notification message store.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code store} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code store} is not open.
         */
        public Builder messageStore(final IEMessageStore store)
        {
            Objects.requireNonNull(
                    store, "message store is null");

            if (!store.isOpen())
            {
                throw (
                    new IllegalArgumentException(
                        "message store is closed"));
            }

            mMessageStore = store;

            return (this);
        } // end of messageStore(IEMessageStore)

        /**
         * Sets maximum number of notifications per
         * {@code HistoricReply} instance. Values must be &gt;
         * zero.
         * @param n maximum number of notifications per historic
         * reply.
         * @return {@code this Builder} instance.
         */
        public Builder notificationsPerReply(final int n)
        {
            if (n <= 0)
            {
                throw (
                    new IllegalArgumentException(
                        "notificationsPerReply <= zero"));
            }

            mNotificationsPerReply = n;

            return (this);
        } // end of notificationsPerReply(int)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class EHistoricPublishFeed
