//
// Copyright 2023 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic.store;

import com.google.common.collect.ImmutableList;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Objects;
import java.util.SortedMap;
import java.util.TreeMap;
import javax.annotation.Nullable;
import net.sf.eBus.feed.historic.IEMessageStore;
import net.sf.eBus.feed.historic.PublishStatusEvent;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.util.ValidationException;
import net.sf.eBus.util.Validator;
import net.sf.eBusx.time.EInterval;
import net.sf.eBusx.time.EInterval.Clusivity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class implements {@link IEMessageStore} providing the
 * ability to store events to and retrieve events from an
 * in-memory
 * {@link ENotificationMessage eBus notification message array}.
 * This array is fixed length and implemented as a ring buffer.
 * When the in-memory message store reaches its maximum allowed
 * size, the oldest message is overwritten with the new message.
 * <p>
 * In-memory message store supports message retrieval based on an
 * {@link EInterval} (as required by {@code IEMessageStore}) and
 * the last <em>n</em> messages stored. The later is <em>not</em>
 * supported by eBus historic feed but provided for application
 * use.
 * </p>
 * <p>
 * Please not that this message store does <em>not</em> persist
 * messages. When this store is closed, then stored messages are
 * discarded.
 * </p>
 * <p>
 * Please note that this message store is homogenous - meaning
 * that all notification messages have the same key (excepting
 * when {@link PublishStatusEvent}s are stored). This store is
 * designed to work within an
 * {@link net.sf.eBus.feed.historic.EHistoricPublishFeed} which
 * guarantees that notification messages passed to
 * {@link #store(ENotificationMessage)} match the store's this
 * store's message key. Therefore {@code store} does not validate
 * the given notification message's key to see if it matches
 * the configured key. In short, this in-memory message store
 * is not designed for independent use but as part of eBus
 * historic message feed subsystem.
 * </p>
 *
 * @see SqlMessageStore
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class InMemoryMessageStore
    implements IEMessageStore
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Message store default initial capacity is {@value}.
     */
    public static final int DEFAULT_STORE_CAPACITY = 1_000;

    /**
     * First message for a given timestamp is stored in array
     * index {@value}.
     */
    private static final int FIRST_INDEX = 0;

    /**
     * Last message for a given timestamp is stored in array
     * index {@value}.
     */
    private static final int LAST_INDEX = 1;

    /**
     * Timestamp value is not set.
     */
    private static final String NOT_SET = "(not set)";

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        LoggerFactory.getLogger(InMemoryMessageStore.class);

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * All application eBus notifications stored have this
     * message key.
     */
    private final EMessageKey mKey;

    /**
     * At most this many messages are stored. Once this limit is
     * reached, as a new message is appended, the oldest message
     * is removed. Defaults to {@link #DEFAULT_STORE_CAPACITY}.
     */
    private final int mMaximumCapacity;

    /**
     * Appends text and publish status messages to this store.
     */
    private final ENotificationMessage[] mStore;

    /**
     * Index into message store mapping message timestamp to
     * store index.
     */
    private final SortedMap<Instant, int[]> mTimeIndex;

    /**
     * Tracks the number of messages in {@link #mStore}. When
     * this count reaches {@link #mMaximumCapacity}, then the
     * oldest message is overwritten and removed from
     * {@link #mTimeIndex}.
     */
    private int mMessageCount;

    /**
     * Insert next message into this index.
     */
    private int mInsertIndex;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a in-memory message store instance based on the
     * builder settings.
     * @param builder contains message store configuration.
     */
    private InMemoryMessageStore(final Builder builder)
    {
        mKey = builder.mKey;
        mMaximumCapacity = builder.mMaxCapacity;
        mStore = new ENotificationMessage[mMaximumCapacity];
        mTimeIndex = new TreeMap<>();

        mMessageCount = 0;
        mInsertIndex = 0;
    } // end of InMemoryMessageStore(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // IEMessageStore Interfaace Implementation.
    //

    /**
     * Returns {@code true} since this in-memory message store
     * is always open.
     * @return {@code open}.
     */
    @Override
    public boolean isOpen()
    {
        return (true);
    } // end of isOpen()

    @Override
    public EMessageKey key()
    {
        return (mKey);
    } // end of key()

    /**
     * Inserts given message into in-memory message store. If
     * message store has reached configured capacity, then oldest
     * store message is replaced with {@code message}.
     * <p>
     * <strong>Note:</strong> this method does <em>not</em>
     * validate if {@code messsage}'s key matches the store's
     * configured message key. This is so that
     * {@link PublishStatusEvent} messages may be stored. This
     * means this message store may be corrupted with invalid
     * messages.
     * {@link net.sf.eBus.feed.historic.EHistoricPublishFeed}
     * guarantees this will not happen <em>if</em> it has sole
     * access to this message store.
     * </p>
     * @param message store this message into in-memory message
     * store.
     *
     * @see #retrieve(EInterval)
     * @see #retrieve(int)
     */
    @Override
    public void store(final ENotificationMessage message)
    {
        final Instant ts =
            Instant.ofEpochMilli(message.timestamp);
        final int[] indices;

        // Has the message store reached its limit?
        if (mMessageCount == mMaximumCapacity)
        {
            // Yes. Remove oldest message from message store and
            // update the time index map accordingly.
            removeOldestMessage(mInsertIndex);

            // Substract one from the message count now that the
            // oldest message is removed. This count will be
            // incremented again when the new message is
            // inserted.
            --mMessageCount;
        }
        // No. But either way, there is now room to enter the
        // next message.

        mStore[mInsertIndex] = message;
        ++mMessageCount;
        indices =
            mTimeIndex.computeIfAbsent(
                ts, k -> new int[] {mInsertIndex, mInsertIndex});
        indices[LAST_INDEX] = mInsertIndex;

        mInsertIndex = incrementIndex(mInsertIndex);
    } // end of store(ENotification message.

    @Override
    @SuppressWarnings({"java:S3776"})
    public Collection<ENotificationMessage> retrieve(final EInterval interval)
    {
        final Instant beginTime;
        final Instant endTime;
        final SortedMap<Instant, int[]> subMap;
        final Instant firstKey;
        final Instant lastKey;
        final ImmutableList.Builder<ENotificationMessage> builder =
            ImmutableList.builder();

        // Because SortedMap.subMap(to, from) uses inclusive from
        // and exclusive to, need to modify the begin and end
        // times to match the interval begin and end clusivities.
        beginTime =
            (interval.beginClusivity == EInterval.Clusivity.INCLUSIVE ?
             interval.beginTime :
             (interval.beginTime).plusNanos(1L));
        endTime =
            (interval.endClusivity == EInterval.Clusivity.EXCLUSIVE ?
             interval.endTime :
             (interval.endTime).plusNanos(1L));

        subMap = mTimeIndex.subMap(beginTime, endTime);

        if (subMap != null &&
            !subMap.isEmpty() &&
            (firstKey = subMap.firstKey()) != null &&
            (lastKey = subMap.lastKey()) != null)
        {
            final int[] beginIndices = subMap.get(firstKey);
            final int[] endIndices = subMap.get(lastKey);

            // Are either timestamp indices missing?
            if (beginIndices == null || endIndices == null)
            {
                // YES and that this WRONG!
                sLogger.warn(
                    "timestamp indices missing for {} and {}",
                    firstKey,
                    lastKey);
            }
            else
            {
                // Because message store is a ring, cannot test
                // if begin index < end index. We know we have
                // reached the end when loop index equals end
                // index. That means end index must be one
                // greater than the last index for inclusive and
                // the first index for exclusive.
                final int beginIndex =
                    (interval.beginClusivity == Clusivity.INCLUSIVE ?
                     beginIndices[FIRST_INDEX] :
                     incrementIndex(beginIndices[LAST_INDEX]));
                final int endIndex =
                    (interval.endClusivity == Clusivity.INCLUSIVE ?
                     incrementIndex(endIndices[LAST_INDEX]) :
                     endIndices[FIRST_INDEX]);
                int index;
                boolean firstIteration;

                // When the in-memory message store is full and
                // end clusivity is inclusive, then begin and
                // end indices will be equal. Since this is the
                // loop complete test, need to distinguish
                // between begin == end index at first and last
                // iteration.
                for (index = beginIndex, firstIteration = true;
                     index != endIndex || firstIteration;
                     index = incrementIndex(index),
                         firstIteration = false)
                {
                    builder.add(mStore[index]);
                }
            }
        }

        return (builder.build());
    } // end of retrieve(EInterval)

    //
    // end of IEMessageStore Interface Impelementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns text containing in-memory message store's current
     * size, maximum allowed size, first and last message
     * timestamps.
     * @return in-memory message store stats as text.
     */
    @Override
    public String toString()
    {
        final Instant startTime = firstTimestamp();
        final Instant endTime = lastTimestamp();

        return (
            String.format(
                "[size=%,d, max size=%,d, start time=%s, end time=%s]",
                mMessageCount,
                mMaximumCapacity,
                (startTime == null ? NOT_SET : startTime),
                (endTime == null ? NOT_SET : endTime)));
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns current in-memory message store size. Returned
     * value is &ge; zero.
     * @return current in-memory message store size.
     */
    public int size()
    {
        return (mMessageCount);
    } // end of size()

    /**
     * Returns in-memory message store maximum allowed capacity.
     * @return in-memory message store maximum allowed capacity.
     */
    public int capacity()
    {
        return (mMaximumCapacity);
    } // end of capacity()

    /**
     * Returns currently first notification message timestamp as
     * an {@code Instant}. If there are no notification messages
     * stored, then returns {@code null}.
     * @return first notification message's timestamp.
     */
    public @Nullable Instant firstTimestamp()
    {
        return (mTimeIndex.firstKey());
    } // end of firstTimestamp()

    /**
     * Returns currently last notification message timestamp as
     * an {@code Instant}. If there are no notification messages
     * stored, then returns {@code null}.
     * @return last notification message's timestamp.
     */
    public @Nullable Instant lastTimestamp()
    {
        return (mTimeIndex.lastKey());
    } // end of lastTimestamp()

    /**
     * Returns up to {@code numMessages} last messages placed
     * into message store. If there are fewer than
     * {@code numMessages} stored, then returns all returned
     * messages. Returned list may be empty but will not be
     * {@code null}.
     * <p>
     * Unlike {@link #retrieve(EInterval)}, this method validates
     * that {@code numMessages} is &gt; zero.
     * </p>
     * @param numMessages Return up to this many store
     * notification messages.
     * @return eBus notification list.
     * @throws IllegalArgumentException
     * if {@code numMessages} &le; zero.
     *
     * @see #retrieve(EInterval)
     */
    public Collection<ENotificationMessage> retrieve(final int numMessages)
    {
        if (numMessages <= 0)
        {
            throw (
                new IllegalArgumentException(
                    "numMessages <= 0"));
        }

        // Return only as many as are available.
        final int n = (mMessageCount < numMessages ?
                       mMessageCount :
                       numMessages);
        int beginIndex = 0;
        int index;
        boolean firstIteration;
        final ImmutableList.Builder<ENotificationMessage> builder =
            ImmutableList.builder();

        // Is the list full?
        if (mMessageCount == mMaximumCapacity)
        {
            // Yes. Calculate the first index.
            beginIndex = (mInsertIndex - (n + 1));

            // Wrapped around the array start?
            if (beginIndex < 0)
            {
                // Yes.
                beginIndex = (mMaximumCapacity + beginIndex);
            }
        }

        for (index = beginIndex, firstIteration = true;
             index != mInsertIndex || firstIteration;
             index = incrementIndex(index),
                 firstIteration = false)
        {
            builder.add(mStore[index]);
        }

        return (builder.build());
    } // end of retrieve(int)

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Resets in-memory message store to its initial state with
     * the notification message store array filled with
     * {@code null} values.
     * <p>
     * <strong>Note:</strong> this method should be called from
     * within an eBus dispatcher thread to prevent this message
     * store from being accessed at the same time as message
     * store or retrieval.
     * </p>
     */
    public void clear()
    {
        mMessageCount = 0;
        mInsertIndex = 0;
        Arrays.fill(mStore, null);
        mTimeIndex.clear();
    } // end of clear()

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    /**
     * Returns a new in-memory message store builder instance.
     * @return new in-memory message store {@link Builder}
     * instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

    /**
     * Updates timestamp index map so that it no longer
     * references the oldest message in the message store.
     * @param index index to oldest message in store.
     */
    private void removeOldestMessage(final int index)
    {
        final ENotificationMessage msg = mStore[index];
        final Instant ts = Instant.ofEpochMilli(msg.timestamp);
        final int[] indices = mTimeIndex.get(ts);

        // Is this a known timestamp?
        // Or is this index missing from the entry?
        if (indices == null ||
            indices[FIRST_INDEX] != index)
        {
            // NO! This is a problem which should never happen.
            // If it does it means there is a bug in this message
            // store implementation.
            throw (
                new IllegalStateException(
                    String.format(
                        "time index missing for %s", ts)));
        }
        // Is this the only message for this timestamp?
        // It is if the end index points to this message's index.
        else if (indices[LAST_INDEX] == index)
        {
            // Yes, so remove this timestamp entry entirely from
            // map.
            mTimeIndex.remove(ts);
        }
        // No, this is not the final message for the timestamp.
        // Update the first index.
        else
        {
            indices[FIRST_INDEX] = incrementIndex(index);
        }
    } // end of removeOldestMessage(ENotificationMessage)

    /**
     * Returns incremented index value which cycles back to zero
     * when index reaches message store array end.
     * @param index increment this index.
     * @return incremented index.
     */
    private int incrementIndex(final int index)
    {
        return ((index + 1) % mMaximumCapacity);
    } // end of incrementIndex(int)

//---------------------------------------------------------------
// Inner Classes.
//

    /**
     * {@link InMemoryMessageStore} instances are created using
     * a {@code Builder}. A {@code Builder} instance is acquired
     * from {@link InMemoryMessageStore#builder()}. The following
     * attributes must be set to successfully build an in-memory
     * message store:
     * <ul>
     *   <li>
     *     non-{@code null} eBus notification
     *     {@link EMessageKey message key} and
     *   </li>
     *   <li>
     *     maximum store capacity &gt; zero.
     *   </li>
     * </ul>
     * <p>
     * Once these values are correctly set, the target
     * {@link InMemoryMessageStore} instance is created by
     * calling
     * {@link Builder#build()}.
     * </p>
     */
    public static final class Builder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private EMessageKey mKey;
        private int mMaxCapacity;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            mMaxCapacity = DEFAULT_STORE_CAPACITY;
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets eBus notification message key. All message stored
         * or retrieved from database must have this message key.
         * @param key eBus notification message key.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code key} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code key} is not an eBus
         * {@link ENotificationMessage notification message}.
         */
        public Builder key(final EMessageKey key)
        {
            Objects.requireNonNull(key, "key is null");

            if (!key.isNotification())
            {
                throw (
                    new IllegalArgumentException(
                        key.className() +
                        " is not a notification message class"));
            }

            mKey = key;

            return (this);
        } // end of key(EMessageKey)

        /**
         * Sets in-memory message store maximum capacity. Once
         * this capacity is reached, the oldest message in the
         * store is replaced with the latest message.
         * @param n message store capacity.
         * @return {@code this Builder} instance.
         * @throws IllegalArgumentException
         * if {@code n} is &le; zero.
         */
        public Builder maximumCapacity(final int n)
        {
            if (n <= 0)
            {
                throw (
                    new IllegalArgumentException(
                        "maximum capacity <= zero"));
            }

            mMaxCapacity = n;

            return (this);
        } // end of maximumCapacity(final int n)

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Returns new {@link InMemoryMessageStore} based on this
         * builder's settings.
         * @return new in-memory message store instance.
         * @throws ValidationException
         * if this builder's settings are incomplete or invalid.
         */
        public InMemoryMessageStore build()
        {
            final Validator problems = new Validator();

            problems.requireNotNull(mKey, "key")
                    .throwException(ValidationException.class);

            return (new InMemoryMessageStore(this));
        } // end of build()
    } // end of class Builder
} // end of class InMemoryMessageStore
