//
// Copyright 2018 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.pattern;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Deque;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TreeSet;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import net.sf.eBus.client.ECondition;
import net.sf.eBus.client.EFeed;
import net.sf.eBus.client.EFeed.FeedScope;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;

/**
 * <p>
 * {@code EventPattern} provides the two pieces of information
 * needed to do pattern matching: 1) the
 * {@link net.sf.eBus.client.ESubscribeFeed notification feeds}
 * providing the events used in the match and 2) the pattern
 * itself. Patterns come in two types:
 * </p>
 * <ul>
 *   <li>
 *     <strong>Ordered:</strong> events must arrive in the event
 *     pattern order to effect a match.
 *   </li>
 *   <li>
 *     <strong>Unordered:</strong> specifies the number of each
 *     event type which must be collected to effect a match where
 *     the events may arrive in any order.
 *   </li>
 * </ul>
 * <p>
 * Once an event series is received matching the pattern, those
 * events are tested against the
 * {@link #patternCondition() pattern condition}. Only if that
 * condition is satisfied is a {@link MatchEvent} generated and
 * forwarded to the subscriber.
 * </p>
 * <p>
 * Event patterns are constructed via {@link EventPattern.Builder}.
 * See that class for detailed instructions on building event
 * patterns.
 * </p>
 *
 * @see EventPattern.Builder
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class EventPattern
{
//---------------------------------------------------------------
// Enums.
//

    /**
     * eBus patterns come in two varieties: ordered and
     * unordered. Ordered patterns specify the order in which
     * events are expected and the minimum and maximum quantity
     * of each event. Unordered patterns only specify the event
     * quantities, allowing events to arrive in any order.
     */
    public enum PatternType
    {
        /**
         * Ordered patterns specify event arrival order and event
         * minimum and maximum quantities.
         */
        ORDERED,

        /**
         * Unordered patterns specify event minimum and maximum
         * quantities only, allowing events to arrive in any
         * order.
         */
        UNORDERED
    } // end of enum PatternType

    /**
     * Delineates the builder states and which operations may be
     * performed for each state.
     */
    private enum PatternState
    {
        /**
         * Initial state needed to define parameters map
         * <em>if</em> not passed in to builder.
         */
        NO_PARAMS (BEGIN_PARAM_MAP_MASK),

        /**
         * Parameter map definition state. Waiting for the next
         * parameter definition or the end of the parameter map
         * definition.
         */
        IN_PARAMS (BEGIN_PARAMETER_MASK |
                   END_PARAM_MAP_MASK),

        IN_PARAMETER(MESSAGE_KEY_MASK |
                     SCOPE_MASK |
                     CONDITION_MASK |
                     END_PARAMETER_MASK),

        /**
         * Top level ordered pattern state. Ready to accept the
         * next single or multiple component or end the pattern.
         */
        IN_ORDERED_PATTERN (BEGIN_SINGLE_MASK |
                            BEGIN_MULTI_MASK |
                            BEGIN_GROUP_MASK |
                            END_GROUP_MASK |
                            UNTIL_MASK |
                            EXCLUSIVE_MASK |
                            PATTERN_COND_MASK |
                            BUILD_MASK),

        /**
         * Top level unordered pattern state. Accepts only single
         * components.
         */
        IN_UNORDERED_PATTERN (BEGIN_SINGLE_MASK |
                              UNTIL_MASK |
                              EXCLUSIVE_MASK |
                              PATTERN_COND_MASK |
                              BUILD_MASK),

        /**
         * Creating a single component. The component condition,
         * minimum, and maximum match counts may be set.
         */
        IN_SINGLE (PREDICATE_MASK |
                   MATCH_COUNT_MASK |
                   END_SINGLE_MASK),

        /**
         * Creating a multi-component. The internal single
         * components may be created as well as the minimum and
         * maximum match counts.
         */
        IN_MULTI (MATCH_COUNT_MASK |
                  ADD_MULTI_SINGLE_MASK |
                  END_MULTI_MASK),

        /**
         * Event pattern is complete. No new components may be
         * entered.
         */
        OUT_PATTERN (0x0);

    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Bit mask specifying which fields may be set in the
         * state.
         */
        private final int mFieldMask;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private PatternState(final int mask)
        {
            mFieldMask = mask;
        } // end of PatternState(int)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the state field mask.
         * @return state field mask.
         */
        public int fieldMask()
        {
            return (mFieldMask);
        } // end of fieldMask()

        /**
         * Returns {@code true} if this state supports the given
         * method mask.
         * @param methodMask method bit mask.
         * @return {@code true} if method is valid.
         */
        public boolean supportsMethod(final int methodMask)
        {
            return ((mFieldMask & methodMask) != 0);
        } // end of supportsMethod(int)

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of PatternState

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * The group {@value} contains all matched events in
     * received order. This means the order may not be
     * chronological according to the event timestamp.
     */
    public static final String ALL_EVENTS = "__ALL_EVENTS__";

    /**
     * The default until predicate always returns {@code true}.
     */
    public static final BiPredicate<List<ENotificationMessage>, ENotificationMessage> DEFAULT_UNTIL =
        (t, u) -> true;

    /**
     * The default minimum and maximum match count is {@value}.
     */
    public static final int DEFAULT_MATCH_COUNT = 1;

    /**
     * Default event match predicate returns {@code true}.
     */
    public static final MatchCondition DEFAULT_PREDICATE =
        (e, g, u) -> true;

    /**
     * Default match event which always returns {@code true}.
     * This condition should be used when subscriber wants all
     * match events returned.
     */
    public static final Predicate<MatchEvent> DEFAULT_CONDITION =
        t -> true;

    /**
     * Used to create a groups name array from a list.
     */
    private static final String[] EMPTY_NAMES = new String[0];

    // State transition field masks.
    private static final int BEGIN_SINGLE_MASK     = 0x00000001;
    private static final int END_SINGLE_MASK       = 0x00000002;
    private static final int BEGIN_MULTI_MASK      = 0x00000004;
    private static final int END_MULTI_MASK        = 0x00000008;
    private static final int ADD_MULTI_SINGLE_MASK = 0x00000010;
    private static final int UNTIL_MASK            = 0x00000020;
    private static final int MATCH_COUNT_MASK      = 0x00000040;
    private static final int CONDITION_MASK        = 0x00000080;
    private static final int BEGIN_GROUP_MASK      = 0x00000100;
    private static final int END_GROUP_MASK        = 0x00000200;
    private static final int BEGIN_PARAM_MAP_MASK  = 0x00000400;
    private static final int END_PARAM_MAP_MASK    = 0x00000800;
    private static final int BEGIN_PARAMETER_MASK  = 0x00001000;
    private static final int END_PARAMETER_MASK    = 0x00002000;
    private static final int MESSAGE_KEY_MASK      = 0x00004000;
    private static final int SCOPE_MASK            = 0x00008000;
    private static final int EXCLUSIVE_MASK        = 0x00010000;
    private static final int PREDICATE_MASK        = 0x00020000;
    private static final int PATTERN_COND_MASK     = 0x00040000;
    private static final int BUILD_MASK            = 0x00080000;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Pattern name used as the {@link MatchEvent} subject.
     */
    protected final String mPatternName;

    /**
     * This pattern is either ordered or unordered.
     */
    private final PatternType mPatternType;

    /**
     * Maps a unique name to its feed information. The keys
     * act as names used in the event pattern.
     */
    private final Map<String, FeedInfo> mParameters;

    /**
     * Components making up the event pattern. If this is an
     * ordered pattern, then list order is significant.
     */
    private final List<PatternComponent> mComponents;

    /**
     * Collected events must satisfy the until predicate.
     */
    private final BiPredicate<List<ENotificationMessage>, ENotificationMessage> mUntil;

    /**
     * Set to {@code true} when events are exclusively matched
     * to an event pattern. This means that if an event could
     * be used in two separate matches
     * <em>of the same pattern</em>, it will be used for only the
     * the first completed match. The second match will be
     * dropped as a result.
     * <p>
     * Exclusive is only within a single pattern. Exclusion does
     * not work between two separate pattern instances.
     * </p>
     */
    private final boolean mExclusive;

    /**
     * Condition applied to matched event.
     */
    private final Predicate<MatchEvent> mPatternCondition;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of EventPattern.
     */
    private EventPattern(final Builder builder)
    {
        mPatternName = builder.patternName();
        mPatternType = builder.patternType();
        mParameters =
            Collections.unmodifiableMap(builder.parameters());
        mComponents =
            Collections.unmodifiableList(builder.components());
        mUntil = builder.until();
        mExclusive = builder.exclusive();
        mPatternCondition = builder.patternCondition();
    } // end of EventPattern(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the user-assigned pattern name.
     * @return pattern name.
     */
    public String patternName()
    {
        return (mPatternName);
    } // end of patternName()

    /**
     * Returns the pattern type.
     * @return pattern type.
     */
    public PatternType patternType()
    {
        return (mPatternType);
    } // end of patternType()

    /**
     * Returns the read-only event pattern parameters map.
     * @return event pattern parameters map.
     */
    public Map<String, FeedInfo> parameters()
    {
        return (mParameters);
    } // end of parameters()

    /**
     * Returns {@code true} if the events matching this pattern
     * are used in one match only. This means that if an event
     * could be used in two separate matches
     * <em>of the same pattern</em>, it will be used for only the
     * the first completed match. The second match will be
     * dropped as a result.
     * <p>
     * Exclusive is only within a single pattern. Exclusion does
     * not work between two separate pattern instances.
     * </p>
     * @return {@code true} if events are used in a single match
     * only.
     */
    public boolean isExclusive()
    {
        return (mExclusive);
    } // end of isExclusive()

    /**
     * Returns {@link FeedInfo} for the given feed name.
     * @param feedName a known feed name from parameters map.
     * @return feed information.
     * @throws IllegalArgumentException
     * if {@code parameter} is not a known feed name.
     */
    public FeedInfo feedInfo(final String feedName)
    {
        if (!mParameters.containsKey(feedName))
        {
            throw (
                new IllegalArgumentException(
                    "\"" + feedName + "\" is an unknown feed name"));
        }

        return (mParameters.get(feedName));
    } // end of feedInfo(String)

    /**
     * Returns the message key associated with the given feed
     * name.
     * @param feedName feed name in parameters map.
     * @return message key.
     * @throws IllegalArgumentException
     * if {@code parameter} is not a known feed name.
     */
    public EMessageKey messageKey(final String feedName)
    {
        return (feedInfo(feedName).messageKey());
    } // end of messageKey(String)

    /**
     * Returns a named, ordered or unordered event pattern
     * builder which has no pre-defined parameters. These
     * parameters must be defined using
     * {@link Builder#beginParameterMap()}/
     * {@link Builder#endParameterMap()} before adding
     * components to the event pattern.
     * @param pName user-assigned pattern name. Used for logging
     * purposes only.
     * @param pType specifies whether the pattern is ordered or
     * unordered.
     * @return event pattern builder.
     * @throws NullPointerException
     * if {@code pName} or {@code pType} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code pName} is an empty string.
     *
     * @see #builder(String, PatternType, Map)
     */
    public static Builder builder(final String pName,
                                  final PatternType pType)
    {
        if (Objects.requireNonNull(
                pName, "pName is null").isEmpty())
        {
            throw (
                new IllegalArgumentException(
                    "pName is empty"));
        }

        Objects.requireNonNull(pType, "pType is null");

        return (new Builder(pName, pType, new HashMap<>(), false));
    } // end of builder(String, PatternType)

    /**
     * Returns a named, ordered or unordered event pattern
     * builder using the given parameters. {@code parameters}
     * maps a unique name to a notification feed. The map keys
     * are used in the event pattern regular expression to
     * reference a particular notification feed.
     * <p>
     * This map may <em>not</em> be {@code null}, empty, or
     * contain {@code null} values.
     * </p>
     * @param pName user-assigned pattern name. Used as the
     * subject for posted {@link MatchEvent match events}.
     * purposes only. May <em>not</em> be {@code null} or
     * empty.
     * @param pType specifies whether the pattern is ordered or
     * unordered.
     * @param parameters event pattern parameters.
     * @return event pattern builder.
     * @throws NullPointerException
     * if {@code pName}, {@code pType}, or {@code parameters} is
     * {@code null}.
     * @throws IllegalArgumentException
     * if {@code pName} is an empty string or {@code parameters}
     * is either empty or contains a {@code null} value.
     *
     * @see #builder(String, PatternType)
     */
    public static Builder builder(final String pName,
                                  final PatternType pType,
                                  final Map<String, FeedInfo> parameters)
    {
        if (Objects.requireNonNull(
                pName, "pName is null").isEmpty())
        {
            throw (
                new IllegalArgumentException(
                    "pName is empty"));
        }

        Objects.requireNonNull(pType, "pType is null");

        validateParameters(
            Objects.requireNonNull(
                parameters, "parameters is null"));

        return (new Builder(pName, pType, parameters, true));
    } // end of builder(String, PatternType, Map)

    /**
     * Returns the maximum number of events that can be collected
     * by this pattern.
     * @return maximum event count for all components.
     */
    /* package */ int maximumEvents()
    {
        int retval = 0;

        retval = mComponents.stream()
                            .map(pc -> pc.maximumMatchCount())
                            .reduce(retval, Integer::sum);

        return (retval);
    } // end of maximumEvents()

    /**
     * Returns the number of components in this pattern.
     * @return pattern component count.
     */
    /* package */ int componentCount()
    {
        return (mComponents.size());
    } // end of componentCount()

    /**
     * Returns a read-only list containing the event pattern
     * components.
     * @return event pattern components.
     */
    /* package */ List<PatternComponent> components()
    {
        return (mComponents);
    } // end of components()

    /**
     * Returns the condition associated with the entire
     * pattern. Only if the match event satisfies this
     * condition is it forwarded to the subscriber.
     * @return match event condition.
     */
    /* package */ Predicate<MatchEvent> patternCondition()
    {
        return (mPatternCondition);
    } // end of patternCondition()

    /**
     * Returns the event pattern capture until predicate.
     * @return until predicate.
     */
    /* package */ BiPredicate<List<ENotificationMessage>, ENotificationMessage> until()
    {
        return (mUntil);
    } // end of until()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Throws an exception if {@code parameters} is either
     * {@code null}, empty, or contains a {@code null} value.
     * Otherwise returns. This method is called for effect only.
     * @param parameters validate these event pattern parameters.
     * @throws NullPointerException
     * if {@code parameters} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code parameters} is either empty or contains a
     * {@code null} value.
     */
    private static void
        validateParameters(final Map<String, FeedInfo> parameters)
    {
        if (parameters.isEmpty())
        {
            throw (
                new IllegalArgumentException(
                    "parameters is empty"));
        }

        parameters.entrySet()
                  .stream()
                  .filter(entry -> (entry.getValue() == null))
                  .forEachOrdered(
                      entry ->
                      {
                          throw (
                              new IllegalArgumentException(
                                  "parameters key " +
                                  entry.getKey() +
                                  " maps to a null value"));
                      });
    } // end of validateParameters(Map)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * This class provides the information needed by
     * {@link EPatternFeed} to
     * {@link net.sf.eBus.client.ESubscribeFeed.Builder build}
     * to a notification message feed. {@code FeedInfo} instances
     * may be created directly by the user or indirectly via
     * {@link Builder#beginParameter(String)}/
     * {@link Builder#endParameter()} calls.
     */
    public static final class FeedInfo
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Unique identifier. Used to index into state transition
         * table. Set when the subscription feed is opened.
         */
        private int mTransitionId;

        /**
         * Notification message key.
         */
        private final EMessageKey mKey;

        /**
         * Feed scope.
         */
        private final FeedScope mScope;

        /**
         * Optional subscription feed condition.
         */
        private final ECondition mCondition;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates subscription feed information for the given
         * notification message key and feed scope. The feed
         * condition is set to {@link EFeed#NOTIFY_METHOD}.
         * @param key required notification message key.
         * @param scope required feed scope.
         */
        public FeedInfo(final EMessageKey key,
                        final FeedScope scope)
        {
            this (key, scope, EFeed.NO_CONDITION);
        } // end of FeedInfo(EMessageKey, FeedScope)

        /**
         * Creates subscription feed information for the given
         * notification message key, feed scope, and optional
         * feed condition. If {@code condition} is {@code null},
         * then the condition is set to
         * {@link EFeed#NO_CONDITION}.
         * @param key required notification message key.
         * @param scope required feed scope.
         * @param condition optional subscription condition. May
         * be {@code null}.
         * @throws NullPointerException
         * if either {@code key} or {@code scope} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code key} is not a notification message.
         */
        public FeedInfo(final EMessageKey key,
                        final FeedScope scope,
                        final ECondition condition)
        {
            mKey =
                Objects.requireNonNull(key, "null message key");
            mScope =
                Objects.requireNonNull(scope, "null feed scope");
            mCondition = condition;

            if (!key.isNotification())
            {
                throw (
                    new IllegalArgumentException(
                        key + " not a notification message key"));
            }
        } // end of FeedInfo(int,EMessageKey,FeedScope,ECondition)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the subscription notification message key.
         * @return message key.
         */
        public EMessageKey messageKey()
        {
            return (mKey);
        } // end of messageKey()

        /**
         * Returns the subscription feed scope.
         * @return feed scope.
         */
        public FeedScope scope()
        {
            return (mScope);
        } // end of scope()

        /**
         * Returns the optional subscription feed condition. May
         * return {@code null}.
         * @return feed condition.
         */
        public ECondition condition()
        {
            return (mCondition);
        } // end of condition()

        /**
         * Returns the unique transition identifier.
         * @return transition identifier.
         */
        /* package */ int transitionId()
        {
            return (mTransitionId);
        } // end of transitionId()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the transition identifier.
         * @param id transition identifier.
         * @return {@code this FeedInfo} instance
         */
        /* package */ FeedInfo transitionId(final int id)
        {
            mTransitionId = id;

            return (this);
        } // end of transitionId(int)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class FeedInfo

    /**
     * {@code Builder} is the only way to create an
     * {@code EventFeed} instance. Event patterns are built in
     * two stages: parameters and pattern. Since the pattern
     * references the parameters, parameters must be defined
     * first. This can be done by either providing a parameters
     * map to the builder or defining parameters in the builder.
     * <h1>Defining Event Parameters</h1>
     * <h2>Building Parameters</h2>
     * <p>
     * The following annotated code example demonstrates how
     * event parameters are defined using a an
     * {@code EventPattern.Builder}:
     * </p>
     * <pre><code><strong style="color:ForestGreen"> // Pattern name "Trade Blip" is the MatchEvent subject.</strong>
     * <strong style="color:ForestGreen">// This is an ordered pattern meaning events must arrive in the specified order to match the pattern.</strong>
     * final EventPattern.Builder builder = EventPattern.builder("Trade Blip", EventPattern.PatternType.ORDERED);
     * <strong style="color:ForestGreen">// First parameter is an order report for the ACME equity symbol.</strong>
     * final MessageKey orderKey = new EMessageKey(OrderReport.class, "ACME");
     * <strong style="color:ForestGreen">// Second parameter is a trade report for the ACME equity symbol.</strong>
     * final MessageKey tradeKey = new EMessageKey(EquityTrade.class, "ACME");
     *
     * builder.beginParameterMap()                <strong style="color:ForestGreen">// Must specify this before defining parameters</strong>
     *        .beginParameter("ord")              <strong style="color:ForestGreen">// Parameter name is used in the pattern below.</strong>
     *        .messageKey(orderKey)               <strong style="color:ForestGreen">// 1. Specify a ESubscribeFeed (required).</strong>
     *        .scope(EFeed.FeedScope.REMOTE_ONLY) <strong style="color:ForestGreen">// 2. Specify the subscribe feed scope (required).</strong>
     *                                            <strong style="color:ForestGreen">// 3. Subscription condition is optional and not defined.</strong>
     *        .endParameter()                     <strong style="color:ForestGreen">// Parameter "ord" defined.</strong>
     *        .beginParameter("trd")
     *        .messageKey(tradeKey)
     *        .scope(EFeed.FeedScope.REMOTE_ONLY)
     *        .endParameter()                     <strong style="color:ForestGreen">// Parameter "trd" defined.</strong>
     *        .endParameterMap()                  <strong style="color:ForestGreen">// All parameters defined.</strong>
     *        <strong style="color:ForestGreen">// Ordered pattern defined starting here.</strong></code></pre>
     * <h2>Defining Parameter Maps</h2>
     * <p>
     * If multiple event parameters are based on the same
     * parameters, then defining a parameter map once for all
     * patterns is desirable. The map definition is
     * {@code Map<String, EventPattern.FeedInfo>} where the
     * {@code String} key is the parameter name used in the
     * preceding example and the {@code FeedInfo} defines the
     * {@code ESubscribeFeed}. The parameter map definition is
     * similar to the previous {@code EventPattern.Builder}
     * example:
     * </p>
     * <pre><code><strong style="color:ForestGreen"> // All parameters use the new word feed.</strong>
     * final EMessageKey orderKey = new EMessageKey(OrderReport.class, "ACME");
     * final EMessageKey tradeKey = new EMessageKey(EquityTrade.class, "ACME");
     * final Map&lt;String, EventPattern.FeedInfo&gt; <strong style="color:BrickRed">params</strong> = new HashMap&lt;&gt;();
     *
     * <strong style="color:ForestGreen">// Defining "ord" parameter.</strong>
     * params.put("ord",                                                <strong style="color:ForestGreen">// Parameter name is used in pattern definition.</strong>
     *            new EventPattern.FeedInfo(orderKey,                   <strong style="color:ForestGreen">// 1. Define ESubscribeFeed message key (required).</strong>
     *                                      EFeed.FeedScope.REMOTE_ONLY <strong style="color:ForestGreen">// 2. Define subscribe feed scope (required).</strong>
     *                                                                  <strong style="color:ForestGreen">// 3. Subscription condition is optional and not provided here.</strong>
     *                                     ));
     *
     * <strong style="color:ForestGreen">// Defining "trd" parameter.</strong>
     * params.put("trd",
     *            new EventPattern.FeedInfo(tradeKey,
     *                                      EFeed.FeedScope.REMOTE_ONLY));
     *
     * <strong style="color:ForestGreen">// Create builder using the parameter map.</strong>
     * final EventPattern.Builder builder = EventPattern.builder("Trade Blip", EventPattern.PatternType.ORDERED, <strong style="color:BrickRed">params</strong>);</code></pre>
     * <h1>Define Event Patterns</h1>
     * With the event parameters defined, it is now time to
     * define the event pattern. Patterns come in two types:
     * ordered and unordered.
     * <h2>Ordered Pattern</h2>
     * Events must arrive in the pattern defined order to
     * generate a match. Order patterns are defined by two
     * components: single and multiple.
     * <h3>Single Component</h3>
     * <p>
     * A single component is associated with a single parameter
     * and has two optional properties: match count and match
     * condition.
     * </p>
     * <p>
     * A match count specifies how the minimum and maximum times
     * an event must contiguously appear in the pattern.
     * {@code matchCount(2, 4)} means that the event must appear
     * at least twice in succession and at most four times. If
     * the event occurs only once and is followed by a different
     * event or more than 4 times in succession, then the pattern
     * match fails.
     * </p>
     * <p>
     * A match condition is a
     * {@link MatchCondition} where the two
     * arguments are the latest event and a read-only list of
     * the previously matched events. This method returns
     * {@code true} if the current event matches the pattern
     * condition, allowing the matching process to continue or
     * complete. When {@code false} is returned, then this
     * current match fails.
     * <p>
     * The match condition takes three arguments: the latest,
     * unmatched event, the capturing groups map, and the
     * user-defined data map.
     * </p>
     * <h4><a id="CapturingGroups">CapturingGroups</a></h4>
     * <p>
     * Like {@link java.util.regex.Pattern}, one or more named
     * capturing groups map be defined using
     * {@link Builder#beginGroup(String)} and
     * {@link Builder#endGroup(String)}. Any matching events
     * which occur between {@code beginGroup} and
     * {@code endGroup} are stored in a {@code List} associated
     * with the group name. Capturing group lists are retrieved
     * from the group map using the group name as the key.
     * </p>
     * <p>
     * All event patterns have one capturing group:
     * {@link EventPattern#ALL_EVENTS}. The "all events" group
     * contains all matched events.
     * </p>
     * <h4><a id="UserCache">User Cache</a></h4>
     * <p>
     * eBus provides pattern client with a map to store
     * user-defined data. This map is carried through the
     * matching process, allowing {@link MatchCondition}s to
     * access information calculated in earlier matches. If a
     * pattern is successfully matched, then this forwarded in
     * {@link MatchEvent#userCache} in case this user-defined
     * information is needed for post-match processing.
     * </p>
     * <p>
     * The user-defined map is typed
     * <code>Map&lt;Object, Object&gt;</code> so as to allow the
     * user full flexibility with respect what {@code Object}s
     * may be used as keys and values.
     * </p>
     * <h3>Multi-Component</h3>
     * <p>
     * A multi-component acts like a Java regular expression
     * character class with a quantifier
     * <code>[abc]{<em>n</em>, <em>m</em>}</code>.
     * Multi-components are composed of multiple single
     * components. Each single component may have an optional
     * match condition <em>but not a match count</em>. That is
     * because the match count is applied to the multi-component
     * as a whole. The following is a multi-component example
     * picks up where the previous examples left off. The pattern
     * parameters are defined and the pattern builder
     * instantiated. Now it is time to define the event pattern
     * itself.
     * </p>
     * <pre><code> builder.beginGroup("g0")      <strong style="color:ForestGreen">// opening the "g0" named capturing group</strong>
     *        .beginMultiComponent() <strong style="color:ForestGreen">// Begin the multi-component definition.</strong>
     *        .matchCount(1, 2)      <strong style="color:ForestGreen">// The subcomponents may appear once or twice</strong>
     *        <strong style="color:ForestGreen">// MatchCondition parameters:</strong>
     *        <strong style="color:ForestGreen">// e: latest event</strong>
     *        <strong style="color:ForestGreen">// g: named capturing groups map where key is the group name.</strong>
     *        <strong style="color:ForestGreen">// u: user-defined data cache.</strong>
     *        .addSubordinate("ord", <strong style="color:ForestGreen">// Either the first event is an order with quantity &gt; 1,000 OR ...</strong>
     *                        (e, g, u) -&gt; {
     *                            final OrderReportMessage ord = (OrderReportMessage) e;
     *                            final boolean retcode = (ord.tradedQuantity &gt; 1_000);
     *
     *                            <strong style="color:ForestGreen">// If the order event meets the condition, then store the trade.</strong>
     *                            if (retcode) {
     *                                <strong style="color:ForestGreen">// Average price calculation requires:</strong>
     *                                <strong style="color:ForestGreen">// 1. Sumation of trade prices.</strong>
     *                                <strong style="color:ForestGreen">// 2. Number of trades.</strong>
     *                                u.put(sum, BigDecimal.valueOf(ord.tradedPrice));
     *                                u.put(count, BigDecimal.ONE);
     *                            }
     *
     *                            return (retcode);
     *                        })
     *        .addSubordinate("trd", <strong style="color:ForestGreen">// ... a trade at the currently lowest price for the trading session.</strong>
     *                        (e, g, u) -&gt; {
     *                            final EquityTradeMessage trd = (EquityTradeMessage) e;
     *                            final boolean retcode = (trd.tradeType == PriceType.LOW);
     *
     *                            if (retcode) {
     *                                u.put(sum, (trd.trade).price);
     *                                u.put(count, BigDecimal.ONE);
     *                            }
     *
     *                            return (retcode);
     *                        })
     *        .endMultiComponent() <strong style="color:ForestGreen">// Multi-component definition ended.</strong>
     *        .endGroup("g0")
     *        .beginGroup("g1")                 <strong style="color:ForestGreen">// Collect all the middle trades.</strong>
     *        .beginSingleComponent("trd")
     *        .matchCount(4, Integer.MAX_VALUE) <strong style="color:ForestGreen">// Look for at least four trades with increasing prices.</strong>
     *        .matchCondition(                  <strong style="color:ForestGreen">// Get the all matched events list (so far) and calculate the average trade price.</strong>
     *             (e, g, u) -&gt; {
     *                 final EquityTradeMessage trd = (EquityTradeMessage) e;
     *                 final BigDecimal trdPx = (trd.trade).price;
     *                 final BigDecimal pxSum = (BigDecimal) u.get(sum);
     *                 final BigDecimal numTrades = (BigDecimal) u.get(count);
     *                 final BigDecimal avgPx = pxSum.divide(numTrades);
     *                 final boolean retcode = (trdPx.compareTo(avgPx) &gt; 0);
     *
     *                 <strong style="color:ForestGreen">// If this trade met the condition, then add its price to the sum.</strong>
     *                 if (retcode) {
     *                     u.put(sum, pxSum.add(trdPx));
     *                     u.put(count, numTrades.add(BigDecimal.ONE));
     *                 }
     *
     *                 return (retcode);
     *             })
     *        .endSingleComponent()
     *        .endGroup("g1")
     *        .beginSingleComponent("trd") <strong style="color:ForestGreen">// Look for a final trade with a quantity less than 80% of the previous trade.</strong>
     *        .matchCondition(
     *             (e, g, u) -&gt; {
     *                 final EquityTrade trade = (EquityTrade) e;
     *                 final List&lt;ENotificationMessage&gt; trades = g.get(EventPattern.ALL_EVENTS);
     *                 final EquityTrade prevTrade = (EquityTrade) (trades.get(trades.size() - 1));
     *
     *                 return ((trade.trade).size &lt; (int) (0.8 * (prevTrade.trade).size));
     *             })
     *        .endSingleComponent(); <strong style="color:ForestGreen">// End of pattern definition.</strong></code></pre>
     * <h2>Unordered Patterns</h2>
     * <p>
     * An unordered pattern matches events based on match count
     * only, allowing events to arrive in no particular order.
     * Once the desired number of events are collected, then a
     * {@link MatchEvent} is generated and passed to the
     * subscriber.
     * </p>
     * <p>
     * The following looks for at least 5 consecutive orders and
     * trades demonstrating a price rise and there must be at
     * least one order and one trade. This example uses the same
     * parameter map as used in the ordered pattern example.
     * </p>
     * <pre><code> builder.beginSingleComponent("ord")
     *        .matchCount(4)
     *        .endSingleComponent()
     *        .beginSingleComponent("trd")
     *        .matchCount(4)
     *        .endSingleComponent()
     * </code></pre>
     * <p>
     * The above pattern is not enough to guarantee the pattern
     * requirement for at least one order and one trade. This
     * is done using a
     * <a href="PatternCondition">pattern condition</a>
     * (which see).
     * </p>
     * <h2>Pattern Attributes</h2>
     * <p>
     * There are three attributes which apply to the entire
     * pattern:
     * {@link Builder#until(BiPredicate) until(condition)},
     * {@link Builder#isExclusive(boolean) isExclusive(flag)} and
     * {@link Builder#patternCondition(Predicate)}.
     * </p>
     * <h3>Pattern Until Condition</h3>
     * <p>
     * {@code Builder.until(condition)} says that the pattern
     * match is valid as long as {@code condition} returns
     * {@code true}. If the condition returns {@code false}, then
     * the current match is abandoned. The most common use is
     * by comparing the first and latest message timestamps and
     * continuing the match as long as the time difference is
     * within a time limit. The default {@code until} condition
     * always returns {@code true}.
     * </p>
     * <pre><code> builder.until(<strong style="color:ForestGreen">// Keep matching as long as the first and latest events are within one hour.</strong>
     *               <strong style="color:ForestGreen">// t: all collected matching events so far.</strong>
     *               <strong style="color:ForestGreen">// e: latest event.</strong>
     *               (t, e) -&gt;
     *               {
     *                   final long duration = (e.timestamp - (t.get(0)).timestamp);
     *
     *                   return (duration &lt;= ONE_HOUR_LIMIT);
     *               })</code></pre>
     * <h3>Pattern IsExclusive Flag</h3>
     * <p>
     * {@code Builder.isExclusive(flag)} specifies whether events
     * appearing in one successful within a pattern may be used
     * for another match <em>within the same pattern.</em> If
     * {@code flag} is {@code true}, then events may not be
     * shared between matches. This means that once a match
     * occurs, then all in-progress matches are thrown away and
     * matching starts from scratch.
     * </p>
     * <p>
     * Please note that exclusivity applies only within a pattern
     * and note between patterns. The same event may be used to
     * satisfy two different patterns.
     * </p>
     * <pre><code> builder.isExclusive(true);</code></pre>
     * <h3><a id="PatternCondition">Pattern Condition</a></h3>
     * <p>
     * A pattern condition is applied to the generated
     * {@link MatchEvent}. Only if the condition returns
     * {@code true} is the match event posted to the subscriber
     * <em>and</em> if the pattern is exclusive are the other
     * in-progress matches discarded.
     * </p>
     * <p>
     * The pattern condition is optional. If not defined, the
     * default pattern condition always returns {@code true}.
     * </p>
     * <p>
     * Continuing with the unordered pattern example, the
     * following pattern condition guarantees at least one
     * order and one trade.
     * </p>
     * <pre><code> builder.patternCondition(
     *                 p -&gt;
     *                 {
     *                     final List&lt;ENotificationMessage&gt; events = p.group(EventPattern.ALL_EVENTS);
     *                     final boolean hasOrder = containsClass(OrderReport.class, events); <strong style="color:ForestGreen">// containsClass returns true if the list contains at least one instance of the class.</strong>
     *                     final boolean hasTrade = containsClass(EquityTrade.class, events);
     *
     *                     return (hasOrder &amp;&amp; hasTrade);
     *                 });
     * </code></pre>
     * <h2>Pattern Instantiation</h2>
     * <p>
     * Once the event parameters, event pattern, and pattern
     * attributes are defined, an {@link EventPattern} instance
     * is created using {@link Builder#build()}.
     * </p>
     * <pre><code>final EventPattern pattern = builder.build();</code></pre>
     *
     * @see #builder(String, EventPattern.PatternType)
     * @see #builder(String, EventPattern.PatternType, Map)
     */
    public static final class Builder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * User-assigned (hopefully) unique pattern name.
         */
        private final String mPatternName;

        /**
         * Specifies whether this is an ordered or unordered
         * pattern.
         */
        private final PatternType mPatternType;

        /**
         * Maps a unique name to its feed information. The keys
         * act as names used in the event pattern.
         */
        private final Map<String, FeedInfo> mParameters;

        /**
         * Contains successfully created pattern event components.
         * Ordering is significant if {@link #mPatternType} is
         * {@link PatternType#ORDERED}.
         */
        private final List<PatternComponent> mComponents;

        /**
         * Currently open match groups. This list acts as a stack
         * with the most recently opened group. Groups must be
         * closed in last-in-first-out (LIFO) order. The
         * bottom-most group is {@link #ALL_EVENTS} which may
         * never be closed.
         */
        private final Deque<String> mActiveGroups;

        /**
         * Contains both active and inactive group names.
         * Used to verify that group names are unique.
         */
        private final Set<String> mGroupNames;

        /**
         * The "until" predicate determines if collected events
         * occur with a given period. This period is most likely
         * a time duration but does not have to be. It may be
         * based on message fields other than
         * {@link ENotificationMessage#timestamp}.
         * <p>
         * This predicate takes two parameters: 1) the list of
         * all currently collected events and 2) the latest
         * event candidate for adding to the list. Returns
         * {@code true} if the candidate is within the collection
         * framework and {@code false} if not. Defaults to
         * {@link #DEFAULT_UNTIL} which always returns true.
         * </p>
         */
        private BiPredicate<List<ENotificationMessage>, ENotificationMessage> mUntil;

        /**
         * Set to {@code true} if pattern matching is exclusive.
         */
        private boolean mExclusive;

        /**
         * Currently collecting this component type.
         */
        private PatternState mState;

        //
        // Temporary holding data.
        // The following data members hold on to input data
        // waiting for the end state.

        /**
         * Parameter name. Used both to define a parameter or
         * to associated parameter with a single component.
         */
        private String mParamName;

        /**
         * The event message key.
         */
        private EMessageKey mEventKey;

        /**
         * Event message feed scope. Required for parameter
         * definition.
         */
        private FeedScope mScope;

        /**
         * Optional event subscription feed condition. May be
         * {@code null}.
         */
        private ECondition mSubCondition;

        /**
         * Minimum component match count.
         */
        private int mMinMatchCount;

        /**
         * Maximum component match count.
         */
        private int mMaxMatchCount;

        /**
         * This condition is applied to a newly completed match
         * event which contains all the matched notification
         * messages. Only if this condition returns {@code true}
         * if the {@link MatchEvent} delivered to the subscriber.
         * The default condition is {@link #DEFAULT_CONDITION}.
         */
        private Predicate<MatchEvent> mPatternCondition;

        /**
         * Predicate applied to received event.
         */
        private MatchCondition mMatchCondition;

        /**
         * Multi-component subordinate single component list.
         * Set to {@code null} when not defining a
         * multi-component.
         */
        private final List<SinglePatternComponent> mSubComponents;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates an event pattern builder for the given
         * parameters.
         * @param patternName pattern name used as the
         * {@code MatchEvent} subject.
         * @param pType pattern type - ordered or unordered.
         * @param parameters pattern subscribe feeds.
         * @param parmsDefined {@code true} if the client
         * provided a parameters map.
         */
        private Builder(final String patternName,
                        final PatternType pType,
                        final Map<String, FeedInfo> parameters,
                        final boolean parmsDefined)
        {
            mPatternName = patternName;
            mPatternType = pType;
            mComponents = new ArrayList<>();
            mParameters = parameters;
            mActiveGroups = new LinkedList<>();
            mGroupNames = new TreeSet<>();
            mSubComponents = new ArrayList<>();

            // Initialize the group names with all groups.
            mActiveGroups.addFirst(ALL_EVENTS);
            mGroupNames.add(ALL_EVENTS);
            mUntil = DEFAULT_UNTIL;
            mExclusive = false;
            mPatternCondition = DEFAULT_CONDITION;

            // Set the state based on whether parameters were
            // provided and, if so, whether this is an ordered
            // or unordered pattern.
            if (!parmsDefined)
            {
                mState = PatternState.NO_PARAMS;
            }
            else
            {
                if (pType == PatternType.ORDERED)
                {
                    mState = PatternState.IN_ORDERED_PATTERN;
                }
                else
                {
                    mState = PatternState.IN_UNORDERED_PATTERN;
                }

                // Set the parameter identifiers.
                assignTransitionIds(parameters);
            }
        } // end of Builder(PatternType, Map, boolean)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the user-assigned pattern name.
         * @return pattern name.
         */
        private String patternName()
        {
            return (mPatternName);
        } // end of patternName()

        /**
         * Returns the pattern type.
         * @return pattern type.
         */
        private PatternType patternType()
        {
            return (mPatternType);
        } // end of patternType()

        /**
         * Returns the parameter name-to-feed information map.
         * @return event parameter map.
         */
        private Map<String, FeedInfo> parameters()
        {
            return (mParameters);
        } // end of parameters()

        /**
         * Returns the event pattern components.
         * @return event pattern components.
         */
        private List<PatternComponent> components()
        {
            return (mComponents);
        } // end of components()

        /**
         * Returns the event pattern until predicate.
         * @return until predicate
         */
        private BiPredicate<List<ENotificationMessage>, ENotificationMessage> until()
        {
            return (mUntil);
        } // end of until()

        /**
         * Returns the exclusive flag.
         * @return exclusive flag.
         */
        private boolean exclusive()
        {
            return (mExclusive);
        } // end of exclusive()

        /**
         * Returns the condition associated with the entire
         * pattern. Only if the match event satisfies this
         * condition is it forwarded to the subscriber.
         * @return match event condition.
         */
        private Predicate<MatchEvent> patternCondition()
        {
            return (mPatternCondition);
        } // end of patternCondition()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Begins the parameter map definition.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if parameter map cannot be defined in the current
         * state.
         *
         * @see #beginParameter(String)
         * @see #endParameterMap()
         */
        public Builder beginParameterMap()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(BEGIN_PARAM_MAP_MASK))
            {
                // No. Cannot call this method from the current
                // state.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support beginParameterMap"));
            }

            // Ready to start entering parameter definitions.
            mState = PatternState.IN_PARAMS;

            return (this);
        } // end of beginParameterMap()

        /**
         * Ends the parameter map definition which begins the
         * event pattern definition.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if not defining a parameter map or parameter map is
         * empty (no parameters defined).
         *
         * @see #beginParameterMap()
         * @see #beginSingleComponent(String)
         */
        public Builder endParameterMap()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(END_PARAM_MAP_MASK))
            {
                // No. Cannot call this method from the current
                // state.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support endParameterMap"));
            }

            // Were any parameters defined?
            if (mParameters.isEmpty())
            {
                throw (
                    new IllegalStateException(
                        "no parameters defined"));
            }

            // Parameter map defined. Move on to the ordered or
            // unordered event pattern definition.
            mState = (mPatternType == PatternType.ORDERED ?
                      PatternState.IN_ORDERED_PATTERN :
                      PatternState.IN_UNORDERED_PATTERN);

            // Assign the feed identifiers.
            // Set the parameter identifiers.
            assignTransitionIds(mParameters);

            return (this);
        } // end of endParameterMap()

        /**
         * Begin a parameter definition for the given parameter
         * name. The associated notification message key and feed
         * scope must be defined next. The subscription condition
         * is optional.
         * @param name parameter name.
         * @return {@code this} event pattern builder.
         * @throws NullPointerException
         * if {@code name} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code name} is an empty string.
         * @throws IllegalStateException
         * if the parameter map is already defined, a parameter
         * may not be defined in the current state, or the
         * parameter map already contains the named parameter.
         *
         * @see #endParameter()
         * @see #messageKey(EMessageKey)
         * @see #scope(EFeed.FeedScope)
         * @see #condition(ECondition)
         */
        public Builder beginParameter(final String name)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(BEGIN_PARAMETER_MASK))
            {
                // No. Cannot call this method from the current
                // state.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support beginParameter"));
            }

            // Is the name valid?
            if (name == null)
            {
                throw (
                    new NullPointerException(
                        "parameter name is null"));
            }

            if (name.isEmpty())
            {
                // No. Parameter name cannot empty.
                throw (
                    new IllegalArgumentException(
                        "parameter name is empty"));
            }

            // Is this parameter already defined?
            if (mParameters.containsKey(name))
            {
                throw (
                    new IllegalStateException(
                        "\"" + name + "\" already defined"));
            }

            mParamName = name;
            mState = PatternState.IN_PARAMETER;

            return (this);
        } // end of beginParameter()

        /**
         * Ends the event pattern parameter definition using the
         * previously specified name, event message key, and
         * event feed scope.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if a parameter cannot be ended in the current state or
         * the parameter is not completely defined.
         *
         * @see #beginParameter(String)
         */
        public Builder endParameter()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(END_PARAMETER_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support endParameter"));
            }

            // Are the necessary data members defined?
            if (mEventKey == null)
            {
                // Notification message key is missing.
                throw (
                    new IllegalStateException(
                        "event message key missing"));
            }

            if (mScope == null)
            {
                // Subscription feed scope missing.
                throw (
                    new IllegalStateException(
                        "feed scope missing"));
            }
            // Subscription condition may be null.

            // Add the parmeter definition to the map.
            mParameters.put(
                mParamName,
                new FeedInfo(mEventKey, mScope, mSubCondition));

            // Clear the temporary data for the next parameter
            // definition.
            mParamName = null;
            mEventKey = null;
            mScope = null;
            mSubCondition = null;

            mState = PatternState.IN_PARAMS;

            return (this);
        } // end of endParameter()

        /**
         * Sets the parameter notification message key.
         * @param key notification message key.
         * @return {@code this} event pattern builder.
         * @throws NullPointerException
         * if {@code key} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code key} is not a notification message key.
         * @throws IllegalStateException
         * if the notification message key cannot be defined in
         * the current state or the notification message key is
         * already defined.
         *
         * @see #scope(EFeed.FeedScope)
         * @see #condition(ECondition)
         */
        public Builder messageKey(final EMessageKey key)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(MESSAGE_KEY_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support messageKey"));
            }

            // Was a message key provided?
            if (key == null)
            {
                throw (new NullPointerException("key is null"));
            }

            // Was a notification message key provided?
            if (!key.isNotification())
            {
                throw (
                    new IllegalArgumentException(
                        key + " is not a notification key"));
            }

            // Is the notification key already defined?
            if (mEventKey != null)
            {
                throw (
                    new IllegalStateException(
                        "notification key already defined"));
            }

            // Everything checks out.
            mEventKey = key;

            return (this);
        } // end of messageKey(EMessageKey)

        /**
         * Sets the subscription feed scope.
         * @param scope feed scope.
         * @return {@code this} event pattern builder.
         * @throws NullPointerException
         * if {@code scope} is {@code null}.
         * @throws IllegalStateException
         * if the scope cannot be defined in the current state or
         * the feed scope is already defined.
         *
         * @see #messageKey(EMessageKey)
         * @see #condition(ECondition)
         */
        public Builder scope(final FeedScope scope)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(SCOPE_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support scope"));
            }

            // Was a scope provided?
            if (scope == null)
            {
                throw (new NullPointerException("scope is null"));
            }

            // Is the scope already defined?
            if (mScope != null)
            {
                throw (
                    new IllegalStateException(
                        "scope already defined"));
            }

            // Everything checks out.
            mScope = scope;

            return (this);
        } // end of scope(FeedScope)

        /**
         * Sets the optional subscription feed condition.
         * @param condition feed condition. May be {@code null}.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if the condition cannot be defined in the current
         * state or the condition is already defined.
         */
        public Builder condition(final ECondition condition)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(CONDITION_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support condition"));
            }

            // Is the scope already defined?
            if (mSubCondition != null)
            {
                throw (
                    new IllegalStateException(
                        "condition already defined"));
            }

            // Everything checks out.
            mSubCondition = condition;

            return (this);
        } // end of condition(ECondition)

        /**
         * Begins the collection of event pattern single parameter
         * component.
         * @param paramName component uses this feed name.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if a component cannot be defined in the current state.
         * @throws NullPointerException
         * if {@code paramName} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code paramName} is empty or does not reference
         * a known parameter.
         */
        public Builder beginSingleComponent(final String paramName)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(BEGIN_SINGLE_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support beginSingleComponent"));
            }

            // Is a valid parameter name provided?
            validateParamName(paramName);

            // Everything checks out. Store away the parameter
            // name and start collecting single component
            // information.
            mParamName = paramName;
            mState = PatternState.IN_SINGLE;

            // Set the default minimum, maximum match counts and
            // condition.
            mMinMatchCount = DEFAULT_MATCH_COUNT;
            mMaxMatchCount = DEFAULT_MATCH_COUNT;
            mMatchCondition = DEFAULT_PREDICATE;

            return (this);
        } // end of beginSingleComponent(String)

        /**
         * Adds the defined single event pattern component to the
         * component list.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if a single event pattern component is not being
         * defined.
         */
        public Builder endSingleComponent()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(END_SINGLE_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support endSingleComponent"));
            }

            // Add the single component to the component list.
            final String[] groupNames =
                mActiveGroups.toArray(EMPTY_NAMES);
            final FeedInfo fInfo = validateParamName(mParamName);

            mComponents.add(
                new SinglePatternComponent(
                    mMinMatchCount,
                    mMaxMatchCount,
                    groupNames,
                    fInfo.transitionId(),
                    mParamName,
                    mMatchCondition));

            // Reset the component data.
            mMinMatchCount = DEFAULT_MATCH_COUNT;
            mMaxMatchCount = DEFAULT_MATCH_COUNT;
            mMatchCondition = DEFAULT_PREDICATE;

            mState = (mPatternType == PatternType.ORDERED ?
                      PatternState.IN_ORDERED_PATTERN :
                      PatternState.IN_UNORDERED_PATTERN);

            return (this);
        } // end of endSingleComponent()

        public Builder beginMultiComponent()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(BEGIN_MULTI_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support beginMultiComponent"));
            }


            mState = PatternState.IN_MULTI;

            return (this);
        } // end of beginMultiComponent()

        public Builder endMultiComponent()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(END_MULTI_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support endMultiComponent"));
            }

            if (mSubComponents.isEmpty())
            {
                throw (
                    new IllegalStateException(
                        "no subcomponents defined for multi-component"));
            }

            // Add the multi component to the component list.
            final String[] groupNames =
                mActiveGroups.toArray(new String[0]);
            final SinglePatternComponent[] components =
                mSubComponents.toArray(
                    new SinglePatternComponent[0]);

            mComponents.add(
                new MultiPatternComponent(mMinMatchCount,
                                          mMaxMatchCount,
                                          groupNames,
                                          components));

            // Reset the component data. Clear out the
            // sub-components list for the next multi-component.
            mMinMatchCount = DEFAULT_MATCH_COUNT;
            mMaxMatchCount = DEFAULT_MATCH_COUNT;
            mSubComponents.clear();

            // Since a multi-component only in ordered patterns,
            // then the next state is the ordered pattern state.
            mState = PatternState.IN_ORDERED_PATTERN;

            return (this);
        } // end of endMultiComponent()

        /**
         * Creates a new named capture group. All captured events
         * between this {@code beginGroup} and matching
         * {@code endGroup} call will be stored in this group's
         * capture list.
         * @param groupName capture group name.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if not at the top-level event pattern state (that is,
         * not inside a pattern component definition).
         * @throws IllegalArgumentException
         * if {@code groupName} is either {@code null}, empty, or
         * redundant. A capture group name may only be used once.
         */
        public Builder beginGroup(final String groupName)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(BEGIN_GROUP_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support beginGroup"));
            }

            // Is the group name valid?
            validateGroupName(groupName);

            if (mGroupNames.contains(groupName))
            {
                throw (
                    new IllegalArgumentException(
                        groupName + " is not unique"));
            }

            mActiveGroups.addFirst(groupName);
            mGroupNames.add(groupName);

            return (this);
        } // end of beginGroup(String)

        /**
         * Closes the named capture group. The named group must
         * be the most recently opened group. This is because
         * capture groups must be opened and closed in
         * last-in-first-out (LIFO) order.
         * @param groupName capture group name.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if not at the top-level event pattern state (that is,
         * not inside a pattern component definition).
         * @throws IllegalArgumentException
         * if {@code groupName} is either {@code null}, empty, is
         * unknown, is {@link #ALL_EVENTS}, or is not the most
         * recently opened capture group.
         */
        public Builder endGroup(final String groupName)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(END_GROUP_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support endGroup"));
            }

            if (!mGroupNames.contains(groupName))
            {
                throw (
                    new IllegalArgumentException(
                        "\"" + groupName + "\" is not known"));
            }

            // Not allowed to remove the "all events" group.
            if (ALL_EVENTS.equals(groupName))
            {
                throw (
                    new IllegalArgumentException(
                        "cannot end " + ALL_EVENTS + " group"));
            }

            // Groups must be opened and closed in LIFO order.
            // Make sure that the given group name matches the
            // name on top of the active groups list.
            if (!groupName.equals(mActiveGroups.peekFirst()))
            {
                throw (
                    new IllegalStateException(
                        groupName + " not most recently opened group"));
            }

            // Remove the most-recently added group from the
            // active group stack.
            mActiveGroups.removeFirst();

            return (this);
        } // end of endGroup(String)

        /**
         * Sets the minimum and maximum match counts to the same
         * value.
         * @param minMax minimum and maximum component match
         * count.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if a event pattern component is not being defined.
         * @throws IllegalArgumentException
         * if either {@code min} or {@code max} is &lt; zero or
         * {@code min} &gt; {@code max}.
         */
        public Builder matchCount(final int minMax)
        {
            return (this.matchCount(minMax, minMax));
        } // end of matchCount(int)

        /**
         * Sets the minimum and maximum match counts to the given
         * values.
         * @param min minimum component match count.
         * @param max maximum component match count.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if a event pattern component is not being defined.
         * @throws IllegalArgumentException
         * if either {@code min} or {@code max} is &lt; zero or
         * {@code min} &gt; {@code max}.
         */
        public Builder matchCount(final int min,
                                  final int max)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(MATCH_COUNT_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support matchCount"));
            }

            // Are the match counts valid?
            if (min < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "min (" + min + ") < zero"));
            }

            if (max < min)
            {
                throw (
                    new IllegalArgumentException(
                        "min (" + min + ") > max (" + max + ")"));
            }

            mMinMatchCount = min;
            mMaxMatchCount = max;

            return (this);
        } // end of matchCount(int, int)

        /**
         * Sets the component match condition to the given
         * predicate. If {@code condition} is {@code null} then
         * sets the match condition to the default
         * {@link #DEFAULT_PREDICATE} which always returns
         * {@code true}.
         * @param p component match condition.
         * @return {@code this} event pattern builder.
         * @throws IllegalStateException
         * if a event pattern component is not being defined.
         */
        public Builder matchCondition(final MatchCondition p)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(PREDICATE_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support matchCondition"));
            }

            // Is condition null?
            if (p == null)
            {
                // Yes. Use the default condition which returns
                // true.
                mMatchCondition = DEFAULT_PREDICATE;
            }
            else
            {
                mMatchCondition = p;
            }

            return (this);
        } // end of matchCondition(MatchCondition)

        /**
         * Adds a subordinate, unconditional single component to
         * the current multi-component. The single component
         * match condition is set to {@link #DEFAULT_PREDICATE}.
         * @param paramName valid parameter name.
         * @return {@code this} event pattern builder.
         * @throws IllegalArgumentException
         * if {@code paramName} is not a valid parameter name.
         */
        public Builder addSubordinate(final String paramName)
        {
            return (addSubordinate(paramName, DEFAULT_PREDICATE));
        } // end of addSubordinate(String)

        /**
         * Adds a subordinate, conditional single component to
         * the current multi-component. The condition is optional
         * and may be {@code null}. If {@code predicate} is
         * {@code null}, then set to {@link #DEFAULT_PREDICATE}.
         * @param paramName valid parameter name.
         * @param predicate component match condition. May be
         * {@code null}.
         * @return {@code this} event pattern builder.
         * @throws IllegalArgumentException
         * if {@code paramName} is not a valid parameter name.
         */
        public Builder addSubordinate(final String paramName,
                                      final MatchCondition predicate)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(ADD_MULTI_SINGLE_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support addSubordinate"));
            }

            final FeedInfo fInfo = validateParamName(paramName);

            // Add the subordinate single component to the
            // multi-component.
            // Is a valid parameter name provided?
            // Is a message key provided?
            mSubComponents.add(
                new SinglePatternComponent(
                    DEFAULT_MATCH_COUNT,
                    DEFAULT_MATCH_COUNT,
                    EMPTY_NAMES,
                    fInfo.transitionId(),
                    paramName,
                    (predicate == null ?
                     DEFAULT_PREDICATE :
                     predicate)));

            return (this);
        } // end of addSubordinate(String, Predicate)

        public Builder until(final BiPredicate<List<ENotificationMessage>, ENotificationMessage> p)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(UNTIL_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support until"));
            }

            // Is duration valid?
            if (p == null)
            {
                throw (
                    new NullPointerException(
                        "duration is null"));
            }

            mUntil = p;

            return (this);
        } // end of until(BiPredicate)

        public Builder isExclusive(final boolean flag)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(EXCLUSIVE_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support isExclusive"));
            }

            mExclusive = flag;

            return (this);
        } // end of isExclusive(boolean)

        public Builder patternCondition(Predicate<MatchEvent> p)
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(PATTERN_COND_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support patternCondition"));
            }

            mPatternCondition =
                (p == null ? DEFAULT_CONDITION : p);

            return (this);
        } // end of patternCondition(Predicate<>)

        /**
         * Returns a new event pattern constructed from the
         * parameter map, components, and until duration entered
         * into {@code this} builder.
         * @return the built event pattern.
         */
        public EventPattern build()
        {
            // Does the current state support this method?
            if (!mState.supportsMethod(BUILD_MASK))
            {
                // No.
                throw (
                    new IllegalStateException(
                        "state " +
                        mState +
                        " does not support build"));
            }

            // Event pattern construction now complete.
            mState = PatternState.OUT_PATTERN;

            return (new EventPattern(this));
        } // end of build()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Throws an exception if {@code groupName} is not a
         * valid capture group name. If the group name is valid,
         * then returns. This method is called for effect only.
         * @param groupName capture group name.
         * @throws IllegalArgumentException
         * if {@code groupName} is either {@code null} or empty.
         */
        private void validateGroupName(final String groupName)
        {
            if (groupName == null)
            {
                throw (
                    new NullPointerException(
                        "groupName is null"));
            }

            if (groupName.isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "groupName is empty"));
            }
        } // end of validateGroupName(String)

        /**
         * Throws an exception if {@code paramName} is not a
         * valid feed parameter name. If the name is valid,
         * then returns. This method is called for effect only.
         * @param paramName component uses this feed name.
         * @throws IllegalArgumentException
         * if {@code paramName} is empty or does not reference
         * a known parameter.
         */
        private FeedInfo validateParamName(final String paramName)
        {
            if (paramName == null)
            {
                throw (
                    new NullPointerException(
                        "paramName is null"));
            }

            if (paramName.isEmpty())
            {
                throw (
                    new IllegalArgumentException(
                        "paramName is empty"));
            }

            if (!mParameters.containsKey(paramName))
            {
                throw (
                    new IllegalArgumentException(
                        "\"" +
                        paramName +
                        "\" is not a known parameter"));
            }

            return (mParameters.get(paramName));
        } // end of validateParamName(String)

        /**
         * Sets the transition identifiers for the given
         * event parameters.
         * @param parameters event parameters map.
         */
        private static void assignTransitionIds(final Map<String, FeedInfo> parameters)
        {
            int tId = 0;

            for (FeedInfo fInfo : parameters.values())
            {
                fInfo.transitionId(tId);
                ++tId;
            }
        } // end of assignTransitionIds(Map)
    } // end of Builder

    /**
     * Base class for single- and multi- pattern components.
     */
    /* package */ static abstract class PatternComponent
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * This component must be matched at least this many
         * times to satisfy the pattern. Defaults to one.
         */
        protected final int mMinMatchCount;

        /**
         * This component must be matched at most this many times
         * to satisfy the pattern. Defaults to one.
         */
        protected final int mMaxMatchCount;

        /**
         * This component is associated with these group names.
         * This will be empty for single components which are
         * part of a multiple component.
         */
        protected final String[] mGroupNames;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a event pattern component with the given
         * minimum and maximum match count.
         * @param minCount minimum component match count.
         * @param maxCount maximum component match count.
         * @param groupNames component added to these match
         * groups.
         */
        protected PatternComponent(final int minCount,
                                   final int maxCount,
                                   final String[] groupNames)
        {
            mMinMatchCount = minCount;
            mMaxMatchCount = maxCount;
            mGroupNames = groupNames;
        } // end of PatternComponent(int, int, String[])

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the minimum component match count.
         * @return match count.
         */
        public final int minimumMatchCount()
        {
            return (mMinMatchCount);
        } // end of minimumMatchCount()

        /**
         * Returns the maximum component match count.
         * @return match count.
         */
        public final int maximumMatchCount()
        {
            return (mMaxMatchCount);
        } // end of maximumMatchCount()

        /**
         * Returns the associated match group names.
         * @return group names.
         */
        public final String[] groupNames()
        {
            return (mGroupNames);
        } // end of groupNames()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of PatternComponent

    /**
     * Matches a single event type.
     */
    /* package */ static final class SinglePatternComponent
        extends PatternComponent
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Unique transition identifier. Used as an index into
         * the FSM table.
         */
        private final int mTransitionId;

        /**
         * This component matches this feed name.
         */
        private final String mParameter;

        /**
         * Condition must be true to accept event message.
         */
        private final MatchCondition mCondition;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new single event component for the given
         * feed name and component condition.
         * @param transId unique transition identifier.
         * @param minCount minimum component match count.
         * @param maxCount maximum component match count.
         * @param groupNames component added to these match
         * groups.
         * @param parameter event feed name.
         * @param condition component condition.
         */
        private SinglePatternComponent(final int minCount,
                                       final int maxCount,
                                       final String[] groupNames,
                                       final int transId,
                                       final String parameter,
                                       final MatchCondition condition)
        {
            super (minCount, maxCount, groupNames);

            mTransitionId = transId;
            mParameter = parameter;
            mCondition = condition;
        } // end of SinglePatternComponent(...)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the unique transition identifier.
         * @return transition identifier.
         */
        public final int transitionIdentifier()
        {
            return (mTransitionId);
        } // end of transitionIdentifier()

        /**
         * Returns the parameter name.
         * @return parameter name.
         */
        public String parameter()
        {
            return (mParameter);
        } // end of parameter()

        /**
         * Returns the component condition.
         * @return component condition.
         */
        public MatchCondition condition()
        {
            return (mCondition);
        } // end of condition()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of SinglePatternComponent

    /**
     * Matches one of several event types.
     */
    /* package */ static final class MultiPatternComponent
        extends PatternComponent
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Individual single components making up this
         * multi-component.
         */
        private final SinglePatternComponent[] mComponents;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private MultiPatternComponent(final int minCount,
                                      final int maxCount,
                                      final String[] groupNames,
                                      final SinglePatternComponent[] components)
        {
            super (minCount, maxCount, groupNames);

            mComponents = components;
        } // end of MultiPatternComponent(List)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the subordinate single components list.
         * @return subordinate single components.
         */
        public SinglePatternComponent[] components()
        {
            return (mComponents);
        } // end of components()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of MultiPatternComponent
} // end of class EventPattern
