//
// Copyright 2018 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.pattern;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.eBus.client.EClient;
import net.sf.eBus.client.EFeed;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.ESubscribeFeed;
import static net.sf.eBus.client.ESubscribeFeed.FEED_STATUS_METHOD;
import net.sf.eBus.client.ESubscriber;
import net.sf.eBus.client.FeedStatusCallback;
import net.sf.eBus.client.IESubscribeFeed;
import net.sf.eBus.client.NotifyCallback;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.util.ValidationException;
import net.sf.eBus.util.Validator;


/**
 * eBus pattern feeds sit between
 * {@link IESubscribeFeed eBus notification events} and an
 * {@link ESubscriber eBus client}, searching for a
 * client-specified  {@link EventPattern event pattern} in the
 * notification event stream. When a match is found, a
 * {@link MatchEvent} is posted to the client via the configured
 * notification callback, just like any other notification.
 * <p>
 * Follow these steps when using a pattern feed:
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 1:</strong> Implement
 * the {@link ESubscriber} interface.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 2:</strong> Define an
 * {@link EventPattern}.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 3:</strong> Open a
 * pattern feed for the subscriber and pattern defined above.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 4 (optional):</strong>
 * Do not override
 * {@link ESubscriber} interface methods. Instead, set callbacks
 * using
 * {@link EPatternFeed.PatternBuilder#statusCallback(FeedStatusCallback)}
 * and/or
 * {@link EPatternFeed.PatternBuilder#notifyCallback(NotifyCallback)}
 * expressions.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 5:</strong> Subscribe
 * to the open pattern feed.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 6:</strong> Wait for an
 * {@link EFeedState#UP up}
 * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed) feed status}.
 * This callback will occur before any {@link MatchEvent}s are
 * delivered. If the feed state is {@link EFeedState#DOWN down},
 * then no matches will be delivered until the feed state comes
 * back up.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 7:</strong> When the
 * pattern feed is up, wait for {@code MatchEvent}s to arrive.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 8:</strong> When the
 * subscriber is shutting down, {@link #close() close} the feed.
 * </p>
 * <h2>Example use of <code>EPatternFeed</code></h2>
 * <pre><code>import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.EPatternFeed;
import net.sf.eBus.client.ESubscriber;
import net.sf.eBus.messages.EventPattern;
import net.sf.eBus.messages.MatchEvent;

<strong style="color:ForestGreen">Step 1: Implement the ESubscriber interface.</strong>
public final class TradeAlgo implements ESubscriber
{
    // Monitor this stock symbol.
    private final String mSymbol;

    // Store the feed here so it can be used to unsubscribe.
    private EPatternFeed mFeed;

    public TradeAlgo(final String symbol) {
        mSymbol = symbol;
    }

    &#64;Override public void startup() {
        <strong style="color:ForestGreen">Step 2: Define event pattern.</strong>
        final EventPattern pattern = (see {@link EventPattern} to learn how to create a pattern.)

        <strong style="color:ForestGreen">Step 3: Open the pattern feed.</strong>
        mFeed = EPatternFeed.open(this, pattern);

        <strong style="color:ForestGreen">Step 4: ESubscriber interface method overridden.</strong>

        <strong style="color:ForestGreen">Step 5: Subscribe to the pattern feed.</strong>
        mFeed.subscribe();
    }


    <strong style="color:ForestGreen">Step 6: Wait for EFeedState.UP feed status.</strong>
    &#64;Override public void feedStatus(final EFeedState feedState, final IESubscribeFeed feed) {
        // What is the feed state?
        if (feedState == EFeedState.DOWN) {
            // Down. At least one pattern is down. Expect no matches until all
            // subordinate pattern feeds are up.
        } else {
            // Up. Expect to receive match notifications.
        }
    }

    <strong style="color:ForestGreen">Step 7: Wait for matches to arrive.</strong>
    &#64;Override public void notify(final ENotificationMessage msg, final IESubscribeFeed feed) {
        final MatchEvent match = (MatchEvent) msg;

        Match handling code here.
    }

    &#64;Override public void shutdown() {
        <strong style="color:ForestGreen">Step 8: When subscriber is shutting down, retract subscription feed.</strong>
        // mFeed.unsubscribe() is not necessary since close() will unsubscribe.
        if (mFeed != null) {
            mFeed.close();
            mFeed = null;
        }
    }
}</code></pre>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@SuppressWarnings ({"java:S6411"})
public abstract class EPatternFeed
    extends EFeed
    implements ESubscriber,
               IESubscribeFeed
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    protected static final Logger sLogger =
        Logger.getLogger(EPatternFeed.class.getName());

    /**
     * Used to generate unique match frame identifiers.
     */
    private static int sNextFrameId = 0;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Pattern name used as the {@link MatchEvent} subject.
     */
    protected final String mPatternName;

    /**
     * Publish match events on this message key. The
     * {@link EMessageKey#subject() message key subject} is
     * {@link #mPatternName}.
     */
    protected final EMessageKey mPubKey;

    /**
     * Collected matching events must satisfy this until
     * predicate. The collected events list and latest event are
     * the predicate parameters. This condition may be based on
     * any event field but will most likely use
     * {@link net.sf.eBus.messages.EMessage#timestamp}.
     */
    protected final BiPredicate<List<ENotificationMessage>, ENotificationMessage> mUntil;

    /**
     * Set to {@code true} if an event may be used for only one
     * match. If this is the case, then all extant match frames
     * are cleared when a single match completes.
     * <p>
     * <strong>Note:</strong> exclusivity applies only within a
     * pattern and not between patterns. Even if two patterns are
     * marked exclusive, a single event may still be used to
     * satisfy those two patterns.
     * </p>
     */
    protected final boolean mIsExclusive;

    /**
     * Match event is posted to subscriber if-and-only-if the
     * match event satisfies this condition. While this condition
     * may appear to play the same role as a feed condition,
     * the difference is that if this condition returns
     * {@code true} and the pattern is exclusive, that all
     * related match frames are removed.
     */
    protected final Predicate<MatchEvent> mCondition;

    /**
     * Maps the {@link EFeed#feedId() feed identifier} to the
     * subordinate notify feed. These feeds are defined by the
     * {@link EventPattern#parameters()}.
     */
    protected final List<ESubscribeFeed> mSubFeeds;

    /**
     * Maps an accepted notification event to the match frames
     * in which it appears. This map is used only for exclusive
     * pattern feeds. When a match frame completely matches the
     * pattern, then this map is used to find the match frames
     * referencing the same events.
     * <p>
     * Set to {@code null} for non-exclusive.
     * </p>
     */
    protected final Map<ENotificationMessage, List<AbstractMatchFrame>> mEventFrames;

    /**
     * Feed status callback. If not explicitly set by client,
     * then defaults to
     * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}.
     */
    protected final FeedStatusCallback<IESubscribeFeed> mStatusCallback;

    /**
     * Notification message callback. If not explicity set by
     * client, then defaults to
     * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}.
     */
    protected final NotifyCallback mNotifyCallback;

    /**
     * Bit mask for all subordinate feeds. Bits are mapped to
     * feed identifiers.
     */
    protected long mAllSubFeedsMask;

    /**
     * Bit mask used to determine if all subordinate feeds are
     * up. When a feed is up, its matching bit is set in this
     * bit mask. When this mask equals {@link #mAllSubFeedsMask},
     * then this event feed is up.
     */
    protected long mSubFeedMask;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new eBus pattern feed based on builder settings.
     * @param builder contains pattern feed settings.
     */
    @SuppressWarnings ("unchecked")
    protected EPatternFeed(final PatternBuilder builder)
    {
        super (builder);

        final EventPattern pattern = builder.mPattern;

        mPatternName = pattern.patternName();
        mPubKey =
            new EMessageKey(MatchEvent.class, mPatternName);
        mUntil = pattern.until();
        mIsExclusive = pattern.isExclusive();
        mCondition = pattern.patternCondition();
        mEventFrames = (mIsExclusive ? new HashMap<>() : null);
        mStatusCallback = builder.mStatusCallback;
        mNotifyCallback = builder.mNotifyCallback;

        mSubFeeds = new ArrayList<>();
        mSubFeedMask = 0L;
        mAllSubFeedsMask = 0L;
    } // end of EPatternFeed(PatternBuilder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Applies {@code event} to the pattern. If this event
     * completes the pattern, then a {@link MatchEvent} is
     * created, tested with the match condition, and, if the match
     * event satisfies the condition, the match event is
     * forwarded to the subscriber.
     * @param event apply this event to the pattern.
     * @param eventId unique identifier associated with the
     * event type.
     */
    protected abstract void matchEvent(ENotificationMessage event,
                                       int eventId);

    //
    // end of Abstract Method Declarations.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Overrides.
    //

    /**
     * Retracts all in place, subordinate feed subscriptions.
     */
    @Override
    protected final void inactivate()
    {
        // Retract the subscription in case it is in place.
        unsubscribe();
    } // end of inactivate()

    //
    // end of Abstract Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // IESubscribeFeed Interface Implementation.
    //

    /**
     * Returns a message key with {@link MatchEvent} as the class
     * and the event pattern name as the subject.
     * @return pattern feed key.
     */
    @Override
    public final EMessageKey key()
    {
        return (mPubKey);
    } // end of key()

    /**
     * Activates this event pattern feed by putting the
     * subordinate subscription feeds in place.
     * <p>
     * Nothing is done if already subscribed.
     * </p>
     * @throws IllegalStateException
     * if this feed is closed or if the client did not override
     * {@link ESubscriber} methods nor put the required callbacks
     * in place.
     *
     * @see #unsubscribe()
     * @see #close()
     */
    @Override
    public final void subscribe()
    {
        if (!mIsActive.get())
        {
            throw (new IllegalStateException(FEED_IS_INACTIVE));
        }

        // Do nothing if already subscribed.
        if (!mInPlace)
        {
            if (sLogger.isLoggable(Level.FINEST))
            {
                final StringBuilder output = new StringBuilder();

                output.append(mPubKey)
                      .append(" subscribing to ")
                      .append(mSubFeeds.size())
                      .append(":");

                mSubFeeds.forEach(
                    feed ->
                        output.append('\n').append(feed));
            }
            else if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format(
                        "%s: subscribing to %d subordinate feeds.",
                        mPubKey,
                        mSubFeeds.size()));
            }

            // Subscribe to subordinate feeds.
            mSubFeeds.stream()
                     .forEachOrdered(ESubscribeFeed::subscribe);

            // Feed advertised and subordinate subcriptions are
            // now in place.
            mInPlace = true;
        }
    } // end of subscribe()

    @Override
    public final void unsubscribe()
    {
        // Is the feed in place?
        if (mInPlace)
        {
            // Yes.
            if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format("%s subscriber %d, feed %d: unsubscribing.",
                        mEClient.location(),
                        mEClient.clientId(),
                        mFeedId));
            }

            // Retract the subscriptions.
            mSubFeeds.forEach(ESubscribeFeed::unsubscribe);
        }
    } // end of unsubscribe()

    //
    // end of IESubscribeFeed Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Event Handlers.
    //

    /**
     * Updates the feed bit in the subordinate feed mask. If the
     * feed state is up, then the feed bit is set; otherwise the
     * feed bit is cleared. If the composite subordinate feed
     * mask changes state, then this feed posts an update to its
     * publish state.
     * @param state subordinate feed's new state.
     * @param feed subordinate subscription feed.
     */
    protected final void onFeedStateUpdate(final EFeedState state,
                                           final IESubscribeFeed feed)
    {
        final long startMask = mSubFeedMask;
        final int feedId = feed.feedId();
        final long feedMask = (1L << feedId);
        boolean postUpdate = false;

        // Up or down?
        if (state == EFeedState.UP)
        {
            // Up. Set the feed's bit.
            mSubFeedMask |= feedMask;
        }
        // Down or unknown. Clear the feed's bit.
        else
        {
            mSubFeedMask &= ~(feedMask);
        }

        // Did the subordinate feed mask change?
        if (startMask == mSubFeedMask)
        {
            // No. no-op.
        }
        // Yes, it did change. Is the feed now completely up?
        else if (mSubFeedMask == mAllSubFeedsMask)
        {
            // Yes. Let the world know that this feed will now
            // start matching events.
            mFeedState = EFeedState.UP;
            postUpdate = true;
        }
        // Was the feed completely up?
        else if (startMask == mAllSubFeedsMask)
        {
            // Yes. Well now it is down.
            mFeedState = EFeedState.DOWN;
            postUpdate = true;
        }

        if (postUpdate)
        {
            EClient.dispatch(
                new StatusTask<>(
                    mFeedState, this, mStatusCallback),
                mEClient.target());
        }
    } // end of onFeedStateUpdate(EFeedState, ESubscribeFeed)

    /**
     * Runs {@code event} through the state machine if-and-only-if
     * all the subordinate feeds are up.
     * @param event inbound notification event.
     * @param feed event is from this subordinate feed.
     */
    protected final void onEvent(final ENotificationMessage event,
                                 final IESubscribeFeed feed)
    {
        // Are all the subordinate feeds up?
        if (mSubFeedMask == mAllSubFeedsMask)
        {
            // Yes. Go head and run this event through the state
            // machine.
            matchEvent(event, feed.feedId());
        }
        // No, not all the subordinate feeds are up.
        // Ignore events until this happens.
    } // end of onEvent(ENotificationMessage, ESubscribeFeed)

    //
    // end of Event Handlers.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if this is an exclusive pattern feed.
     * Exclusive match events do not share events between
     * instances of the same pattern feed.
     * <p>
     * Events may be shared between different pattern feeds.
     * </p>
     * @return {@code true} if the match pattern is exclusive.
     */
    public final boolean isExclusive()
    {
        return (mIsExclusive);
    } // end of isExclusive()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Adds a mapping from the given event to the match frame
     * which now contains the event. Note: this method assumes
     * that the caller previously determined this is an exclusive
     * pattern feed.
     * @param event event added to the given match frame.
     * @param mf match frame containing the event.
     */
    protected final void addMapping(final ENotificationMessage event,
                                    final AbstractMatchFrame mf)
    {
        List<AbstractMatchFrame> frames = mEventFrames.get(event);

        if (frames == null)
        {
            frames = new ArrayList<>();
            mEventFrames.put(event, frames);
        }

        frames.add(mf);
    } // end of addMapping(ENotification, AbstractMatchFrame)

    /**
     * Marks all of the match frames associated with the given
     * events as defunct. Note: this method assumes that the
     * caller previously determined this is an exclusive
     * pattern feed. Also removes each of the listed events from
     * the event-to-match frame map.
     * <p>
     * Marking a match frame as defunct allows frames to be
     * lazily removed from the frames list when encountered
     * during the match process.
     * </p>
     * @param events the events comprising a completely matched,
     * exclusive pattern.
     */
    protected final void markDefunct(final List<ENotificationMessage> events)
    {
        List<AbstractMatchFrame> frames;

        for (ENotificationMessage event : events)
        {
            if ((frames = mEventFrames.remove(event)) != null)
            {
                for (AbstractMatchFrame frame : frames)
                {
                    frame.setDefunct();
                }
            }
        }
    } // end of markDefunct(List<>)

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    protected static boolean componentTest(final ENotificationMessage e,
                                           final AbstractMatchFrame mf,
                                           final MatchCondition mc)
    {
        boolean retcode;

        try
        {
            final Map<String, List<ENotificationMessage>> groups =
                mf.groupMap();
            final Map<Object, Object> userCache = mf.userCache();

            retcode = mc.test(e, groups, userCache);
        }
        catch (Throwable tex)
        {
            sLogger.log(Level.WARNING,
                        NOTIFY_METHOD,
                        String.format(
                            "%s match condition exception",
                            mf.mPatternName));

            retcode = false;
        }

        return (retcode);
    } // end of componentTest(ENotificationMessage, MatchFrame)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Based class for pattern feed builders. Provides methods
     * for setting the event pattern, status and notify
     * callbacks. If status and/or notify callback methods are
     * <em>not</em> provided, then interface methods
     * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}
     * and
     * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}
     * are used.
     *
     * @param <F> leaf pattern feed class.
     * @param <B> leaf pattern feed builder class.
     */
    public abstract static class PatternBuilder<F extends EPatternFeed,
                                                B extends PatternBuilder<F, ?>>
        extends EFeed.Builder<F, ESubscriber, B>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Event pattern.
         */
        protected EventPattern mPattern;

        /**
         * Feed status callback. If not explicitly set by client,
         * then defaults to
         * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}.
         */
        protected FeedStatusCallback<IESubscribeFeed> mStatusCallback;

        /**
         * Notification message callback. If not explicity set by
         * client, then defaults to
         * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}.
         */
        protected NotifyCallback mNotifyCallback;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        protected PatternBuilder(final Class<F> tc)
        {
            super (tc);

            this.mScope = FeedScope.LOCAL_ONLY;
        } // end of PatternBuilder(Class)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets event pattern used to match against notification
         * stream.
         * @param pattern event pattern.
         * @return {@code this} builder instance.
         * @throws NullPointerException
         * if {@code pattern} is {@code null}.
         */
        public final B eventPattern(final EventPattern pattern)
        {
            Objects.requireNonNull(pattern, "pattern is null");

            mPattern = pattern;

            return (self());
        } // end of eventPattern(EventPattern)

        /**
         * Puts the subscribe feed status callback in place. If
         * {@code cb} is not {@code null}, subscribe status
         * updates will be passed to {@code cb} rather than
         * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}.
         * The reverse is true, if {@code cb} is {@code null},
         * then updates are posted to the
         * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}
         * override.
         * <p>
         * An example using this method is:
         * </p>
         * <pre><code>statusCallback(
    (fs, f) &rarr;
    {
        if (fs == EFeedState.DOWN) {
            <strong style="color:ForestGreen">// Clean up in-progress work.</strong>
        }
    }</code></pre>
         * @param cb subscribe status update callback. May be
         * {@code null}.
         * @return {@code this Builder} instance.
         */
        public final B statusCallback(final FeedStatusCallback<IESubscribeFeed> cb)
        {
            mStatusCallback = cb;

            return (self());
        } // end of statusCallback(FeedStatusCallback<>)

        /**
         * Puts the notification message callback in place. If
         * {@code cb} is not {@code null}, then notification
         * messages will be passed to {@code cb} rather than
         * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}.
         * A {@code null cb} means that notification messages will be
         * passed to the
         * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}
         * override.
         * <p>
         * An example showing how to use this method is:
         * {@code notifyCallback(this::locationUpdate)}
         * </p>
         * @param cb pass notification messages back to target
         * via this callback.
         * @return {@code this Builder} instance.
         */
        public final B notifyCallback(final NotifyCallback cb)
        {
            mNotifyCallback = cb;

            return (self());
        } // end of notifyCallback(NotifyCallback)

        /**
         * Throws {@code OperationNotSupportedException} because
         * pattern feed is local only and may be be changed.
         * @param scope ignored.
         * @return does not return.
         * @throws UnsupportedOperationException
         * because pattern feed scope may not be changed.
         */
        @Override
        public final B scope(final FeedScope scope)
        {
            throw (
                new UnsupportedOperationException(
                    "pattern feed scope is local only and may not be changed"));
        } // end of scope(FeedScope)

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Validates builder settings. Missing or invalid
         * settings are placed into {@code problems} which
         * results in a {@link ValidationException} thrown when
         * pattern feed build is attempted.
         * @param problems report invalid settings in this
         * validator.
         * @return {@code problems}.
         */
        @Override
        public Validator validate(final Validator problems)
        {
            // If the status and notify callbacks are not
            // set, then use the interface methods if defined.
            // Did the subscriber override feedStatus?
            if (mStatusCallback == null &&
                mTarget != null &&
                isOverridden(FEED_STATUS_METHOD,
                             EFeedState.class,
                             IESubscribeFeed.class))
            {
                // Yes. Use the override method.
                mStatusCallback =
                    ((ESubscriber) mTarget)::feedStatus;
            }

            // Did the subscriber override notify?
            if (mNotifyCallback == null &&
                mTarget != null &&
                isOverridden(NOTIFY_METHOD,
                             ENotificationMessage.class,
                             IESubscribeFeed.class))
            {
                // Yes. Use the override method.
                mNotifyCallback =
                    ((ESubscriber) mTarget)::notify;
            }

            return (super.validate(problems)
                         .requireNotNull(mPattern, "pattern")
                         .requireNotNull(mStatusCallback,
                                         "statusCallback")
                         .requireNotNull(mNotifyCallback,
                                         "notifyCallback"));
        } // end of validate(Validator)
    } // end of class PatternBuilder<T extends EPatternFeed

    /**
     * A match frame tracks an in-progress match. When a frame
     * contains a complete pattern match the frame is used to
     * generate the {@link MatchEvent} posted to the subscriber.
     * Match frames are kept until either an event violates the
     * pattern or the match completes.
     * <p>
     * Base class for {@code EOrderPatternFeed.MatchFrame} and
     * {@code EUnorderedPatternFeed.MatchFrame}.
     * </p>
     */
    protected abstract static class AbstractMatchFrame
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Match frame created by this pattern. Used for logging
         * purposes.
         */
        protected final String mPatternName;

        /**
         * Contains all events matching the pattern so far.
         * Since these events are repeatedly referenced by the
         * matching algorithm, this list is kept for quick
         * reference.
         */
        protected final List<ENotificationMessage> mAllEvents;

        /**
         * Cache for storing interim, user-defined data
         * calculated while performing a {@link MatchCondition}.
         */
        protected final Map<Object, Object> mUserCache;

        /**
         * {@code true} if this match frame is for an exclusive
         * pattern. That means events may not be used for
         * multiple matches within the same pattern.
         */
        protected final boolean mIsExclusive;

        /**
         * Set to {@code true} when a match frame is 1) used for
         * an exclusive pattern, 2) the pattern is completely
         * matched, and 3) this frame contains common events with
         * the completely matched frame. Otherwise, will be
         * {@code false}. Marking a match frame as defunct allows
         * for "lazy removal" from the frames queue.
         */
        protected boolean mIsDefunct;

        /**
         * Every match frame has a unique identifier. Used for
         * logging purposes.
         */
        private final int mFrameId;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new match frame for the given pattern and
         * exclusivity flag.
         * @param patternName pattern name.
         * @param isExclusive {@code true} if the pattern is
         * exclusive.
         */
        protected AbstractMatchFrame(final String patternName,
                                     final boolean isExclusive)
        {
            mPatternName = patternName;
            mAllEvents = new ArrayList<>();
            mUserCache = new HashMap<>();
            mIsExclusive = isExclusive;
            mIsDefunct = false;
            mFrameId = nextFrameIdentifier();
        } // end of AbstractMatchFrame(String, boolean)

        /**
         * Copies the given frame data into this frame.
         * @param frame copy this frame.
         */
        protected AbstractMatchFrame(final AbstractMatchFrame frame)
        {
            mPatternName = frame.mPatternName;
            mIsExclusive = frame.mIsExclusive;
            mAllEvents = new ArrayList<>(frame.mAllEvents);
            mUserCache = new HashMap<>(frame.mUserCache);
            mIsDefunct = false;
            mFrameId = nextFrameIdentifier();
        } // end of AbstractMatchFrame(AbstractMatchFrame)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Declarations.
        //

        /**
         * Returns a <em>read-only copy</em> of the group map.
         * This is necessary since the returned map is passed to
         * the client-defined match condition and the client is
         * not allowed to modify this map.
         * @return group map read-only copy.
         */
        protected abstract Map<String, List<ENotificationMessage>> groupMap();

        //
        // end of Abstract Method Declarations.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        @Override
        public String toString()
        {
            int i = 0;
            final StringBuilder retval = new StringBuilder();

            retval.append(mPatternName)
                  .append(' ')
                  .append(mFrameId)
                  .append("\nEvents:");

            for (ENotificationMessage event : mAllEvents)
            {
                retval.append("\n  [")
                      .append(i)
                      .append("] ")
                      .append(event);
                ++i;
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the pattern name.
         * @return pattern name.
         */
        public final String patternName()
        {
            return (mPatternName);
        } // end of patternName()

        /**
         * Returns the unique match frame identifier.
         * @return match frame identifier.
         */
        public final int frameId()
        {
            return (mFrameId);
        } // end of frameId()

        /**
         * Returns {@code true} if this is an exclusive feed;
         * {@code false} otherwise.
         * @return {@code true} if this is an exclusive feed;
         */
        public final boolean isExclusive()
        {
            return (mIsExclusive);
        } // end of isExclusive()

        /**
         * Returns the "all events" collection as an unmodifiable
         * list.
         * @return unmodifiable all events list.
         */
        public final List<ENotificationMessage> allEvents()
        {
            return (Collections.unmodifiableList(mAllEvents));
        } // end of allEvents();

        /**
         * Returns the user-defined match data cache.
         * @return user match data cache.
         */
        public final Map<Object, Object> userCache()
        {
            return (mUserCache);
        } // end of userCache()

        /**
         * Returns {@code true} if this match frame is defunct
         * due to a prior exclusive matching and, therefore, may
         * no longer be used.
         * @return {@code true} if defunct.
         */
        public final boolean isDefunct()
        {
            return (mIsDefunct);
        } // end of isDefunct()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Marks this match frame as defunct. Next time this
         * frame is removed from the frame queue, it will be
         * ignored.
         */
        public final void setDefunct()
        {
            mIsDefunct = true;
        } // end of setDefunct()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Returns the next unique match frame identifier.
         * @return match frame identifier.
         */
        private static int nextFrameIdentifier()
        {
            int retval = sNextFrameId;

            ++sNextFrameId;

            return (retval);
        } // end of nextFrameIdentifier()
    } // end of class AbstractMatchFrame
} // end of class EPatternFeed
