//
// Copyright 2023 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic.store;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.Objects;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.eBus.feed.historic.EHistoricSubscribeFeed.PastComparator;
import net.sf.eBus.feed.historic.IEMessageStore;
import net.sf.eBus.feed.historic.PublishStatusEvent;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.util.ValidationException;
import net.sf.eBus.util.Validator;
import net.sf.eBusx.time.EInterval;

/**
 * This class implements {@link IEMessageStore} providing the
 * ability to store events to and retrieve events from a
 * {@link Connection Java SQL connection}. An application using
 * this class is responsible for:
 * <ul>
 *   <li>
 *     Defining the stored
 *     {@link EMessageKey notification message key}.
 *   </li>
 *   <li>
 *     Providing an {@link IInsertGenerator} instance used to
 *     generate SQL statement for storing eBus notification
 *     message (as defined by message key) or
 *    {@link PublishStatusEvent} into target database.
 *   </li>
 *   <li>
 *     Providing an {@link IRetrieveGenerator} instance used to
 *     generate SQL statement for retrieving eBus notification
 *     messages or {@code PublishStatusEvent} from target
 *     database based on message key and {@link EInterval
 *     interval}.
 *   </li>
 *   <li>
 *     Providing an {@link IMessageDecoder} instance used to
 *     translate {@link ResultSet} into eBus notification
 *     message (either store's target message key or
 *     {@code PublishStatusEvent}).
 *   </li>
 * </ul>
 * <p>
 * <strong>Note:</strong> application is responsible for
 * inserting, retrieving, and decoding both the message type
 * defined by the message key <em>and</em>
 * {@link PublishStatusEvent} messages. Also note that the
 * notification message type must be inserted and retrieved using
 * a single SQL statement. Notification types too complex for
 * this requirement cannot use {@code SqlMessageStore}.
 * </p>
 * <p>
 * There is a one-to-one mapping between an
 * {@code SqlMessageStore} and eBus notification message key.
 * But the application developer is free to design SQL tables in
 * any way deemed best. Most likely this means that there will be
 * a single table for a given eBus notification message class
 * containing the message subject. This allows a single
 * {@code SqlMessageStore} instance to be shared among multiple
 * {@code EHistoricPublishFeed}s. In this case, it is the
 * application's responsibility to provide thread-safe message
 * store and retrieval.
 * </p>
 * <p>
 * Simply put, the application is responsible for interfacing
 * {@code SqlMessageStore} with the application database,
 * translating eBus notification messages between database and
 * Java.
 * </p>
 * <h2>Example</h2>
 * <p>
 * The following code is used to insert and retrieve a
 * {@code TopOfBookMessage} notification into and from an SQL
 * table. Notification message definition (note: see
 * {@link ENotificationMessage} and
 * {@link net.sf.eBus.messages.EMessage} for inherited message
 * fields):
 * </p>
 * <pre><code>public final class TopOfBookMessage extends ENotificationMessage {
    public final PriceSize bid;
    public final PriceSize ask;
    public final PriceType priceType; // enum specifying if this is the opening, closing, low, high, or latest price.

    // Builder class not shown.
}</code></pre>
 * <p>
 * {@code PriceSize} definition:
 * </p>
 * <pre><code>public final class PriceSize extends EField {
    public final Decimal6f price; // uses decimal4j for prices.
    public final int size;
    public final Trend trend; // enum specifying whether this price is up or down in relation to previous message.

    // Builder class not shown.
}</code></pre>
 * <p>
 * The following SQL data definitions used to store the
 * top-of-book message are in Postgresql:
 * </p>
 * <pre><code>CREATE TABLE top_of_book (
  ticker_symbol      varchar(50) NOT NULL,
  publisher_id       bigint NOT NULL,
  publisher_position integer NOT NULL,
  bid                price_size NULL,
  ask                price_size NULL,
  price_type         price_type NOT NULL,
  PRIMARY KEY (
    message_timestamp,
    publisher_id,
    publisher_position
  )
);

-- Table indices not shown.

CREATE TYPE price_Size AS (
  price       NUMERIC(15, 6),
  size        integer,
  price_trend trend
);

CREATE TYPE price_type AS ENUM ( ... );
CREATE TYPE trend AS ENUM ( ... );
</code></pre>
 * <p>
 * The {@link PublishStatusEvent} messages are stored in the
 * table:
 * </p>
 * <pre><code>CREATE TABLE publish_status_event (
  subject            varchar(500) NOT NULL,
  message_timestamp  timestamp NOT NULL,
  publisher_id       bigint NOT NULL,
  publisher_position integer NOT NULL,
  message_key        message_key NOT NULL,
  feed_state         feed_state NOT NULL,
  PRIMARY KEY (
    message_timestamp,
    publisher_id,
    publisher_position
  )
);

-- Table indices not shown.

CREATE TYPE message_key AS (
  message_class   varchar(500),
  message_subject varchar(500)
);

CREATE TYPE feed_state AS ENUM (
  'UNKNOWN',
  'DOWN',
  'UP'
);</code></pre>
 * <p>
 * {@link IInsertGenerator#insertStatement(ENotificationMessage)}
 * implementation returns the following insert statement for
 * a top-of-book message:
 * </p>
 * <p>
 * {@code INSERT INTO top_of_book VALUES ('ACME', '2024-01-06 07:30:53.113', 2001, 1, ROW (12.076400, 1200, 'NA'::trend), ROW (12.076700, 600, 'NA'::trend), 'LATEST'::price_type)}
 * </p>
 * <p>
 * The insert statement for a {@code PublishStatusEvent} is:
 * </p>
 * <p>
 * {@code INSERT INTO publish_status_event VALUES ('/marketdata/ACME', '2024-01-06 07:30:54.547', 2001, 15, ROW ('com.lightspeedmd.com', '/marketdata/ACME', 'DOWN'::feed_state)}
 * </p>
 * <p>
 * {@link IRetrieveGenerator#retrieveStatement(EMessageKey, EInterval)}
 * for top-of-book messages is:
 * </p>
 * <p>
 * {@code SELECT ticker_symbol, message_timestamp, publisher_id, publisher_position, (bid).price, (bid).size, (bid).price_trend, (ask).price, (ask).size, (ask).price_trend, price_type FROM top_of_book WHERE message_timestamp >= '2024-01-06 07:47:43.52010225' AND message_timestamp < '2024-01-06 07:47:44.18792075'}
 * </p>
 * <p>
 * The {@code ResultSet} to top-of-book message code is:
 * </p>
 * <pre><code>// Note: result set columns start at index 1.
final String tickerSymbol = rs.getString(1);
final Instant timestamp = (rs.getTimestamp(2)).toInstant();
final long publisherId = rs.getLong(3);
final int position = rs.getInt(4);
final Decimal6f bidPrice = Decimal6f.valueOf(rs.getBigDecimal(5));
final int bidSize = rs.getInt(6);
final Trend bidTrend = Trend.valueOf(rs.getString(7));
final Decimal6f askPrice = Decimal6f.valueOf(rs.getBigDecimal(8));
final int askSize = rs.getInt(9);
final Trend askTrend = Trend.valueOf(rs.getString(10));
final PriceType priceType = PriceType.valueOf(rs.getString(11));
final PriceSize bid = (PriceSize.builder()).price(bidPrice)
                                           .size(bidSize)
                                           .trend(bidTrend)
                                           .build();
final PriceSize ask = (PriceSize.builder()).price(askPrice)
                                           .size(askSize)
                                           .trend(askTrend)
                                           .build();
final TopOfBookMessage.Builder builder = TopOfBookMessage.builder();

builder.subject(tickerSymbol)
       .timestamp(timestamp)
       .publisherId(publisherId)
       .position(position)
       .bid(bid)
       .ask(ask)
       .priceType(priceType)
       .build());</code></pre>
 *
 * @see InMemoryMessageStore
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class SqlMessageStore
    implements IEMessageStore
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        Logger.getLogger(SqlMessageStore.class.getName());

    /**
     * Used to sort historical messages by timestamp, publisher
     * identifier, and message position.
     */
    private static final PastComparator sPastComparator =
        new PastComparator();

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Database connection used to append and retrieve events.
     */
    private final Connection mDbConnection;

    /**
     * All application eBus notifications stored have this
     * message key.
     */
    private final EMessageKey mKey;

    /**
     * Set to message class {@link PublishStatusEvent} and
     * {@link #mKey target message subject}.
     */
    private final EMessageKey mPublishStatusKey;

    /**
     * Returns SQL statement used to insert an eBus notification
     * message into database.
     */
    private final IInsertGenerator mInsertGenerator;

    /**
     * Returns SQL statement used to retrieve eBus notifications
     * from database for a given time interval
     */
    private final IRetrieveGenerator mRetrieveGenerator;

    /**
     * Used to convert retrieve result set into target eBus
     * notification message.
     */
    private final IMessageDecoder mMessageDecoder;

    /**
     * Tracks number of events inserted into message store since
     * message store opening.
     */
    private int mInsertCount;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new SQL-based event store using the builder
     * settings.
     * @param builder contains event store settings.
     */
    private SqlMessageStore(final Builder builder)
    {
        mDbConnection = builder.mDbConnection;
        mKey = builder.mKey;
        mPublishStatusKey =
            new EMessageKey(
                PublishStatusEvent.class, mKey.subject());
        mInsertGenerator = builder.mInsertGenerator;
        mRetrieveGenerator = builder.mRetrieveGenerator;
        mMessageDecoder = builder.mMessageDecoder;

        mInsertCount = 0;
    } // end of SqlMessageStore(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // IEMessageStore Interface Impelementation.
    //

    @Override
    public boolean isOpen()
    {
        boolean retcode;

        try
        {
            retcode = !mDbConnection.isClosed();
        }
        catch (SQLException sqlex)
        {
            // If an exception is thrown, then assume connection
            // is not open.
            retcode = false;
        }

        return (retcode);
    } // end of isOpen()

    @Override
    public EMessageKey key()
    {
        return (mKey);
    } // end of key()

    @Override
    public void store(final ENotificationMessage message)
    {
        final String insertSql =
            mInsertGenerator.insertStatement(message);

        try (Statement statement = mDbConnection.createStatement())
        {
            statement.executeUpdate(insertSql);

            ++mInsertCount;
        }
        catch(SQLException jex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "SQL message store insert failed, SQL:%n\"%s\"",
                    insertSql),
                jex);
        }
    } // end of insert(ENotificationMessage)

    @Override
    public Collection<ENotificationMessage> retrieve(final EInterval interval)
    {
        // TODO: Need to update retrieve statement with where
        //       clause generated from subscription condition.
        final String messageSql =
            mRetrieveGenerator.retrieveStatement(mKey, interval);
        final String pseSql =
            mRetrieveGenerator.retrieveStatement(
                mPublishStatusKey, interval);
        int rowIndex = 0;
        final SortedSet<ENotificationMessage> retval =
            new TreeSet<>(sPastComparator);

        // 1. Retrieve the target messages.
        try (Statement statement = mDbConnection.createStatement())
        {
            final ResultSet rs =
                statement.executeQuery(messageSql);

            while (rs.next())
            {
                retval.add(mMessageDecoder.toMessage(mKey, rs));
            }
        }
        catch (SQLException sqlex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "Failed to retrieve messages from store, sql:%n\"%s\"",
                    messageSql),
                sqlex);
        }

        // 2. Retrieve publish status events.
        try (Statement statement = mDbConnection.createStatement())
        {
            final ResultSet rs = statement.executeQuery(pseSql);

            while (rs.next())
            {
                retval.add(
                    mMessageDecoder.toMessage(
                        mPublishStatusKey, rs));
                ++rowIndex;
            }
        }
        catch (SQLException sqlex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "Failed to retrieve messages from store, row index=%,d, sql:%n\"%s\"",
                    rowIndex,
                    pseSql),
                sqlex);
        }

        return (retval);
    } // end of retrieve()

    //
    // end of IEMessageStore Interface Impelementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns number of events inserted into message store since
     * message store opening.
     * @return return event insertion count.
     */
    public int insertCount()
    {
        return (mInsertCount);
    } // end of insertCount()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Returns a new {@link Builder} instance for a given data
     * base connection.
     * @param dbConnection database connection encapsulated by
     * {@code SqlMessageStore}.
     * @return a new {@code Builder} instance.
     * @throws NullPointerException
     * if {@code connection} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code connection} is closed or read-only.
     * @throws SQLException
     * if a database access error occurs when attempting to
     * ascertain whether the connection is closed or read-only.
     */
    public static Builder builder(final Connection dbConnection)
        throws SQLException
    {
        Objects.requireNonNull(
            dbConnection, "DB connection is null");

        if (dbConnection.isClosed())
        {
            throw (
                new IllegalArgumentException(
                    "DB connection is closed"));
        }

        if (dbConnection.isReadOnly())
        {
            throw (
                new IllegalArgumentException(
                    "DB connection is read-only"));
        }

        return (new Builder(dbConnection));
    } // end of builder(Connection)

//---------------------------------------------------------------
// Inner Classes.
//

    /**
     * {@link SqlMessageStore} instances created using a
     * {@code Builder} instance. A {@code Builder} instance is
     * acquired from {@link SqlMessageStore#builder(Connection)}.
     * The following attributes must be set to successfully
     * build an SQL message store:
     * <ul>
     *   <li>
     *     non-{@code null} eBus notification
     *     {@link EMessageKey message key},
     *   </li>
     *   <li>
     *     non-{@code null} SQL insert statement generator,
     *   </li>
     *   <li>
     *     non-{@code null} SQL select statement generator, and
     *   </li>
     *   <li>
     *     non-{@code null} {@code ResultSet}-to-eBus
     *     notification message decoder.
     *   </li>
     * </ul>
     * <p>
     * Once these values are correctly set, the target
     * {@link SqlMessageStore} instance is created by calling
     * {@link Builder#build()}.
     * </p>
     */
    public static final class Builder
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Database connection used to insert and retrieve
         * eBus notifications.
         */
        private final Connection mDbConnection;

        /**
         * Message key of all notification messages stored in
         * this database.
         */
        private EMessageKey mKey;

        /**
         * Generates SQL statement used to insert an eBus
         * notification message into database.
         */
        private IInsertGenerator mInsertGenerator;

        /**
         * Generates SQL statement used to retrieve eBus messages
         * from store.
         */
        private IRetrieveGenerator mRetrieveGenerator;

        /**
         * Used to convert retrieve result set to target eBus
         * notification.
         */
        private IMessageDecoder mMessageDecoder;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder(final Connection dbConnection)
        {
            mDbConnection = dbConnection;
        } // end of Builder(Connection)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets eBus notification message key. All message stored
         * or retrieved from database must have this message key.
         * @param key eBus notification message key.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code key} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code key} is not an eBus
         * {@link ENotificationMessage notification message}.
         */
        public Builder key(final EMessageKey key)
        {
            Objects.requireNonNull(key, "key is null");

            if (!key.isNotification())
            {
                throw (
                    new IllegalArgumentException(
                        key.className() +
                        " is not a notification message class"));
            }

            mKey = key;

            return (this);
        } // end of key(EMessageKey)

        /**
         * Sets SQL insert statement generator used to insert an
         * eBus notification message into database.
         * @param generator SQL insert statement generator.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code generator} is {@code null}.
         */
        public Builder insertGenerator(final IInsertGenerator generator)
        {
            mInsertGenerator =
                Objects.requireNonNull(
                    generator, "generator is null");

            return (this);
        } // end of insertGenerator(IInsertGenerator)

        /**
         * Sets SQL retrieve statement generator used to retrieve
         * eBus notification messages from database for a given
         * message key and {@link EInterval interval}.
         * @param generator SQL select statement generator.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code generator} is {@code null}.
         */
        public Builder retrieveGenerator(final IRetrieveGenerator generator)
        {
            mRetrieveGenerator =
                Objects.requireNonNull(
                    generator, "generator is null");

            return (this);
        } // end of retrieveGenerator(final IRetrieveGenerator generator)

        /**
         * Sets decoder used to translate a result set (with
         * given
         * {@link #key(net.sf.eBus.messages.EMessageKey) message key})
         * into an eBus notification message.
         * @param decoder result set decoder.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code decoder} is {@code null}.
         */
        public Builder messageDecoder(final IMessageDecoder decoder)
        {
            mMessageDecoder =
                Objects.requireNonNull(
                    decoder, "decoder is null");

            return (this);
        } // end of messageDecoder(IMessageDecoder)

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Returns new {@link SqlMessageStore} based on this
         * builder's settings.
         * @return new SQL message store instance.
         * @throws ValidationException
         * if this builder's settings are incomplete or invalid.
         */
        public SqlMessageStore build()
        {
            final Validator problems = new Validator();

            problems.requireNotNull(mKey, "key")
                    .requireNotNull(mInsertGenerator,
                                    "insertGenerator")
                    .requireNotNull(mRetrieveGenerator,
                                    "retrieveGenerator")
                    .requireNotNull(mMessageDecoder,
                                    "messageDecoder")
                    .throwException(ValidationException.class);

            return (new SqlMessageStore(this));
        } // end of build()
    } // end of class Builder
} // end of class SqlMessageStore
