//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import net.sf.eBus.messages.ENotificationMessage;

/**
 * An eBus client implementing {@link IEHistoricSubscriber} may
 * choose to implement this functional interface as a
 * {@code private} method instead of implementing the
 * {@code public}
 * {@link IEHistoricSubscriber#notify(ENotificationMessage, EHistoricSubscribeFeed)}
 * method. This private {@code HistoricNotifyCallback} method is
 * then passed to
 * {@link EHistoricSubscribeFeed.Builder#notifyCallback(HistoricNotifyCallback)}.
 * <p>
 * <strong>Note:</strong> this callback may only be set when
 * building an {@code EHistoricSubscribeFeed} instance. Once
 * built, the notification callback target may
 * <em>not</em> be changed.
 * </p>
 * <p>
 * <strong>Note:</strong> failure to implement
 * {@code IEHistoricSubscriber.notify} or to pass a
 * {@code HistoricNotifyCallback} method to
 * {@code Builder.notifyCallback} results in a
 * {@code Builder.build()} failure.
 * </p>
 *
 * @see EHistoricSubscribeFeed
 * @see HistoricFeedStatusCallback
 * @see HistoricFeedDoneCallback
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@FunctionalInterface
public interface HistoricNotifyCallback
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * An incoming notification message from an historic
     * notification feed.
     * @param msg eBus notification message.
     * @param feed message from this historic notification feed.
     */
    void call(ENotificationMessage msg,
              EHistoricSubscribeFeed feed);
} // end of interface HistoricNotifyCallback

