//
// Copyright 2018 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.pattern;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.logging.Level;
import net.sf.eBus.client.EClient;
import net.sf.eBus.client.ESubscribeFeed;
import static net.sf.eBus.feed.pattern.EventPattern.ALL_EVENTS;
import net.sf.eBus.feed.pattern.EventPattern.FeedInfo;
import net.sf.eBus.feed.pattern.EventPattern.PatternComponent;
import net.sf.eBus.feed.pattern.EventPattern.SinglePatternComponent;
import net.sf.eBus.messages.ENotificationMessage;

/**
 * Unordered patterns do not require that events arrive in a
 * specified order but only that the minimum and maximum event
 * quantities are met.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

/* package */ final class EUnorderedPatternFeed
    extends EPatternFeed
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Unordered pattern used to generate new match frames.
     */
    private final ArgsInfo[] mArgs;

    /**
     * The currently active match frames. Contains the collected
     * events for each match group. This list will be replaced
     * each time a new event is received.
     * <p>
     * Note: this data member may <em>not</em> be moved up to
     * super class because {@code MatchFrame} is a
     * {@code EUnorderedPatternFeed} inner class.
     * </p>
     */
    private Queue<MatchFrame> mMatchFrames;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new unordered pattern feed instance for the
     * given publish feed subject and subscriber client.
     * @param builder contains unordered patter feed settings.
     */
    public EUnorderedPatternFeed(final Builder builder)
    {
        super (builder);

        mArgs =
            new ArgsInfo[(builder.mPattern).componentCount()];
        mMatchFrames = new LinkedList<>();

        // Create the subcribe feeds based on the pattern
        // parameters.
        setSubscribeFeeds(builder.mPattern);
    } // end of EUnorderedPatternFeed(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Overrides.
    //

    /**
     * Applies {@code event} to the pattern arguments list.
     * @param event event applied to collected events list.
     * @param eventId event identifier used as the implied
     * capturing group index.
     */
    @Override
    protected void matchEvent(final ENotificationMessage event,
                              final int eventId)
    {
        // 1. Start with a new match frame list.
        final Queue<MatchFrame> frames = new LinkedList<>();
        MatchFrame currFrame;

        if (sLogger.isLoggable(Level.FINEST))
        {
            sLogger.finest(
                String.format(
                    "%s: received event:%n%s",
                    mPubKey,
                    event));
        }
        else
        {
            sLogger.finer(
                String.format(
                    "%s: received %s event.",
                    mPubKey,
                    event.key()));
        }

        // 2. Create a new match frame which begins life in
        //    the start state. This is done in case the event
        //    is the start of the pattern.
        mMatchFrames.add(
            new MatchFrame(mPatternName, mIsExclusive, mArgs));

        // 3. For each existing match frame ...
        while (!mMatchFrames.isEmpty())
        {
            // 4 ... take the next frame and apply the event to
            //   the frame state.
            currFrame = mMatchFrames.poll();

            // Ignore defunct match frames.
            if (currFrame.isDefunct())
            {
                // no-op.
            }
            // 5. Is this match frame's duration still within the
            //    event pattern "until" duration?
            else if (mUntil.test(currFrame.allEvents(), event))
            {
                // Yes. Continue processing the match frame.
                matchEvent(event, eventId, currFrame, frames);
            }
            // No. Put the unmatched frame back on the queue
            // so the next event can be applied to it.
            else
            {
                frames.add(currFrame);
            }
        }

        // 11. Replace the current match frame queue with the
        //     newly generated frame queue.
        mMatchFrames = frames;
    } // end of matchEvent(ENotificationMessage, int)

    //
    // end of Abstract Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new {@code Builder} instance used to construct
     * an unordered pattern beed.
     * @return new unordered pattern feed {@code Builder}
     * instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

    /**
     * Creates subscribe feed for each of pattern's underlying
     * components.
     * @param pattern create subscribe feeds for this pattern's
     * components.
     */
    private void setSubscribeFeeds(final EventPattern pattern)
    {
        final Map<String, FeedInfo> params = pattern.parameters();
        final List<PatternComponent> components =
            pattern.components();
        SinglePatternComponent spc;
        FeedInfo feedInfo;
        ESubscribeFeed subFeed;
        int feedId;

        for (PatternComponent pc : components)
        {
            spc = (SinglePatternComponent) pc;
            feedInfo = params.get(spc.parameter());
            subFeed =
                (ESubscribeFeed.builder())
                    .target(this)
                    .messageKey(feedInfo.messageKey())
                    .scope(feedInfo.scope())
                    .condition(feedInfo.condition())
                    .statusCallback(this::onFeedStateUpdate)
                    .notifyCallback(this::onEvent)
                    .build();
            feedId = subFeed.feedId();

            mSubFeeds.add(subFeed);
            mAllSubFeedsMask |= (1L << feedId);

            mArgs[feedId] = new ArgsInfo(feedId, spc);
        }
    } // end of setSubscribeFeeds(EventPattern)

    @SuppressWarnings({"java:S3776"})
    private void matchEvent(final ENotificationMessage event,
                            final int eventId,
                            final MatchFrame mf,
                            final Queue<MatchFrame> frames)
    {
        final MatchFrame originalMF = new MatchFrame(mf);
        boolean addFlag;
        boolean matchFlag = false;

        if (sLogger.isLoggable(Level.FINEST))
        {
            sLogger.finest(
                String.format(
                    "%s: event ID=%d%n%s%nframe=%s",
                    mPatternName,
                    eventId,
                    event,
                    mf));
        }

        // 6. Was event successfully added to the frame?
        //    Is this frame matched?
        addFlag = mf.addEvent(eventId, event);
        if (addFlag)
        {
            if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format(
                        "%s: %s event added, frame:%n%s",
                        mPubKey,
                        event.key(),
                        mf));
            }

            // If this is an exclusive pattern, then add the
            // event-to-match frame mapping.
            if (mIsExclusive)
            {
                addMapping(event, mf);
            }

            matchFlag = mf.isMatched();
            if (matchFlag)
            {
                // 7. Yes. Generate the match event and see if
                //    the match condition is satisfied.
                final MatchEvent me =
                    mf.generateMatch(mPubKey.subject());

                if (mCondition.test(me))
                {
                    // 8. Yes. Post the match event.
                    EClient.dispatch(
                        new NotifyTask(
                            me,
                            NO_CONDITION,
                            this,
                            mNotifyCallback),
                        mEClient.target());

                    // Is this an exclusive event pattern?
                    if (mIsExclusive)
                    {
                        // Yes. Throw away all in progress match
                        // frames and start from scratch with the
                        // next event.
                        markDefunct(mf.allEvents());
                    }
                }
            }
        }

        // 9. Add this match frame back on to the list if:
        // 1. The match frame is not empty AND
        // 2. Event not added to frame OR
        // 3. The frame is not matched yet OR
        // 4. The frame matched AND
        //    the pattern is *not* exclusive AND
        //    the frame is *not* maxed out.
        if (!mf.isEmpty() &&
            (!addFlag ||
             !matchFlag ||
             (!mIsExclusive && !mf.isMaxedOut())))
        {
            if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format(
                        "Frame %s %d enqueued.",
                        mf.patternName(),
                        mf.frameId()));
            }

            frames.add(mf);
        }

        // 10. If the event was:
        // 1. Added to the match frame AND
        // 2. The original match frame is not empty AND
        // 3. was not matched OR
        // 4. original match frame is not defunct,
        // then put the original, pre-update match frame back on
        // the queue.
        if (addFlag &&
            !originalMF.isEmpty() &&
            (!matchFlag || !originalMF.isDefunct()))
        {
            if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format(
                        "Frame %s %d enqueued.",
                        originalMF.patternName(),
                        originalMF.frameId()));
            }

            frames.add(originalMF);
        }
    } // end of matchEvent(...)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Builder class used to construct an unordered pattern feed.
     * Used to set event pattern, feed status, and notify
     * callbacks.
     *
     * @see EUnorderedPatternFeed#builder()
     */
    public static final class Builder
        extends PatternBuilder<EUnorderedPatternFeed, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * {@code EUnorderPatternFeed} instance builder.
         */
        private Builder()
        {
            super (EUnorderedPatternFeed.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Implementatins.
        //

        /**
         * Returns {@code this} reference.
         * @return {@code this} reference.
         */
        @Override
        protected Builder self()
        {
            return (this);
        } // end of self()

        /**
         * Returns a new unordered pattern feed instance based
         * on this builder's settings.
         * @return new unordered pattern feed.
         */
        @Override
        protected EUnorderedPatternFeed buildImpl()
        {
            return (new EUnorderedPatternFeed(this));
        } // end of buildImpl()

        //
        // end of Abstract Method Implementations.
        //-------------------------------------------------------
    } // end of class Builder

    /**
     * Contains the information needed to construct a
     * {@link PatternArguments} object.
     */
    private final class ArgsInfo
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private final int mEventId;
        private final String mEventName;
        private final int mMinQuantity;
        private final int mMaxQuantity;
        private final MatchCondition mCondition;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private ArgsInfo(final int eventId,
                         final SinglePatternComponent spc)
        {
            mEventId = eventId;
            mEventName = spc.parameter();
            mMinQuantity = spc.minimumMatchCount();
            mMaxQuantity = spc.maximumMatchCount();
            mCondition = spc.condition();
        } // end of ArgsInfo(int, SinglePatternComponent)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        public int eventId()
        {
            return (mEventId);
        } // end of eventId()

        public String eventName()
        {
            return (mEventName);
        } // end of eventName()

        public int minimumQuantity()
        {
            return (mMinQuantity);
        } // end of minimumQuantity()

        public int maximumQuantity()
        {
            return (mMaxQuantity);
        } // end of maximumQuantity()

        public MatchCondition condition()
        {
            return (mCondition);
        } // end of condition()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of ArgInfo

    /**
     * Collects the events for a single subscription, tracking
     * the minimum and maximum event quantities.
     */
    private static final class PatternArguments
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The pattern parameter name for this event.
         */
        private final String mEventName;

        /**
         * The event feed identifier.
         */
        private final int mEventId;

        /**
         * This many events must be collected in order to
         * consider the pattern matched.
         */
        private final int mMinQuantity;

        /**
         * The maximum number of collected events allowed by the
         * pattern. When this value is exceeded, the pattern is
         * no longer matched.
         */
        private final int mMaxQuantity;

        /**
         * Event must satisfy this condition in order to be
         * accepted.
         */
        private final MatchCondition mCondition;

        /**
         * The collected events. List size must be &le;
         * {@link #mMaxQuantity}.
         */
        private final List<ENotificationMessage> mEvents;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new pattern arguments list based on the
         * given pattern component.
         * @param info pattern arguments information.
         */
        private PatternArguments(final ArgsInfo info)
        {
            this (info.eventName(),
                  info.eventId(),
                  info.minimumQuantity(),
                  info.maximumQuantity(),
                  info.condition(),
                  new ArrayList<>(info.maximumQuantity()));
        } // end of PatternArguments(ArgsInfo)

        /**
         * Creates a new pattern arguments list based on the
         * given parameters.
         * @param eventName collecting events of this name.
         * @param eventId unique event identifier.
         * @param minQty minimum number of events to be collected.
         * @param maxQty maximum number of events collected.
         * @param condition match condition.
         * @param events collected events list.
         */
        private PatternArguments(final String eventName,
                                 final int eventId,
                                 final int minQty,
                                 final int maxQty,
                                 final MatchCondition condition,
                                 final List<ENotificationMessage> events)
        {
            mEventName = eventName;
            mEventId = eventId;
            mMinQuantity = minQty;
            mMaxQuantity = maxQty;
            mCondition = condition;
            mEvents = events;
        } // end of PatternArguments(...)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the unique event name.
         * @return event name.
         */
        public String eventName()
        {
            return (mEventName);
        } // end of eventName()

        /**
         * Returns the unique event feed identifier.
         * @return feed identifier.
         */
        public int eventId()
        {
            return (mEventId);
        } // end of eventId()

        /**
         * Returns the number of collected events.
         * @return collected event count.
         */
        public int eventCount()
        {
            return (mEvents.size());
        } // end of eventCount()

        /**
         * Returns {@code true} if the collected events is &ge;
         * to the minimum event quantity; {@code false} if
         * collected events is &lt; minimum quantity.
         * @return {@code true} if the minimum event quantity
         * is reached.
         */
        public boolean atMinimum()
        {
            return (mEvents.size() >= mMinQuantity);
        } // end of atMinimum()

        /**
         * Returns {@code true} if the number of collected events
         * equals the maximum event count; otherwise returns
         * {@code false}.
         * @return {@code true} if the maximum event count is
         * reached.
         */
        public boolean atMaximum()
        {
            return (mEvents.size() == mMaxQuantity);
        } // end of atMaximum()

        /**
         * Returns the collected events list.
         * @return collected events.
         */
        public List<ENotificationMessage> events()
        {
            return (mEvents);
        } // end of events()

        /**
         * Returns a deep copy of {@code this PatternArguments}
         * object.
         * @return pattern arguments deep copy.
         */
        public PatternArguments copy()
        {
            final List<ENotificationMessage> events =
                new ArrayList<>(mMaxQuantity);

            events.addAll(mEvents);

            return (new PatternArguments(mEventName,
                                         mEventId,
                                         mMinQuantity,
                                         mMaxQuantity,
                                         mCondition,
                                         events));
        } // end of copy()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Returns {@code true} if {@code event} is successfully
         * added to collection; {@code false} if the collection
         * is at the maximum or the event does not satisfy the
         * match condition. An event is added to the collection
         * if it is not already in the collection and the
         * collection size has not met its maximum allowed size.
         * @param event add this event to collection.
         * @param groups collected group map.
         * @return {@code true} if {@code event} added to
         * collection.
         */
        public boolean addEvent(final ENotificationMessage event,
                                final MatchFrame mf)
        {
            final boolean retcode =
                (!mEvents.contains(event) &&
                 !atMaximum() &&
                 componentTest(event, mf, mCondition));

            if (retcode)
            {
                mEvents.add(event);
            }

            return (retcode);
        } // end of addEvent(ENotificationMessage)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class PatternArguments

    /**
     * A match frame contains one {@link PatternArguments} for
     * each pattern component.
     */
    private static final class MatchFrame
        extends AbstractMatchFrame
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Collected event arguments.
         */
        private final PatternArguments[] mArguments;

        /**
         * Set to {@code true} if no events have yet been added
         * to this frame. Set to {@code false} once an event is
         * added.
         */
        private boolean mEmptyFlag;

        /**
         * Set to {@code true} when all pattern arguments are at
         * their minimum count.
         */
        private boolean mMatchFlag;

        /**
         * Set to {@code true} when all pattern arguments are at
         * the maximum collection count.
         */
        private boolean mMaxFlag;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a match frame based on the given event pattern.
         * @param patternName pattern creating this match frame.
         * @param isExclusive flag denoting whether this is an
         * exclusive pattern or not.
         * @param maxEvents maximum number of events collected
         * by the pattern.
         * @param info generate match frame from pattern
         * arguments information..
         */
        private MatchFrame(final String patternName,
                           final boolean isExclusive,
                           final ArgsInfo[] info)
        {
            super (patternName, isExclusive);

            mArguments = new PatternArguments[info.length];
            mEmptyFlag = true;

            // Convert the pattern components into arguments.
            final int numInfo = info.length;
            int i;

            for (i = 0; i < numInfo; ++i)
            {
                mArguments[info[i].eventId()] =
                    new PatternArguments(info[i]);
            }
        } // end of PatternArguments(String, boolean, ArgsInfo[])

        /**
         * Creates a deep copy of the given match frame.
         * @param frame copy this match frame.
         */
        private MatchFrame(final MatchFrame frame)
        {
            super (frame);

            final PatternArguments[] args = frame.mArguments;
            final int numArgs = args.length;
            int i;

            mArguments = new PatternArguments[numArgs];
            mEmptyFlag = frame.mEmptyFlag;

            for (i = 0; i < numArgs; ++i)
            {
                mArguments[i] = args[i].copy();
            }
        } // end of MatchFrame(MatchFrame)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Overrides.
        //

        /**
         * Returns a read-only map containing only the
         * "all events" group list (which is also read-only).
         * @return read-only group map.
         */
        @Override
        protected Map<String, List<ENotificationMessage>> groupMap()
        {
            final int numArgs = mArguments.length;
            int i;
            final Map<String, List<ENotificationMessage>> retval =
                new HashMap<>(numArgs + 1);

            retval.put(ALL_EVENTS,
                       Collections.unmodifiableList(mAllEvents));

            // Generate the group map based on the event name.
            for (i = 0; i < numArgs; ++i)
            {
                retval.put(mArguments[i].eventName(),
                           Collections.unmodifiableList(
                               mArguments[i].events()));
            }

            return (Collections.unmodifiableMap(retval));
        } // end of groupMap()

        //
        // end of Abstract Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns {@code true} if this match frame contains no
         * events.
         * @return {@code true} if empty.
         */
        public final boolean isEmpty()
        {
            return (mEmptyFlag);
        } // end of isEmpty()

        /**
         * Returns {@code true} if all argument collections meet
         * the minimum match count; otherwise returns
         * {@code false} - meaning this match frame is not
         * complete.
         * @return {@code true} if the match frame is complete
         * and a {@link MatchElement} may be generated from this
         * frame.
         */
        public boolean isMatched()
        {
            return (mMatchFlag);
        } // end of isMatched()

        /**
         * Returns {@code true} if all pattern arguments are at
         * their respective maximums. If {@code true} this means
         * this match frame may be discarded since no more events
         * may be added to this frame.
         * @return {@code true} if all pattern arguments are
         * maxed out.
         */
        public boolean isMaxedOut()
        {
            return (mMaxFlag);
        } // end of isMaxOut()

        /**
         * Generates a match event based on this match frames
         * capture groups.
         * @param subject notification message subject.
         * @return match event.
         */
        public MatchEvent generateMatch(final String subject)
        {
            return ((MatchEvent.builder()).subject(subject)
                                          .groups(groupMap())
                                          .userCache(userCache())
                                          .build());
        } // end of generateMatch(String)

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Returns {@code true} if {@code event} is successfully
         * added to the specified event list; {@code false} if
         * either {@code event} did not satisfy the match
         * condition or the argument list is at its maximum and
         * no more events may be added.
         * @param eventId index into pattern arguments array.
         * @param event add this event to pattern arguments.
         * @return {@code true} if {@code event} added to
         * event list.
         */
        public boolean addEvent(final int eventId,
                                final ENotificationMessage event)
        {
            final boolean retcode =
                mArguments[eventId].addEvent(event, this);

            // If this event was added, then check if this frame
            // is now matched.
            if (retcode)
            {
                final int numArgs = mArguments.length;
                int i;

                // This frame is no longer empty.
                mEmptyFlag = false;

                // Store the event into the "all events" list.
                mAllEvents.add(event);

                mMatchFlag = true;
                mMaxFlag = true;

                for (i = 0; i < numArgs; ++i)
                {
                    mMatchFlag =
                        (mMatchFlag && mArguments[i].atMinimum());
                    mMaxFlag =
                        (mMaxFlag && mArguments[i].atMaximum());
                }
            }

            return (retcode);
        } // end of addEvent(int, ENotificationMessage)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class MatchFrame
} // end of class EUnorderedPatternFeed
