//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import net.sf.eBus.client.EFeedState;

/**
 * An eBus client implementing {@link IEHistoricPublisher} may
 * choose to implement this functional interface as a
 * {@code private} method instead of implementing the
 * {@code public}
 * {@link IEHistoricPublisher#publishStatus(EFeedState, EHistoricPublishFeed)}
 * method. This private {@code HistoricPublishStateCallback}
 * method is then passed to
 * {@link EHistoricPublishFeed.Builder#statusCallback(HistoricPublishStatusCallback)}.
 * <p>
 * <strong>Note:</strong> this callback may only be set when
 * building an {@code EHistoricPublishFeed} instance. Once built,
 * the publisher feed status callback target may <em>not</em> be
 * changed.
 * </p>
 * <p>
 * If not overridden or replaced with a
 * {@code HistoricPublishStatusCallback},
 * {@code IEHistoricPublisher.publishStatus(EFeedState, EHistoricPublishFeed)}
 * does nothing.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface HistoricPublishStatusCallback
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * {@link EHistoricPublishFeed} calls this method to inform
     * a publisher that the notification feed state is either
     * {@code EFeedState.UP} or {@code EFeedState.DOWN}. This
     * callback is provided as a courtesy only since historic
     * publishers are always free to publish notification messages
     * on an historic publish feed (as long as
     * {@link EHistoricPublishFeed#isFeedUp()} returns
     * {@code true}).
     * @param feedState {@link EFeedState#UP} if there is at
     * least one subscriber to this notification feed; otherwise
     * set to {@link EFeedState#DOWN}.
     * @param feed historic publish feed reporting this feed
     * state change.
     */
    void call(EFeedState feedState,
              EHistoricPublishFeed feed);
} // end of interface HistoricPublishStatusCallback

