//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.EObject;

/**
 * Classes wanting to send eBus notifications using an
 * {@link EHistoricPublishFeed} need to implement this interface.
 * Note that unlike {@link net.sf.eBus.client.EPublishFeed}, an
 * historic publisher is always free to publish notification
 * messages as long as 1) the historic publish feed is open,
 * 2) the feed is advertised, and 3) the publish feed state is
 * {@link EFeedState#UP up}.
 * <p>
 * An historic publisher is required to provide a unique
 * identifier. This identifier is used order past notifications
 * set to an historic subscriber. It also allows historic
 * subscribers to differentiate between publisher notification
 * streams. {@link PublishStatusEvent} contains this publisher
 * identifier so historic subscriber can determine which
 * publisher's stream is now up or down.
 * </p>
 * <p>
 * See {@link EHistoricPublishFeed} for detailed explanation on
 * how to create and use an historic publish feed.
 * </p>
 *
 * @see EHistoricPublishFeed
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public interface IEHistoricPublisher
    extends EObject
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * Returns historic publisher unique identifier. Note that
     * this uniqueness is <em>not</em> enforced by eBus. Users
     * are encouraged to ensure uniqueness so that publishers
     * may be matched to notification messages.
     * @return historic publisher identifier.
     */
    long publisherId();

    /**
     * eBus is informing the historic publisher that there is at
     * least one active subscriber to the publisher's
     * notification feed. This callback is provided as a courtesy
     * only since historic publishers are always free to publish
     * notification messages on an historic publish feed (as long
     * as {@link EHistoricPublishFeed#isFeedUp()} returns
     * {@code true}).
     * <p>
     * By default this method does nothing.
     * </p>
     * @param feedState {@link EFeedState#UP} if there is at
     * least one subscriber to this notification feed; otherwise
     * set to {@link EFeedState#DOWN}.
     * @param feed historic publish feed reporting this feed
     * state change.
     */
    default void publishStatus(EFeedState feedState,
                               EHistoricPublishFeed feed)
    {}
} // end of interface IEHistoricPublisher

