//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import java.util.Objects;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.util.Validator;

/**
 * Notification event used to report and persist publish feed
 * status updates. This is necessary for historic notification
 * retrieval because knowing when a feed is down impacts
 * statistical analysis.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class PublishStatusEvent
    extends ENotificationMessage
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x060300L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Feed state applies to this notification message key.
     */
    public final EMessageKey key;

    /**
     * Latest publisher feed state.
     */
    public final EFeedState feedState;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new publish status event instance based on the
     * builder's settings.
     * @param builder contains validated publish status event
     * settings.
     */
    private PublishStatusEvent(final Builder builder)
    {
        super (builder);

        key = builder.mKey;
        feedState = builder.mFeedState;
    } // end of PublishStatusEvent(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns textual representation of publish status event.
     * @return publish status event as text.
     */
    @Override
    public String toString()
    {
        final StringBuilder retval = new StringBuilder();

        return (retval.append(super.toString())
                      .append("\n                   key: ")
                      .append(key)
                      .append("\n            feed state: ")
                      .append(feedState)
                      .toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns a new {@code PublishStatusEvent} builder
     * instance. A builder instance should be used to create only
     * one publish status event instance.
     * @return {@code PublishStatusEvent} builder.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Builder class for {@code PublishStatusEvent} message
     * instances. A {@code Builder} instance is acquired using
     * the {@link PublishStatusEvent#builder()} method. Each
     * builder instance should used to build only one
     * {@code PublisherStatusEvent} instance.
     * <p>
     * {@code Builder} requires: publisher identifier be set to
     * an integer values &gt; zero, message key is set to a
     * non-{@code null} notification, and feed state is set to
     * a non-{@code null} value.
     * </p>
     *
     * @see PublishStatusEvent
     */
    public static final class Builder
        extends ENotificationMessage.Builder<PublishStatusEvent,
                                             Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Published notification message key.
         */
        private EMessageKey mKey;

        /**
         * Current publisher feed state.
         */
        private EFeedState mFeedState;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Private constructor since builder instances are
         * created by {@link PublishStatusEvent#builder()}
         * method only.
         */
        private Builder()
        {
            super (PublishStatusEvent.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns a new {@code PublishStatusEvent} instance
         * based on this builder's configuration.
         * @return new {@code PublishStatusEvent} instance.
         */
        @Override
        protected PublishStatusEvent buildImpl()
        {
            return (new PublishStatusEvent(this));
        } // end of buildImpl()

        /**
         * Checks if this builder is correctly configured before
         * attempting to build the target message instance.
         * A valid {@code PublishStatusEvent} requires that
         * publisher identifier, message key and feed state are
         * set. Calls {@code super.validate} to check super class
         * fields.
         * @param problems place invalid fields into this list.
         * @return {@code problems} is returned to all method
         * chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            return (super.validate(problems)
                         .requireTrue(mPubId > NO_PUB_ID,
                                      "publisherId",
                                      "publisherId not set")
                         .requireNotNull(mKey, "key")
                         .requireNotNull(mFeedState, "feedState"));
        } // end of validate(Validator)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets message key to which feed status applies.
         * @param key message key.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code key} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code key} is not for a notification message.
         */
        public Builder key(final EMessageKey key)
        {
            Objects.requireNonNull(key, "message key is null");

            if (!key.isNotification())
            {
                throw (
                    new IllegalArgumentException(
                        "not a notification key"));
            }

            mKey = key;

            return (this);
        } // end of key(EMessageKey)

        /**
         * Sets publisher feed state.
         * @param feedState publisher feed state.
         * @return {@code this Builder} instance.
         */
        public Builder feedState(final EFeedState feedState)
        {
            mFeedState =
                Objects.requireNonNull(
                    feedState, "feed state is null");

            return (this);
        } // end of feedState(EFeedState)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class PublishStatusEvent
