//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import net.sf.eBus.feed.historic.EHistoricSubscribeFeed.HistoricFeedState;

/**
 * An eBus client implementing {@link IEHistoricSubscriber} may
 * choose to implement this functional interface as a
 * {@code private} method instead of implementing the
 * {@code public}
 * {@link IEHistoricSubscriber#feedDone(EHistoricSubscribeFeed.HistoricFeedState, EHistoricSubscribeFeed)}
 * method. This private {@code HistoricFeedDoneCallback} method
 * is then passed to
 * {@link EHistoricSubscribeFeed.Builder#doneCallback(HistoricFeedDoneCallback)}.
 * <p>
 * <strong>Note:</strong> this callback may only be set when
 * building an {@code EHistoricSubscribeFeed} instance. Once
 * built, the feed done callback target may
 * <em>not</em> be changed.
 * </p>
 * <p>
 * <strong>Note:</strong> failure to implement
 * {@code IEHistoricSubscriber.feedDone} or to pass a
 * {@code HistoricFeedDoneCallback} method to
 * {@code Builder.doneCallback} results in a
 * {@code Builder.build()} failure.
 * </p>
 *
 * @see EHistoricSubscribeFeed
 * @see HistoricNotifyCallback
 * @see HistoricFeedStatusCallback
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@FunctionalInterface
public interface HistoricFeedDoneCallback
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * eBus calls this method when an historic subscribe feed
     * has reached completion. The feed state will be either
     * {@link HistoricFeedState#DONE_SUCCESS} or
     * {@link HistoricFeedState#DONE_ERROR}. If the historic feed
     * completion is in error, then
     * {@link EHistoricSubscribeFeed#errorCause()} may be called
     * to retrieve the exception which caused the error.
     * @param feedState historic subscribe feed final state.
     * @param feed historic subscribe feed.
     */
    void call(HistoricFeedState feedState,
              EHistoricSubscribeFeed feed);
} // end of interface HistoricFeedDoneCallback

