//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Timer;
import java.util.concurrent.atomic.AtomicReference;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import net.sf.eBus.client.ECondition;
import net.sf.eBus.client.EFeed;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.ERequestFeed;
import net.sf.eBus.client.ERequestor;
import net.sf.eBus.client.ESubscribeFeed;
import net.sf.eBus.client.ESubscriber;
import net.sf.eBus.client.IERequestFeed;
import net.sf.eBus.client.IESubscribeFeed;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.util.TimerEvent;
import net.sf.eBus.util.TimerTask;
import net.sf.eBus.util.ValidationException;
import net.sf.eBus.util.Validator;
import net.sf.eBusx.time.EInterval;
import net.sf.eBusx.time.EInterval.Clusivity;

/**
 * The historic subscriber feed extends {@link IESubscribeFeed}'s
 * ability to receive live notification messages with the ability
 * to seamlessly retrieve past notification messages from
 * {@link IEHistoricPublisher historic publishers}. The historic
 * subscribe feed accomplishes this merge by:
 * <ol>
 *   <li>
 *     On start-up, subscribing to live notification and
 *     {@link PublishStatusEvent} feeds (the publish status event
 *     reports each individual <em>historic</em> publisher
 *     status).
 *   </li>
 *   <li>
 *     Again on  start-up, {@link HistoricRequest requesting}
 *     previously published notification messages from historic
 *     publishers <em>if</em> historic subscriber requested them.
 *   </li>
 *   <li>
 *     Past notifications contained in
 *     {@link HistoricReply historic replies} and live
 *     notifications are placed into a single list with care
 *     taken to ensure each message appears only once in the
 *     list.
 *   </li>
 *   <li>
 *     Once all the replies are received the received
 *     notification message list is sorted by time (millisecond
 *     granularity), publisher identifier, and publisher-set
 *     message position and delivered to
 *     {@link IEHistoricSubscriber#notify(ENotificationMessage, EHistoricSubscribeFeed)}
 *     one at a time.
 *   </li>
 *   <li>
 *     After this all live notifications are forwarded to the
 *     historic subscriber as the messages arrive.
 *   </li>
 * </ol>
 * <p>
 * Note: an historic subscriber may also receives live
 * notification messages from
 * {@link net.sf.eBus.client.EPublisher EPublisher}s but, of
 * course, none of the previously published messages. Further,
 * an historic subscriber may receive past notification messages
 * from any {@link net.sf.eBus.client.EReplier replier}
 * supporting {@link HistoricRequest}, {@link HistoricReply}
 * messages.
 * </p>
 * <h2>Using Historic Subscribe Feed</h2>
 * <p>
 * Follow these steps for using an historic subscribe feed:
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 1:</strong> Implement
 * {@link IEHistoricSubscriber} interface.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 2:</strong> Use an
 * {@link EHistoricSubscribeFeed.Builder} instance (obtained from
 * {@link EHistoricSubscribeFeed#builder(EMessageKey, IEHistoricSubscriber)})
 * to create an historic subscribe feed for a given message key
 * and historic subscriber instance. An {@link ECondition} may
 * be associated with the feed and only those notification
 * messages matching
 * {@link ECondition#test(net.sf.eBus.messages.EMessage) match}
 * the condition are forwarded to the historic subscriber. If no
 * condition is defined, then all messages are forwarded to the
 * subscriber.
 * </p>
 * <p>
 * Use
 * {@link Builder#doneCallback(HistoricFeedDoneCallback)},
 * {@link Builder#statusCallback(HistoricFeedStatusCallback)}
 * and {@link Builder#notifyCallback(HistoricNotifyCallback)} to
 * set Java lambda expressions used in place of
 * {@link IEHistoricSubscriber} interface methods.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 3:</strong>
 * {@link #startup() Start up} historic subscribe feed. Call this
 * method directly rather than using
 * {@link net.sf.eBus.client.EFeed#register(net.sf.eBus.client.EObject)}
 * and
 * {@link net.sf.eBus.client.EFeed#startup(net.sf.eBus.client.EObject)}
 * because {@code EHistoricSubscribeFeed} is an eBus hybrid
 * object and runs in the {@code IEHistoricSubscriber}'s
 * dispatcher. The underlying live notification and historic
 * notification <strong>request</strong> feeds are created,
 * subscribed, and historic notification request placed at this
 * time.
 * </p>
 * <p>
 * If the historic subscribe feed has fixed, future end time,
 * a timer is set to terminate the feed at the future time.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 4:</strong>
 * {@link #subscribe() Subscribe} to open historic feed.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 5:</strong> Wait for
 * {@link IEHistoricSubscriber#notify(ENotificationMessage, EHistoricSubscribeFeed) notification messages}
 * and
 * {@link IEHistoricSubscriber#feedStatus(PublishStatusEvent, EHistoricSubscribeFeed) publisher status events}
 * to arrive. Note that the historic subscriber receives
 * feed status updates on a <em>per publisher</em> basis rather
 * than for the entire feed. This allows the historic subscriber
 * to know if a particular publisher's notification stream is
 * interrupted.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 6:</strong> When the
 * historic subscribe feed reaches a fixed end point (see more
 * on
 * <a href="#FeedInterval">setting historic subscribe feed intervals</a>
 * below),
 * {@link IEHistoricSubscriber#feedDone(EHistoricSubscribeFeed.HistoricFeedState, EHistoricSubscribeFeed)}
 * is called to let the historic subscriber know that the feed
 * is ended and no more callbacks will be issued. This is not
 * the case if the historic subscribe feed end time is set to
 * on-going.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 7:</strong> When the
 * historic subscriber is shutting down,
 * {@link #shutdown() shutdown} the feed.
 * </p>
 * <h2>Example use of <code>EHistoricSubscribeFeed</code></h2>
 * <pre><code>import java.time.Instant;
import net.sf.eBus.client.EFeed.FeedScope;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.feed.historic.IEHistoricSubscriber;
import net.sf.eBus.feed.historic.EHistoricSubscribeFeed;
import net.sf.eBus.feed.historic.EHistoricSubscribeFeed.HistoricFeedState;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBusx.time.EInterval.Clusivity;

<strong style="color:ForestGreen">Step 1: Implement the IEHistoricSubscriber interface.</strong>
public class CatalogSubscriber implements IEHistoricSubscriber
{
    // Subscribe to this notification message class/subject key and feed scope.
    private final EMessageKey mKey;
    private final FeedScope mScope;

    // Catalog subscriber name used for logging purposes.
    private final String mName;

    // Request historic messages from this time in the past
    // (inclusive) to a fixed time in the future (exclusive).
    private final Instant mBeginTime;
    private final Instant mEndTime;

    // Store the feed here so it can be used to unsubscribe.
    private EHistoricSubscribeFeed mFeed;

    public CatalogSubscriber(final String subject,
                             final FeedScope scope,
                             final String name,
                             final Instant beginTime,
                             final Instant endTime) {
        mKey = new EMessageKey(CatalogUpdate.class, subject);
        mScope = scope;
        mName = name;
        mBeginTime = beginTime;
        mEndTime = endTime;
        mFeed = null;
    }

    &#64;Override public void startup() {
        try {
            <strong style="color:ForestGreen">Step 2: Open the EHistoricSubscribe feed.</strong>
            final EHistoricSubscribeFeed.Builder builder = EHistoricSubscribeFeed.builder(mKey, this);

            // This subscriber has no associated ECondition and uses IEHistoricSubscriber interface method overrides.
            mFeed = builder.name(mName)
                           .scope(mScope)
                           .from(mBeginTime, Clusivity.INCLUSIVE)
                           .to(mEndTime, Clusivity.EXCLUSIVE)
                           .build();

            <strong style="color:ForestGreen">Step 3: Start the feed.</strong>
            mFeed.startup();

            <strong style="color:ForestGreen">Step 4: Subscribe to the feed.</strong>
            mFeed.subscribe();
        } catch(IllegalArgumentException argex) {
            // Feed open failed. Place recovery code here.
        }
    }

    <strong style="color:ForestGreen">Step 7: When subscriber is shutting down, retract subscription feed.</strong>
    &#64;Override public void shutdown() {
        // mFeed.unsubscribe() is not necessary since close() will unsubscribe.
        if (mFeed != null) {
            mFeed.close();
            mFeed = null;
        }
    }

    <strong style="color:ForestGreen">Step 5: Wait for feed status events and notifications to arrive.</strong>
    &#64;Override public void feedStatus(final PublishStatusEvent msg, final EHistoricSubscribeFeed feed) {
        Publisher feed status handling code here.
    }

    &#64;Override public void notify(final ENotificationMessage msg, final EHistoricSubscribeFeed feed) {
        Notification handling code here.
    }

    <strong style="color:ForestGreen">Step 6: Wait for historic subscribe feed to complete.</strong>
    &#64;Override public void feedDone(final HistoricFeedState feedState, final EHistoricSubscribeFeed feed) {
        Feed completion code here.
    }
}</code></pre>
 * <h1><a id="FeedInterval">Setting historic subscribe feed interval</a></h1>
 * <p>
 * An historic subscribe feed may be set to cover the following
 * time intervals:
 * </p>
 * <ul>
 *   <li>
 *     {@link EHistoricSubscribeFeed.Builder#from(Instant, EInterval.Clusivity)}
 *     and
 *     {@link EHistoricSubscribeFeed.Builder#to(Instant, EInterval.Clusivity) end times}
 *     in the {@link TimeLocation#PAST past}. The historic
 *     subscribe feed only requests past notification messages
 *     for this time interval and does <em>not</em> subscribe to
 *     live notification feeds. Once the feed delivers historic
 *     notifications, the feed terminates, calling
 *     {@code feedDone}. Note that begin time must be prior to
 *     end time.
 *   </li>
 *   <li>
 *     All historic notification messages from
 *     {@link EHistoricSubscribeFeed.Builder#from(Instant, EInterval.Clusivity) begin}
 *      time until
 *     {@link EHistoricSubscribeFeed.Builder#toNow(EInterval.Clusivity) now}.
 *      The historic subscribe feed operates in the same manner
 *      as above when begin and end time are in the past.
 *   </li>
 *   <li>
 *     {@link EHistoricSubscribeFeed.Builder#from(Instant, EInterval.Clusivity) Begin time}
 *     in the
 *     {@link TimeLocation#PAST past} and
 *     {@link EHistoricSubscribeFeed.Builder#to(Instant, EInterval.Clusivity) end time}
 *     in the {@link TimeLocation#FUTURE future}. Historic
 *     subscribe feed requests past notification messages from
 *     begin time until now and subscribes to live notification
 *     feeds. Feed sets a timer expiring at the fixed end time.
 *     When timer expires, calls
 *     {@link IEHistoricSubscriber#feedDone(EHistoricSubscribeFeed.HistoricFeedState, EHistoricSubscribeFeed)}.
 *   </li>
 *   <li>
 *     All live notification messages from
 *     {@link EHistoricSubscribeFeed.Builder#fromNow() now} until
 *     a
 *     {@link EHistoricSubscribeFeed.Builder#to(Instant, EInterval.Clusivity) fixed future end time}.
 *     Historic subscribe feed subscribes to live notification
 *     feeds only. Sets timer to expire when end time is reached.
 *     Calls {@code IEHistoricSubscriber.feedDone} when timer
 *     expires.
 *   </li>
 *   <li>
 *     All live notification messages from
 *     {@link EHistoricSubscribeFeed.Builder#fromNow() now} with
 *     {@link EHistoricSubscribeFeed.Builder#toForever() not fixed end time}.
 *     This is essentially the same as a {@link IESubscribeFeed}
 *     with the exception that feed status updates are per
 *     publisher. Historic subscribe feed does not terminate
 *     until historic subscriber
 *     {@link EHistoricSubscribeFeed#shutdown() shuts down} the
 *     feed.
 *   </li>
 * </ul>
 * <p style="background-color:#ffcccc;padding:5px;border: 2px solid darkred;">
 * When an historic subscriber requests past and live
 * notification messages, the historic subscribe feed
 * <em>attempts</em> to make sure the message stream does not
 * contain redundant or missing messages. But message stream
 * correctness cannot be guaranteed.
 * </p>
 *
 * @see IEHistoricSubscriber
 * @see IESubscribeFeed
 * @see PublishStatusEvent
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@SuppressWarnings ({"java:S1192"})
public final class EHistoricSubscribeFeed
    extends EAbstractHistoricFeed<IEHistoricSubscriber>
    implements ESubscriber,
               ERequestor
{
//---------------------------------------------------------------
// Member enum.
//

    /**
     * Denotes if a timestamp denotes the past, current time,
     * future, or future on-going.
     */
    public enum TimeLocation
    {
        /**
         * Feed begin or end time is in the past.
         */
        PAST (true, false),

        /**
         * Feed begin or end time is the current time. Current
         * time is neither in the past nor the future.
         */
        NOW (false, false),

        /**
         * Feed end time is in the future. Feed begin time may
         * <em>not</em> be in the future.
         */
        FUTURE (false, true),

        /**
         * Feed end time continues into future indefinitely until
         * stopped.
         */
        ON_GOING (false, true);

    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Set to {@code true} if this historic subscription
         * references past notifications.
         */
        private final boolean mIsPast;

        /**
         * Set to {@code true} if this historic subscription
         * references notifications in the future.
         */
        private final boolean mIsFuture;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private TimeLocation(final boolean pastFlag,
                             final boolean futureFlag)
        {
            mIsPast = pastFlag;
            mIsFuture = futureFlag;
        } // end of TimeLocation(boolean, boolean)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns {@code true} if time is located in past.
         * Note that {@link #NOW} is neither in the past or the
         * future.
         * @return {@code true} if time is located in past.
         */
        public boolean isPast()
        {
            return (mIsPast);
        } // end of isPast()

        /**
         * Returns {@code true} if time is located in future.
         * Note that {@link #NOW} is neither in the past or the
         * future.
         * @return {@code true} if time is located in future.
         */
        public boolean isFuture()
        {
            return (mIsFuture);
        } // end of isFuture()

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of enum TimeLocation

    /**
     * Enumerates historic subscribe feed states.
     */
    public enum HistoricFeedState
    {
        /**
         * Historic subscribe feed is open but not yet subscribed
         * to notifications.
         */
        FEED_OPEN,

        /**
         * Feed is collecting past notification message.
         */
        HISTORIC_NOTIFICATIONS,

        /**
         * Feed has collected past notifications and is now
         * processing live notification messages.
         */
        LIVE_NOTIFICATIONS,

        /**
         * Historic subscribe feed successfully finished with
         * both past and live notification messages. If end time
         * location is {@link TimeLocation#ON_GOING}, the feed
         * will never reach this done state.
         */
        DONE_SUCCESS,

        /**
         * Historic subscribe feed is done due to an error. The
         * exception causing this failure may be retrieved by
         * calling .
         */
        DONE_ERROR
    } // end of HistoricFeedState

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * If historic publish feed name is not set, then defaults to
     * {@value} appended with feed index.
     */
    public static final String DEFAULT_SUBSCRIBE_FEED_NAME =
        "EHistoricSubscribeFeed-";

    /**
     * Static timer instance used by historic subscribe feed is
     * {@value}.
     */
    private static final String TIMER_NAME = "HistoricTimer";

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    private static final Logger sLogger =
        Logger.getLogger(EHistoricSubscribeFeed.class.getName());

    /**
     * Timer instance used to set future end times.
     */
    private static final Timer sTimer =
        new Timer(TIMER_NAME, true);

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Forward message to historic subscriber if notification
     * satisfies condition. If condition is not defined, then
     * defaults to {@link net.sf.eBus.client.EFeed#NO_CONDITION}.
     */
    private final ECondition mCondition;

    /**
     * Historic feed status callback. If not explicitly set by
     * client, then defaults to
     * {@link IEHistoricSubscriber#historicFeedStatus(EFeedState, IESubscribeFeed)}.
     */
    private final HistoricFeedDoneCallback mDoneCallback;

    /**
     * Feed status callback. If not explicitly set by client,
     * then defaults to
     * {@link ESubscriber#feedStatus(EFeedState, ESubscribeFeed)}.
     */
    private final HistoricFeedStatusCallback mStatusCallback;

    /**
     * Notification message callback. If not explicitly set by
     * client, then defaults to
     * {@link IEHistoricSubscriber#notify(ENotificationMessage, EHistoricSubscribeFeed)}.
     */
    private final HistoricNotifyCallback mNotifyCallback;

    /**
     * Historic feed may begin either in the past or at the
     * current time but not in the future.
     */
    private final TimeLocation mBeginLocation;

    /**
     * Historic feed begin time. If {@link #mBeginLocation} is
     * {@link TimeLocation#NOW} then will be {@code null}.
     */
    private final Instant mBeginTime;

    /**
     * Historic begin time clusivity. Will be {@code null} if
     * begin time location is now.
     */
    private final Clusivity mBeginClusivity;

    /**
     * Historic feed ends either in the past, now, or the future.
     */
    private final TimeLocation mEndLocation;

    /**
     * Historic feed end time. If {@link #mEndLocation} is
     * {@link TimeLocation#NOW}, then value is set to
     * {@code null}. If end location is
     * {@link TimeLocation#ON_GOING}, then this value is
     * {@link Instant#MAX}.
     */
    private final Instant mEndTime;

    /**
     * Historic end time clusivity. Will be {@code null} if end
     * time location is now or in the future.
     */
    private final Clusivity mEndClusivity;

    /**
     * Current historic subscribe feed state.
     */
    private final AtomicReference<HistoricFeedState> mState;

    /**
     * Contains exception which resulted in {@code mState} being
     * set to {@link HistoricFeedState#DONE_ERROR}.
     */
    private Exception mErrorCause;

    /**
     * Receive live notification messages on this feed.
     */
    private ESubscribeFeed mSubscribeFeed;

    /**
     * Task is scheduled for definite end time.
     */
    private TimerTask mEndTimer;

    /**
     * Receive live publisher feed status messages on this feed.
     */
    private ESubscribeFeed mStatusFeed;

    /**
     * Receive historic notification messages on this feed.
     */
    private ERequestFeed mRequestFeed;

    /**
     * Historic notification request.
     */
    private ERequestFeed.ERequest mRequest;

    /**
     * When this historic subscribe feed is in the
     * {@link HistoricFeedState#HISTORIC_NOTIFICATIONS} state,
     * all past and live notifications are stored in this list
     * for later playback once all past notifications are
     * received.
     */
    private List<ENotificationMessage> mMessages;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new instance of EHistoricSubscribeFeed.
     */
    private EHistoricSubscribeFeed(final Builder builder)
    {
        super (builder);

        mCondition = builder.mCondition;
        mDoneCallback = builder.getDoneCallback();
        mStatusCallback = builder.getStatusCallback();
        mNotifyCallback = builder.getNotifyCallback();
        mBeginLocation = builder.mBeginLocation;
        mBeginTime = builder.mBeginTime;
        mBeginClusivity = builder.mBeginClusivity;
        mEndLocation = builder.mEndLocation;
        mEndTime = builder.mEndTime;
        mEndClusivity = builder.mEndClusivity;

        mState =
            new AtomicReference<>(HistoricFeedState.FEED_OPEN);
    } // end of EHistoricSubscribeFeed(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    /**
     * Closes subordinate notification and request feeds, if
     * open.
     */
    @Override
    protected void doClose()
    {
        // Stop end timer if running.
        if (mEndTimer != null)
        {
            mEndTimer.cancel();
            mEndTimer = null;
        }

        closeFeed(mSubscribeFeed);
        mSubscribeFeed = null;

        closeFeed(mStatusFeed);
        mStatusFeed = null;

        closeFeed(mRequestFeed);
        mRequestFeed = null;

        // Drop all collected messages - if there are any.
        if (mMessages != null)
        {
            mMessages.clear();
            mMessages = null;
        }
    } // end of doClose()

    //
    // end of Abstract Method Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // EObject Interface Implementation.
    //

    /**
     * Returns historic subscribe feed's eBus object name. Used for
     * logging purposes only.
     * @return eBus object name.
     */
    @Override
    public String name()
    {
        return (mName);
    } // end of name()

    /**
     * Places subordinate live notification and publisher
     * status subscriptions (if needed) and requests historic
     * notification messages (again, if needed). If this feed has
     * a fixed, future end time, then sets timer to expire at
     * that end time.
     */
    @Override
    public void startup()
    {
        final ERequestFeed.Builder reqBuilder =
            ERequestFeed.builder();

        if (sLogger.isLoggable(Level.FINE))
        {
            sLogger.fine(String.format("%s: starting.", mName));
        }

        // Does this request feed include future notifications
        // or is it past only?
        if (mEndLocation.isFuture())
        {
            final ESubscribeFeed.Builder subBuilder =
                ESubscribeFeed.builder();
            final ESubscribeFeed.Builder statBuilder =
                ESubscribeFeed.builder();

            // Open subscribe feed to receive future
            // notifications.
            mSubscribeFeed =
                subBuilder.target(this)
                          .messageKey(mKey)
                          .scope(mScope)
                          .statusCallback(this::liveStatus)
                          .notifyCallback(this::liveNotification)
                          .build();

            // Open and subscribe to historic notification feed
            // status.
            mStatusFeed =
                statBuilder.target(this)
                           .messageKey(mStatusKey)
                           .scope(mScope)
                           .statusCallback(this::liveStatus)
                           .notifyCallback(this::livePublisherStatus)
                           .build();

            // Is the end time definite?
            if (mEndLocation == TimeLocation.FUTURE)
            {
                // Yes. Schedule a timer to expire at the end
                // time taking end time clusivity into account.
                mEndTimer =
                    scheduleEndTimer(mEndTime, mEndClusivity);
            }
        }
        // Else this feed is for past notifications only.

        // Does this request feed include historic notifications?
        if (mBeginLocation.isPast())
        {
            // Open request feed to receive historic
            // notifications.
            if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format(
                        "%s: opening %s request feed.",
                        mName,
                        mRequestKey));
            }

            mRequestFeed =
                reqBuilder.target(this)
                          .messageKey(mRequestKey)
                          .scope(mScope)
                          .statusCallback(this::replyStatus)
                          .replyCallback(EReplyMessage.class,
                                         this::emptyHistoricReply)
                          .replyCallback(HistoricReply.class,
                                         this::historicReply)
                          .build();
        }
        // Else this feed is for future notifications only.

        mIsOpen = true;

        if (sLogger.isLoggable(Level.INFO))
        {
            sLogger.info(String.format("%s: started.", mName));
        }
    } // end of startup()

    /**
     * Closes subordinate feeds if open.
     */
    @Override
    public void shutdown()
    {
        if (sLogger.isLoggable(Level.FINE))
        {
            sLogger.fine(
                String.format("{}: shutting down.", mName));
        }

        // Close subordinate feeds if open.
        close();

        if (sLogger.isLoggable(Level.INFO))
        {
            sLogger.info(String.format("{}: shut down.", mName));
        }
    } // end of shutdown()

    //
    // end of EObject Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // ESubscriber Interface Implementation.
    //

    /**
     * Logs latest notification message feed status.
     * @param state latest notification message feed state.
     * @param feed status update applies to this notification
     * feed.
     */
    private void liveStatus(final EFeedState state,
                            final IESubscribeFeed feed)
    {
        if (sLogger.isLoggable(Level.INFO))
        {
            sLogger.info(
                String.format(
                    "%s: notification feed %s is %s.",
                    mName,
                    feed.key(),
                    state));
        }
    } // end of liveStatus(EFeedState, IESubscribeFeed)

    /**
     * If past notification download is in progress, then stores
     * notification message for later playback. Otherwise
     * forwards notification to historic subscriber.
     * @param msg latest {@link PublishStatusEvent} message.
     * @param feed {@code PublishStatusEvent} message feed.
     */
    private void livePublisherStatus(final ENotificationMessage msg,
                                     final IESubscribeFeed feed)
    {
        final PublishStatusEvent pse = (PublishStatusEvent) msg;

        if (sLogger.isLoggable(Level.INFO))
        {
            sLogger.info(
                String.format(
                    "%s: publisher %s notification feed %s is %s.",
                    mName,
                    pse.publisherId,
                    pse.key,
                    pse.feedState));
        }

        // Is this feed in the live state?
        if (mState.get() == HistoricFeedState.LIVE_NOTIFICATIONS)
        {
            // Yes. Forward this update to the historic subscribe
            // feed owner.
            forwardPublisherStatus(pse);
        }
        // Is this feed collecting
        else
        {
            // No. Store this status update for later playback.
            mMessages.add(msg);
        }
    } // end of livePublisherStatus(...)

    /**
     * If past notification download is in progress, then stores
     * notification message for later playback. Otherwise
     * forwards notification to historic subscriber.
     * @param msg latest notification message from feed.
     * @param feed notification message feed.
     */
    private void liveNotification(final ENotificationMessage msg,
                                  final IESubscribeFeed feed)
    {
        // is this message within the specified end time?
        if (!isInEndTime(msg.timestamp))
        {
            // No. Ignore this message.
        }
        // Is there an historic notification request in progress?
        else if (
            mState.get() == HistoricFeedState.LIVE_NOTIFICATIONS)
        {
            // No. Forward this notification message to owner.
            forwardNotification(msg);
        }
        else
        {
            // Yes. Store this status update for later playback.
            mMessages.add(msg);
        }
    } // end of liveNotification(...)

    //
    // end of ESubscriber Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // ERequestor Interface Implementation.
    //

    /**
     * Logs historic request feed status.
     * @param feedState latest request feed status.
     * @param feed request feed.
     */
    private void replyStatus(final EFeedState feedState,
                             final IERequestFeed feed)
    {
        if (sLogger.isLoggable(Level.INFO))
        {
            sLogger.info(
                String.format(
                    "%s: request feed %s is %s.",
                    mName,
                    feed.key(),
                    feedState));
        }

        // Is the reply feed up?
        // Is historic notification request not yet placed?
        if (feedState == EFeedState.UP && mRequest == null)
        {
            // Place historic notification request now.
            placeRequest();
        }
    } // end of replyStatus(EFeedState, IERequestFeed)

    /**
     * Received an {@code EReplyMessage} containing no historic
     * notification messages.
     * @param remaining number of in-progress requests remaining.
     * @param reply reply message
     * @param request historic notification request.
     */
    private void emptyHistoricReply(final int remaining,
                                    final EReplyMessage reply,
                                    final ERequestFeed.ERequest request)
    {
        final ERequestFeed.RequestState state =
            request.requestState();

        if (sLogger.isLoggable(Level.FINER))
        {
            sLogger.finer(
                String.format(
                    "%s: historic reply, request state %s, %d remaining.",
                    mName,
                    state,
                    remaining));
        }

        // Are all repliers now completed?
        if (state == ERequestFeed.RequestState.DONE ||
            state == ERequestFeed.RequestState.CANCELED)
        {
            // Yes, request has reached a final state.
            // Forward historic and live notifications to
            // subscriber.
            playback();
        }
        // There are more replies to come. Wait for the rest.
    } // end of noHistoricMessages(int, EReplyMessage, ERequest)

    /**
     * Processes historic notification reply. Adds messages to
     * collected historic and live notifications.
     * @param remaining number of in-progress requests remaining.
     * @param reply {@code HistoricReply} message.
     * @param request historic notification request.
     */
    private void historicReply(final int remaining,
                               final EReplyMessage reply,
                               final ERequestFeed.ERequest request)
    {
        final HistoricReply histReply = (HistoricReply) reply;
        final int numMessages = (histReply.notifications).length;
        int i;
        final ERequestFeed.RequestState state =
            request.requestState();

        if (sLogger.isLoggable(Level.FINER))
        {
            sLogger.finer(
                String.format(
                    "%s: historic reply with %d notifications, request state %s, %d remaining.",
                    mName,
                    numMessages,
                    state,
                    remaining));
        }

        // Store these historic notifications for later playback.
        for (i = 0; i < numMessages; ++i)
        {
            mMessages.add(histReply.notifications[i]);
        }

        // Are all repliers now completed?
        if (remaining == 0)
        {
            // Yes, request has reached a final state.
            // Forward historic and live notifications to
            // subscriber.
            playback();
        }
    } // end of historicReply(int, EReplyMessage, ERequest)

    //
    // end of ERequestor Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // TimerTaskListener Interface Implementation.
    //

    /**
     * Handles historic future end time expiration. Sets feed
     * state to {@link HistoricFeedState#DONE_SUCCESS}, closes
     * historic subscribe feed, and informs historic subscriber
     * that feed is ended.
     * @param event end time expiration event.
     */
    private void endOfLiveStream(final TimerEvent event)
    {
        if (sLogger.isLoggable(Level.FINE))
        {
            sLogger.fine(
                String.format(
                    "%s: %s live stream ended, state=%s.",
                    mName,
                    mKey,
                    mState.get()));
        }

        if (mState.compareAndSet(HistoricFeedState.LIVE_NOTIFICATIONS,
                                 HistoricFeedState.DONE_SUCCESS))
        {
            // Now that the historic feed completed its work,
            // close the feed.
            close();

            // Let the historic subscriber know that this feed
            // is done.
            forwardFeedDone(mState.get());
        }
    } // end of endOfLiveStream(TimerEvent)

    //
    // end of TimerTaskListener Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns begin time location.
     * @return begin time location.
     */
    public TimeLocation beginLocation()
    {
        return (mBeginLocation);
    } // end of beginLocation()

    /**
     * Returns begin time. If begin time location is
     * {@link TimeLocation#NOW}, then returns {@code null}.
     * @return begin time. May return {@code null}.
     */
    public @Nullable Instant beginTime()
    {
        return (mBeginTime);
    } // end of beginTime()

    /**
     * Returns begin time clusivity. Does not return
     * {@code null}.
     * @return non-{@code null} begin time clusivity.
     */
    public Clusivity beginClusivity()
    {
        return (mBeginClusivity);
    } // end of beginClusivity()

    /**
     * Returns end time location.
     * @return end time location.
     */
    public TimeLocation endLocation()
    {
        return (mEndLocation);
    } // end of endLocation()

    /**
     * Returns end time. If end time location is either
     * {@link TimeLocation#NOW} or {@link TimeLocation#ON_GOING},
     * then returns {@code null}.
     * @return end time. May return {@code null}.
     */
    public @Nullable Instant endTime()
    {
        return (mEndTime);
    } // end of endTime()

    /**
     * Returns end time clusivity. Does not return {@code null}.
     * @return non-{@code null} end time clusivity.
     */
    public Clusivity endClusivity()
    {
        return (mEndClusivity);
    } // end of endClusivity()

    /**
     * Returns current historic subscribe feed state. If
     * {@link HistoricFeedState#DONE_ERROR} is returned, then
     * the exception causing this error may be accessed via
     * {@link #errorCause()}.
     * @return historic subscribe feed state.
     *
     * @see #errorCause()
     */
    public HistoricFeedState state()
    {
        return (mState.get());
    } // end of state()

    /**
     * Returns exception which causes an
     * {@link HistoricFeedState#DONE_ERROR} historic subscribe
     * feed state. Returns {@code null} if there was no such
     * error.
     * @return exception causing error feed state.
     */
    public @Nullable Exception errorCause()
    {
        return (mErrorCause);
    } // end of

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Puts historic notification request and live notification
     * feeds in place. If feed end time is in the past, then only
     * historic notifications are requested. If begin time starts
     * at the current time, then only live notification
     * subscription is placed.
     * <p>
     * Does nothing if historic subscribe feed already has
     * subordinate feeds in place.
     * </p>
     * @throws IllegalStateException
     * if this feed is not open.
     */
    public void subscribe()
    {
        // Is the feed open?
        if (!mIsOpen)
        {
            throw (new IllegalStateException("feed is closed"));
        }

        // Is the subscription already in place?
        if (!mInPlace)
        {
            // No. Place the subscriptions.
            // Does this historic feed require future
            // notification messages?
            if (mEndLocation.isFuture())
            {
                // Yes. Subscribe to the live feed.
                if (sLogger.isLoggable(Level.FINER))
                {
                    sLogger.finer(
                        String.format(
                            "%s: subscribing to %s (%s)",
                            mName,
                            mKey,
                            mScope));
                }

                mStatusFeed.subscribe();
                mSubscribeFeed.subscribe();
            }

            // Does this historic feed request past notification
            // messages?
            if (mBeginLocation.isPast())
            {
                // Yes. Start the feed and place request when
                // past notification replier(s) in place.
                if (sLogger.isLoggable(Level.FINER))
                {
                    sLogger.finer(
                        String.format(
                            "%s: subscribing to %s (%s)",
                            mName,
                            mRequestKey,
                            mScope));
                }

                mState.set(
                    HistoricFeedState.HISTORIC_NOTIFICATIONS);
                mRequestFeed.subscribe();
            }
            else
            {
                // No, not interested in past notifications
                // messages. Only collecting live notificiations.
                mState.set(HistoricFeedState.LIVE_NOTIFICATIONS);
            }

            mInPlace = true;
        }
        // Else the subscription is already up.
    } // end of subscribe()

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    /**
     * Returns a new historic subscribe feed builder for the
     * specified notification message key and historic
     * subscriber.
     * @param key subscribed notification message key.
     * @param subscriber historic subscriber associated with this
     * feed.
     * @return new historic subscribe feed builder.
     * @throws NullPointerException
     * if {@code key} or {@code subscriber} is {@code null}.
     * @throws IllegalArgumentException
     * if {@code key} is not a notification message.
     */
    public static Builder builder(final EMessageKey key,
                                  final IEHistoricSubscriber subscriber)
    {
        Objects.requireNonNull(key, "message key is null");
        Objects.requireNonNull(subscriber, "subscriber is null");

        if (!key.isNotification())
        {
            throw (new IllegalArgumentException(
                "not a notification message key"));
        }

        return (new Builder(subscriber, key));
    } // end of builder(EMessageKey, IEHistoricSubscriber)

    /**
     * Returns a timer task schedule to expire either at the
     * given end time (clusivity is exclusive) or one millisecond
     * after end time (clusivity is inclusive). This timer should
     * be set only when historic subscribe feed is configured to
     * end at a specific time in the future.
     * @param endTime when historic subscribe feed is scheduled
     * to end.
     * @param endClusivity either includes or excludes end time.
     * @return scheduled timer task. Returns {@code null} if
     * end time is at or in the past.
     */
    private @Nullable TimerTask scheduleEndTimer(final Instant endTime,
                                                 final Clusivity endClusivity)
    {
        final long now = System.currentTimeMillis();
        final long et = endTime.toEpochMilli();
        final long delta =
            (endClusivity == Clusivity.EXCLUSIVE ?
             (et - now) :
             ((et - now) + 1L));
        TimerTask retval = null;

        if (delta >= 0L)
        {
            retval = new TimerTask(this::endOfLiveStream);
            sTimer.schedule(retval, delta);
        }

        return (retval);
    } // end of scheduleEndTimer(Instant, Clusivity)

    /**
     * Places historic notification request to all extant
     * notification publishers. If the request fails, then
     * closes this historic subscribe feed, stores the exception
     */
    private void placeRequest()
    {
        final Instant now = Instant.now();
        final EInterval.Builder intervalBuilder =
            EInterval.builder();
        final EInterval interval;

        // This historic subscribe feed is now retrieving
        // past notification messages and publisher status
        // events.
        mState.set(HistoricFeedState.HISTORIC_NOTIFICATIONS);

        // Create the lists to store both past and live
        // notification messages.
        mMessages = new ArrayList<>();

        intervalBuilder.beginTime(mBeginTime)
                       .beginClusivity(mBeginClusivity);

        // Is the request in the past?
        if (mEndLocation.isPast())
        {
            // Yes. Place past end time into interval.
            intervalBuilder.endTime(mEndTime)
                           .endClusivity(mEndClusivity);
        }
        // No. Use the current time as the end time, inclusive.
        else
        {
            intervalBuilder.endTime(now)
                           .endClusivity(Clusivity.INCLUSIVE);
        }

        interval = intervalBuilder.build();

        if (sLogger.isLoggable(Level.FINE))
        {
            sLogger.fine(
                String.format(
                    "%s: placing %s request, interval=%s",
                    mName,
                    mKey,
                    interval));
        }

        try
        {
            final HistoricRequest.Builder builder =
                HistoricRequest.builder();
            final HistoricRequest reqMessage =
                builder.subject(mRequestKey.subject())
                       .timestamp(now)
                       .interval(interval)
                       .build();

            mRequest = mRequestFeed.request(reqMessage);
        }
        catch (IllegalArgumentException |
               IllegalStateException |
               NullPointerException |
               ValidationException jex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "%s: %s historic notification request failed: %s",
                    mName,
                    mKey,
                    jex.getMessage()),
                jex);

            // Close this historic feed upon failure to retrieve
            // historic messages.
            mState.set(HistoricFeedState.DONE_ERROR);
            mErrorCause = jex;
            close();

            // Let the owner know about this failure.
            forwardFeedDone(mState.get());
        }
    } // end of placeRequest()

    /**
     * Returns {@code true} is &lt; ({@link Clusivity#EXCLUSIVE})
     * or &le; ({@link Clusivity#INCLUSIVE}) of the feed end
     * time and {@code false} if outside end time.
     * @param timestamp notification message timestamp.
     * @return {@code true} if notification timestamp is within
     * feed end time.
     */
    private boolean isInEndTime(final long timestamp)
    {
        final long endTime =
            (mEndTime == null ? 0L : mEndTime.toEpochMilli());

        // The timestamp is within the end time if:
        // 1. end time is on going, or
        // 2. end time is inclusive and timestamp <= end time, or
        // 3. end time is exclusive and timstamp < end time.
        // Note: if end time is located in the past or now, then
        // there is no live feed subscription and this method
        // will not be called.
        return (mEndLocation == TimeLocation.ON_GOING ||
                (mEndClusivity == Clusivity.EXCLUSIVE &&
                 timestamp < endTime) ||
                (mEndClusivity == Clusivity.INCLUSIVE &&
                 timestamp <= endTime));
    } // end of isInEndTime(long)

    /**
     * Forwards historic and saved live messages to historic
     * feed subscriber.
     */
    private void playback()
    {
        // 1. Sort messages by timestamp, publisher
        //    identifier, and message position.
        mMessages.sort(new PastComparator());

        // 2. Play back the sorted past messages.
        //    If there is overlap between past and live messages,
        //    removes redundant messages.
        mMessages.forEach(this::playback);

        // 3. Clear the past and live message arrays and drop
        //    reference to them.
        mMessages.clear();
        mMessages = null;

        // 4. Is the historic subscribe feed reached completion?
        if (mEndLocation.mIsPast ||
            mEndLocation == TimeLocation.NOW)
        {
            // 4.1. Yes. End the historic subscribe feed.
            mState.set(HistoricFeedState.DONE_SUCCESS);
            forwardFeedDone(mState.get());
        }
        else
        {
            // 4.2. No. Forward live notifications as they
            //      arrive.
            mState.set(HistoricFeedState.LIVE_NOTIFICATIONS);
        }
    } // end of playback()

    /**
     * Forwards this notification message to the appropriate
     * callback method depending on whether this is a publisher
     * status event or an historic notification.
     * @param msg notification message.
     */
    private void playback(final ENotificationMessage msg)
    {
        // Is this a publisher status event?
        if (msg instanceof PublishStatusEvent)
        {
            // Yes.
            forwardPublisherStatus((PublishStatusEvent) msg);
        }
        else
        {
            // No.
            forwardNotification(msg);
        }
    } // end of playback(ENotificiationMessage)

    /**
     * Forwards historic feed done state to historic subscriber.
     * Catches and logs any exception thrown by subscriber
     * callback.
     * @param state historic feed done state.
     */
    private void forwardFeedDone(final HistoricFeedState state)
    {
        try
        {
            mDoneCallback.call(state, this);
        }
        catch (Exception jex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "%s: historic subscriber callback exception:",
                    mName),
                jex);
        }
    } // end of forwardFeedDone(HistroicFeedState)

    /**
     * Forwards publisher status event to historic subscriber.
     * Catches and logs any exception thrown by subscriber
     * status callback.
     * @param pse publisher status event.
     */
    private void forwardPublisherStatus(final PublishStatusEvent pse)
    {
        try
        {
            mStatusCallback.call(pse, this);
        }
        catch (Exception jex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "%s: subscriber feed status callback exception:",
                    mName),
                jex);
        }
    } // end of forwardPublisherStatus(EFeedState)

    /**
     * Forwards latest notification message to historic
     * subscriber. Catches and logs any exception thrown by
     * subscriber status callback.
     * @param msg notification message.
     */
    private void forwardNotification(final ENotificationMessage msg)
    {
        try
        {
            if (mCondition.test(msg))
            {
                mNotifyCallback.call(msg, this);
            }
        }
        catch(Exception jex)
        {
            sLogger.log(
                Level.WARNING,
                String.format(
                    "%s: subscriber notify callback exception:",
                    mName),
                jex);
        }
    } // end of forwardNotifications(ENotificationMessage)

//---------------------------------------------------------------
// Member data.
//

    /**
     * Builder is used to instantiate
     * {@code EHistoricSubscribeFeed} instances. A builder
     * instance is acquired via
     * {@link #builder(EMessageKey, IEHistoricSubscriber)}
     * which requires that a non-{@code null} notification
     * message key and historic subscriber be provided. This
     * message key defines the notification message class and
     * subject subscribed to and retrieved by the historic
     * subscriber. The historic subscriber owns the historic
     * subscribe feed. The historic subscribe feed is an eBus
     * hybrid object and runs in the owner's dispatcher.
     * <p>
     * This builder requires the following properties be
     * defined (beside those properties required by
     * {@link EAbstractHistoricFeed.Builder}):
     * </p>
     * <ul>
     *   <li>
     *     begin timestamp, location, and clusivity and
     *   </li>
     *   <li>
     *     end timestamp, location, and clusivity.
     *   </li>
     * </ul>
     * <p>
     * It is recommended that each builder instance be used to
     * create only one historic publish feed.
     * </p>
     */
    public static final class Builder
        extends EAbstractHistoricFeed.Builder<IEHistoricSubscriber,
                                              EHistoricSubscribeFeed,
                                              Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Forward only those notifications to historic
         * subscriber which satisfy this condition. Defaults
         * to {@link EFeed#NO_CONDITION} which
         * always returns true.
         */
        private ECondition mCondition;

        /**
         * Historic feed status callback.
         */
        private HistoricFeedDoneCallback mDoneCallback;

        /**
         * Feed status callback. If not explicitly set by client,
         * then defaults to
         * {@link IEHistoricSubscriber#feedStatus(PublishStatusEvent, EHistoricSubscribeFeed)}.
         */
        private HistoricFeedStatusCallback mStatusCallback;

        /**
         * Notification message callback. If not explicity set by
         * client, then defaults to
         * {@link ESubscriber#notify(ENotificationMessage, ESubscribeFeed)}.
         */
        private HistoricNotifyCallback mNotifyCallback;

        /**
         * Historic feed may begin either in the past or at the
         * current time but not in the future.
         */
        private TimeLocation mBeginLocation;

        /**
         * Historic feed begin time. If {@link #mBeginLocation}
         * is {@link TimeLocation#NOW} then will be {@code null}.
         */
        private Instant mBeginTime;

        /**
         * Historic begin time clusivity. Will be {@code null}
         * if begin time location is now.
         */
        private Clusivity mBeginClusivity;

        /**
         * Historic feed ends either in the past, now, or the
         * future.
         */
        private TimeLocation mEndLocation;

        /**
         * Historic feed end time. If {@link #mEndLocation} is
         * either {@link TimeLocation#NOW} or
         * {@link TimeLocation#ON_GOING}, then this value is
         * {@code null}.
         */
        private Instant mEndTime;

        /**
         * Historic end time clusivity. Will be {@code null}
         * if end time location is now or in the future.
         */
        private Clusivity mEndClusivity;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a historic subscribe feed builder for the
         * given notification message key.
         * @param key notification message key.
         * @param subscriber historic subscriber.
         */
        private Builder(final IEHistoricSubscriber subscriber,
                        final EMessageKey key)
        {
            super (subscriber,
                   EHistoricSubscribeFeed.class,
                   key);

            mCondition = EFeed.NO_CONDITION;
        } // end of Builder(EMessageKey)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Implementation.
        //

        /**
         * Returns {@code this Builder} self reference.
         * @return {@code this Builder} self reference.
         */
        @Override
        protected Builder self()
        {
            return (this);
        } // end of self()

        /**
         * Validates historic subscribe feed builder settings.
         * Returns {@code problems} parameter to allow for
         * {@code validate} method chaining.
         * @param problems place validation failures into this
         * list.
         * @return {@code problems} to allow {@code validate}
         * method chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            // An historic subscribe feed is valid if:
            // + abstract historic validation passes,
            // + begin and end time location are not null, and
            // + begin time is before end time.
            return (
                super.validate(problems)
                     .requireNotNull(mBeginLocation,
                                     "beginLocation")
                     .requireNotNull(mEndLocation, "endLocation")
                     .requireTrue((mBeginLocation == TimeLocation.NOW ||
                                   mBeginTime != null),
                                  "beginTime",
                                  "beginTime not set")
                     .requireTrue((mEndLocation == TimeLocation.NOW ||
                                   mEndLocation == TimeLocation.ON_GOING ||
                                   mEndTime != null),
                                  "endTime",
                                  "endTime not set")
                     .requireTrue(beginEndTimeCompare(),
                                  "beginTime",
                                  "beginTime >= endTime"));
        } // end of build()

        /**
         * Returns default historic subscribe feed name. Used
         * when feed name is not explicitly set.
         * @return default historic subscribe feed name.
         */
        @Override
        protected String generateName()
        {
            return (DEFAULT_SUBSCRIBE_FEED_NAME +
                    sFeedIndex.getAndIncrement());
        } // end of generateName()

        /**
         * Returns new historic subscribe feed instance created
         * from builder settings.
         * @return new historic subscribe feed instance.
         */
        @Override
        protected EHistoricSubscribeFeed buildImpl()
        {
            return (new EHistoricSubscribeFeed(this));
        } // end of buildImpl()

        //
        // end of Abstract Method Implementation.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns historic subscriber feed status callback
         * method.
         * @return historic subscriber feed status callback
         * method.
         */
        private HistoricFeedDoneCallback getDoneCallback()
        {
            // Was historic feed status callback defined?
            if (mDoneCallback == null)
            {
                // No. Create a callback to
                // IEHistoricSubscriber.historicFeedStatus()
                // method.
                mDoneCallback = mOwner::feedDone;
            }

            return (mDoneCallback);
        } // end of getHistoricCallback()

        /**
         * Returns historic subscriber feed status callback
         * method.
         * @return historic subscriber feed status callback
         * method.
         */
        private HistoricFeedStatusCallback getStatusCallback()
        {
            // Was status callback explicitly defined?
            if (mStatusCallback == null)
            {
                // No. Create a callback to
                // IEHistoricSubscriber.feedStatus() method.
                mStatusCallback = mOwner::feedStatus;
            }

            return (mStatusCallback);
        } // end of getStatusCallback()

        /**
         * Returns notification callback method.
         * @return notification callback method.
         */
        private HistoricNotifyCallback getNotifyCallback()
        {
            // Was the notify callback explicitly defined?
            if (mNotifyCallback == null)
            {
                // No. Create a callback to the
                // ESubscriber.notify() method.
                mNotifyCallback = mOwner::notify;
            }

            return (mNotifyCallback);
        } // end of getNotifyCallback()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets subscription condition to the given value. May
         * be {@code null} which results in a
         * {@link EFeed#NO_CONDITION} condition.
         * <p>
         * An example using this method is:
         * </p>
         * {@code builder.condition(m -> ((CatalogUpdate) m).category == Category.APPLIANCES)}
         * @param condition subscription condition.
         * @return {@code this Builder} instance.
         */
        public Builder condition(final ECondition condition)
        {
            if (condition == null)
            {
                mCondition = EFeed.NO_CONDITION;
            }
            else
            {
                mCondition = condition;
            }

            return (this);
        } // end of condition(ECondition)

        /**
         * Puts historic subscribe feed completion callback in
         * place. If {@code cb} is not {@code null}, feed
         * completion will be passed to {@code cb} rather than
         * {@link IEHistoricSubscriber#feedDone(EHistoricSubscribeFeed.HistoricFeedState, EHistoricSubscribeFeed)}.
         * The reverse is true, if {@code cb} is {@code null},
         * then updates are posted to the
         * {@code IEHistoricSubscriber.feedDone} override.
         * <p>
         * An example using this method is:
         * </p>
         * <pre><code>doneCallback(
    (fs, f) &rarr;
    {
        if (fs == HistoricFeedState.DONE_ERROR) {
            <strong style="color:ForestGreen">// Handle historic feed error.</strong>
        }
    }</code></pre>
         * @param cb historic feed completion callback. May be
         * {@code null}.
         * @return {@code this Builder} instance.
         */
        public Builder doneCallback(@Nullable final HistoricFeedDoneCallback cb)
        {
            mDoneCallback = cb;

            return (this);
        } // end of doneCallback(HistoricFeedDoneCallback)

        /**
         * Puts historic subscribe feed status callback in place.
         * If {@code cb} is not {@code null}, subscribe status
         * updates will be passed to {@code cb} rather than
         * {@link IEHistoricSubscriber#feedStatus(PublishStatusEvent, EHistoricSubscribeFeed)}.
         * The reverse is true: if {@code cb} is {@code null},
         * then updates are posted to the
         * {@code IEHistoricSubscriber.feedStatus} override.
         * <p>
         * An example using this method is:
         * </p>
         * <pre><code>statusCallback(
    (pse, f) &rarr; {
        if (pse.feedState == EFeedState.DOWN) {
            <strong style="color:ForestGreen">// Clean up in-progress work.</strong>

    }}</code></pre>
         * @param cb historic subscriber status update callback.
         * May be {@code null}.
         * @return {@code this Builder} instance.
         */
        public Builder statusCallback(@Nullable final HistoricFeedStatusCallback cb)
        {
            mStatusCallback = cb;

            return (this);
        } // end of statusCallback(HistoricFeedStatusCallback)

        /**
         * Puts the notification message callback in place. If
         * {@code cb} is not {@code null}, then notification
         * messages will be passed to {@code cb} rather than
         * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}.
         * A {@code null cb} means that notification messages will be
         * passed to the
         * {@link IEHistoricSubscriber#notify(ENotificationMessage, EHistoricSubscribeFeed)}
         * override.
         * <p>
         * An example showing how to use this method is:
         * {@code notifyCallback(this::locationUpdate)}
         * </p>
         * @param cb pass notification messages back to target
         * via this callback.
         * @return {@code this Builder} instance.
         */
        public Builder notifyCallback(@Nullable final HistoricNotifyCallback cb)
        {
            mNotifyCallback = cb;

            return (this);
        } // end of notifyCallback(HistoricNotifyCallback)

        /**
         * Sets historic subscription feed to begin in the
         * past.
         * @param beginTime historic feed begin time in the past.
         * @param beginClusivity historic feed begin time
         * clusivity.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code beginTime} or {@code beginClusivity} is
         * {@code null}.
         * @throws IllegalArgumentException
         * if {@code beginTime} is &ge; current time.
         * @throws IllegalStateException
         * if begin time is already set.
         */
        public Builder from(final Instant beginTime,
                            final Clusivity beginClusivity)
        {
            Objects.requireNonNull(
                beginTime, "begin time is null");
            Objects.requireNonNull(
                beginClusivity, "begin clusivity is null");

            if (beginTime.compareTo(Instant.now()) >= 0)
            {
                throw (
                    new IllegalArgumentException(
                        "begin time is not in the past"));
            }

            if (mBeginLocation != null)
            {
                throw (
                    new IllegalStateException(
                        "begin time already set"));
            }

            mBeginLocation = TimeLocation.PAST;
            mBeginTime = beginTime;
            mBeginClusivity = beginClusivity;

            return (this);
        } // end of from(Instant, Clusivity)

        /**
         * Sets historic subscription feed to begin at the
         * current time. The begin clusivity is set to
         * {@link Clusivity#INCLUSIVE}.
         * @return {@code this Builder} instance.
         * @throws IllegalStateException
         * if begin time is already set.
         */
        public Builder fromNow()
        {
            if (mBeginLocation != null)
            {
                throw (
                    new IllegalStateException(
                        "begin time already set"));
            }

            mBeginLocation = TimeLocation.NOW;
            mBeginClusivity = Clusivity.INCLUSIVE;

            return (this);
        } // end of fromNow()

        /**
         * Sets historic subscription feed to end either in the
         * past or in the future depending on whether
         * {@code endTime} is in the past or future. If end time
         * is at the current time, then begin time must be in the
         * past and not at the current time.
         * @param endTime historic feed end time in past or
         * future.
         * @param endClusivity historic feed end time clusivity.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code endTime} or {@code endClusivity} is
         * {@code null}.
         * @throws IllegalStateException
         * if end time is already set.
         */
        public Builder to(final Instant endTime,
                          final Clusivity endClusivity)
        {
            final int result;

            Objects.requireNonNull(
                endTime, "end time is null");
            Objects.requireNonNull(
                endClusivity, "end clusivity is null");

            if (mEndLocation != null)
            {
                throw (
                    new IllegalStateException(
                        "end time already set"));
            }

            result = endTime.compareTo(Instant.now());
            if (result == 0)
            {
                mEndLocation = TimeLocation.NOW;
                mEndTime = null;
            }
            else
            {
                mEndLocation = (result < 0 ?
                                TimeLocation.PAST :
                                TimeLocation.FUTURE);
                mEndTime = endTime;
            }

            mEndClusivity = endClusivity;

            return (this);
        } // end of to(Instant, Clusivity)

        /**
         * Sets historic subscription feed to end at the current
         * time. This means that historic begin time
         * <em>must</em> be in the past and not set to the
         * current time.
         * @param endClusivity historic subscribe feed ends at or
         * after current time.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code endClusivity} is {@code null}.
         * @throws IllegalStateException
         * if end time is already set.
         */
        public Builder toNow(final Clusivity endClusivity)
        {
            Objects.requireNonNull(
                endClusivity, "end clusivity is null");

            if (mEndLocation != null)
            {
                throw (
                    new IllegalStateException(
                        "end time already set"));
            }

            mEndLocation = TimeLocation.NOW;
            mEndClusivity = endClusivity;

            return (this);
        } // end of toNow(Clusivity)

        /**
         * Sets historic live subscription feed to continue until
         * unsubscribed.
         * @return {@code this Builder} instance.
         */
        public Builder toForever()
        {
            if (mEndLocation != null)
            {
                throw (
                    new IllegalStateException(
                        "end time already set"));
            }

            mEndLocation = TimeLocation.ON_GOING;
            mEndTime = null;
            mEndClusivity = Clusivity.INCLUSIVE;

            return (this);
        } // end of toForever()

        //
        // end of Set Methods.
        //-------------------------------------------------------

        /**
         * Returns {@code true} if begin time is before the
         * end time based on the time locations and, if
         * necessary, time instances; {@code false} if begin
         * time is at or after the end time.
         * @return {@code true} if begin time is before end time.
         */
        private boolean beginEndTimeCompare()
        {
            boolean retcode;

            // Does this feed begin in the past?
            if (mBeginLocation == TimeLocation.PAST)
            {
                // Yes. Was a begin time provided?
                if (mBeginTime == null)
                {
                    // No and that is a problem.
                    retcode = false;
                }
                // Does the feed end now or is it on-going.
                else if (mEndLocation == TimeLocation.NOW ||
                         mEndLocation == TimeLocation.ON_GOING)
                {
                    // Yes. Already checked if the begin time
                    // was prior to current time when setting
                    // it. Everything checks out.
                    retcode = true;
                }
                // Feed end time is either in the past or future,
                // so a time must be provided.
                else if (mEndTime == null)
                {
                    // No end time.
                    retcode = false;
                }
                else
                {
                    // Is the begin time before the end time?
                    retcode = (mBeginTime.isBefore(mEndTime));
                }
            }
            // Begin time is now. End time must be either future
            // or on-going.
            else
            {
                retcode =
                    (mEndLocation == TimeLocation.FUTURE ||
                     mEndLocation == TimeLocation.ON_GOING);
            }

            return (retcode);
        } // end of beginEndTimeCompare()
    } // end of class Builder<T extends ENotificationMessage>

    /**
     * Comparator used to sort historic notification messages
     * by timestamp, publisher identifier, and message position.
     */
    private static final class PastComparator
        implements Comparator<ENotificationMessage>
    {
    //-----------------------------------------------------------
    // Member data.
    //

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Private constructor to prevent instantiation outside
         * of {@code EHistorSubscribeFeed}.
         */
        private PastComparator()
        {}

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Comparator Interface Implemenation.
        //

        /**
         * Compares two eBus notification messages for order.
         * Returns an integer values &lt;, equal to, or &gt;
         * zero if the first argument is &lt;, equal to, or &gt;
         * the second argument. Comparison is based on the
         * following notification properties:
         * <ol>
         *   <li>
         *     message timestamp,
         *   </li>
         *   <li>
         *     publisher identifier, and
         *   </li>
         *   <li>
         *     message position.
         *   </li>
         * </ol>
         * Publisher identifiers are compared only if timestamps
         * are equal. Message positions are compared only if
         * timestamps and publisher identifiers are equal.
         * @param o1 first notification message to be compared.
         * @param o2 second notification message to be compared.
         * @return an integer value &lt;, equal to, or &gt; zero
         * if first argument is &lt;, equal to, or &gt; second
         * argument.
         */
        @Override
        public int compare(final ENotificationMessage o1,
                           final ENotificationMessage o2)
        {
            int retval =
                Long.compare(o1.timestamp, o2.timestamp);

            // Are the message timestamps equal?
            if (retval == 0)
            {
                // Yes. Now compare using publisher identifier.
                retval =
                    Long.compare(o1.publisherId, o2.publisherId);

                // Are the publisher identifiers the same?
                if (retval == 0)
                {
                    // Yes. Finish up by comparing the message
                    // positions. These should *not* be the same.
                    retval = (o1.position - o2.position);
                }
            }

            return (retval);
        } // end of compare(...)

        //
        // end of Comparator Interface Implemenation.
        //-------------------------------------------------------
    } // end of class PastComparator
} // end of class EHistoricSubscribeFeed
