//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

/**
 * An eBus client implementing {@link IEHistoricSubscriber} may
 * choose to implement this functional interface as a
 * {@code private} method instead of implementing the
 * {@code public}
 * {@link IEHistoricSubscriber#feedStatus(PublishStatusEvent, EHistoricSubscribeFeed)}
 * method. This private {@code HistoricFeedStatusCallback} method
 * is then passed to
 * {@link EHistoricSubscribeFeed.Builder#statusCallback(HistoricFeedStatusCallback)}.
 * <p>
 * <strong>Note:</strong> this callback may only be set when
 * building an {@code EHistoricSubscribeFeed} instance. Once
 * built, the publisher feed status callback target may
 * <em>not</em> be changed.
 * </p>
 * <p>
 * <strong>Note:</strong> failure to implement
 * {@code IEHistoricSubscriber.statusCallback} or to pass a
 * {@code HistoricFeedStatusCallback} method to
 * {@code Builder.statusCallback} results in a
 * {@code Builder.build()} failure.
 * </p>
 *
 * @see EHistoricSubscribeFeed
 * @see HistoricNotifyCallback
 * @see HistoricFeedDoneCallback
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@FunctionalInterface
public interface HistoricFeedStatusCallback
{
//---------------------------------------------------------------
// Member methods.
//

    /**
     * eBus calls this method to inform subscriber that the
     * publisher feed state is either {@code EFeedState.UP} or
     * {@code EFeedState.DOWN}.
     * @param event publish status event contains the publisher's
     * latest {@link PublishStatusEvent#feedState feed state} for
     * the given {@link PublishStatusEvent#key message key}.
     * @param feed historic subscribe feed.
     */
    void call(PublishStatusEvent event,
              EHistoricSubscribeFeed feed);
} // end of interface HistoricFeedStatusCallback

