//
// Copyright 2022 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.historic;

import java.io.Serializable;
import java.time.Instant;
import java.util.Objects;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.messages.EReplyInfo;
import net.sf.eBus.messages.ERequestMessage;
import net.sf.eBus.util.Validator;
import net.sf.eBusx.time.EInterval;

/**
 * Request for historic notification messages based on the given
 * date/time interval. This date/time interval
 * <strong>must</strong> be in the past (the interval ends before
 * or at the current wall clock time). Historic notifications
 * are returned via an {@link HistoricReply} message. Note that
 * the returned notifications may contain
 * {@link PublishStatusEvent} messages. This allows the requester
 * to know when notifications are missing due to a
 * {@link EFeedState#DOWN down feed state} and decide if
 * statistical analysis is possible in the face of missing data.
 * <p>
 * An example historic notification request creation (note that
 * requests may only be created using a {@link Builder builder}
 * instance):
 * </p>
 * <pre><code>final HistoricRequest.Builder builder = HistoricRequest.builder();
final EInterval interval = ...; // See net.sf.eBusx.time.EInterval.
final HistoricRequest request = builder.subject(mSubject)
                                       .timestamp(Instant.now())
                                       .interval(interval)
                                       .build();</code></pre>
 * <p>
 * This request is generally used by
 * {@link EHistoricSubscribeFeed} to retrieve past notifications
 * from {@link EHistoricPublishFeed} instances.
 * </p>
 *
 * @see HistoricReply
 * @see PublishStatusEvent
 * @see EHistoricSubscribeFeed
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@EReplyInfo (replyMessageClasses = {HistoricReply.class})
public final class HistoricRequest
    extends ERequestMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x060300L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Retrieve historic notification messages over this
     * date/time interval.
     */
    public final EInterval interval;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new historic request instance based on the
     * builder's settings.
     * @param builder contains valid request settings.
     */
    private HistoricRequest(final Builder builder)
    {
        super (builder);

        this.interval = builder.mInterval;
    } // end of HistoricRequest(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns textual representation of an historic notification
     * message request.
     * @return request message as text.
     */
    @Override
    public String toString()
    {
        final StringBuilder retval = new StringBuilder();

        return (retval.append(super.toString())
                      .append("\n               interval: ")
                      .append(interval)
                      .toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    /**
     * Returns an historic notification request builder instance.
     * It is recommended that a new builder instance be used for
     * each historic request meaning builder instances should not
     * be reused.
     * @return new historic notification request builder
     * instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Builder class for {@code HistoricRequest} message
     * instances. A {@code Builder} instance is acquired using
     * the {@link HistoricRequest#builder()} method. Each builder
     * instance should be used to build only one
     * {@code HistoricRequest} instance.
     * <p>
     * {@code Builder} requires {@link HistoricRequest#interval}
     * be set to a non-{@code null} value.
     * </p>
     *
     * @see HistoricRequest
     */
    public static final class Builder
        extends ERequestMessage.Builder<HistoricRequest, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Retrieve historic notifications over this date/time
         * interval.
         */
        private EInterval mInterval;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Private constructor since builder instances are
         * created by {@link HistoricRequest#builder()} method
         * only.
         */
        private Builder()
        {
            super (HistoricRequest.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Checks if this builder is correctly configured before
         * building target message instance.
         * A valid {@code HistoricRequest} requires that the
         * time interval is set. Calls {@code super.validate}
         * to check super class fields.
         * @param problems place invalid fields into this list.
         * @return {@code problems} is returned to all method
         * chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            return (super.validate(problems)
                         .requireNotNull(mInterval, "interval"));
        } // end of validate(Validator)

        /**
         * Returns a new {@code HistoricRequest} instance based
         * on this builder's configuration.
         * @return new {@code HistoricRequest} instance.
         */
        @Override
        protected HistoricRequest buildImpl()
        {
            return (new HistoricRequest(this));
        } // end of buildImpl()

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the historic notification interval. This interval
         * must be entirely in the past.
         * @param interval retrieve notifications over this
         * historic date/time interval.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code interval} is {@code null}.
         * @throws IllegalArgumentException
         * if {@code interval.endTime} &gt; the current wall
         * clock time.
         */
        public Builder interval(final EInterval interval)
        {
            final Instant now = Instant.now();

            Objects.requireNonNull(interval, "interval is null");

            // Is the interval at or before now?
            if (now.compareTo(interval.endTime) < 0)
            {
                throw (
                    new IllegalArgumentException(
                        "interval is not historic"));
            }

            mInterval = interval;

            return (this);
        } // end of interval(EInterval)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class HistoricRequest
