//
// Copyright 2018, 2020 Charles W. Rapp
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package net.sf.eBus.feed.pattern;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import net.sf.eBus.messages.ELocalOnly;
import net.sf.eBus.messages.ENotificationMessage;
import net.sf.eBus.util.Validator;

/**
 * A match event reports an event pattern match. Contains the
 * <a href="CapturingGroups">capturing group map</a> and the
 * <a href="UserCache">user-defined data cache</a>.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@ELocalOnly
public final class MatchEvent
    extends ENotificationMessage
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x000100L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The collected events for each pattern group. Pattern
     * group {@link EventPattern#ALL_EVENTS} contains all matched
     * events. The remaining map keys (if any) are from the
     * user-defined {@link EventPattern}. The groups map and
     * the notification list values are both read-only.
     */
    public final Map<String, List<ENotificationMessage>> groups;

    /**
     * This map contains the user-defined data stored when
     * {@link MatchCondition}s were executed. This map is not
     * {@code null} but may be empty.
     */
    public final Map<Object, Object> userCache;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new match event based on the given builder
     * configuration.
     * @param builder contains message configuration.
     */
    private MatchEvent(final Builder builder)
    {
        super (builder);

        groups = builder.mGroups;
        userCache = builder.mUserCache;
    } // end of MatchEvent(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns a string containing each of the groups and the
     * group event.
     * @return textual representation of the match event.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public String toString()
    {
        int ei;
        final StringBuilder retval = new StringBuilder();

        retval.append(super.toString())
              .append("\n  Capturing groups:");

        for (Map.Entry<String, List<ENotificationMessage>> entry :
                 groups.entrySet())
        {
            retval.append("\n  group[")
                  .append(entry.getKey())
                  .append(']');

            ei = 0;
            for (ENotificationMessage msg : entry.getValue())
            {
                retval.append("\n    event[")
                      .append(ei)
                      .append("]\n")
                      .append(msg);
                ++ei;
            }
        }

        retval.append("\n  User-defined data:");
        userCache.entrySet()
                 .forEach(
                     entry -> retval.append("\n    ")
                                    .append(entry.getKey())
                                    .append('=')
                                    .append(entry.getValue()));

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the notification list associated with the given
     * group name. May return {@code null} if there is not group
     * with the given name.
     * @param name group name.
     * @return group event list.
     */
    public List<ENotificationMessage> group(final String name)
    {
        return (groups.get(name));
    } // end of group(String)

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Returns a new {@code MatchEvent} builder instance.
     * @return {@code MatchEvent} builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    public static final class Builder
        extends ENotificationMessage.Builder<MatchEvent, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private Map<String, List<ENotificationMessage>> mGroups;
        private Map<Object, Object> mUserCache;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (MatchEvent.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns a new {@code MatchEvent} instance based on
         * this builder's configuration.
         * @return {@code MatchEvent} instance.
         */
        @Override
        protected MatchEvent buildImpl()
        {
            return (new MatchEvent(this));
        } // end of buildImpl()

        /**
         * Checks if this builder is correctly configured before
         * attempting to build the target message instance.
         * A valid {@code MatchEvent} requires that both the
         * message groups and user cache be set.
         * @param problems place invalid fields into this list.
         * @return {@code problems} is returned to all method
         * chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            return (super.validate(problems)
                         .requireNotNull(mGroups, "groups")
                         .requireNotNull(mUserCache, "userCache"));
        } // end of validate(Validator)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the message collection map to the given value.
         * @param groups message collection map. May be empty but
         * not {@code null}.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code groups} is {@code null}.
         */
        public Builder groups(
            final Map<String, List<ENotificationMessage>> groups)
        {
            mGroups =
            Objects.requireNonNull(groups, "groups is null");

            return (this);
        } // end of groups(Map<>)

        /**
         * Sets the user cache to the given value.
         * @param userCache user cache map. May not be
         * {@code null} but may be empty.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code userCache} is {@code null}.
         */
        public Builder userCache(final Map<Object, Object> userCache)
        {
            mUserCache =
                Objects.requireNonNull(
                    userCache, "user cache is null");

            return (this);
        } // end of userCache(Map<>)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class MatchEvent
