//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2018, 2020. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.feed;

import java.io.Serializable;
import java.util.Formatter;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import net.sf.eBus.messages.ELocalOnly;
import net.sf.eBus.messages.ENotificationMessage;


/**
 * A match event reports an event pattern match. Contains the
 * <a href="CapturingGroups">capturing group map</a> and the
 * <a href="UserCache">user-defined data cache</a>.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@ELocalOnly
public final class MatchEvent
    extends ENotificationMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The collected events for each pattern group. Pattern
     * group {@link EventPattern#ALL_EVENTS} contains all matched
     * events. The remaining map keys (if any) are from the
     * user-defined {@link EventPattern}. The groups map and
     * the notification list values are both read-only.
     */
    public final Map<String, List<ENotificationMessage>> groups;

    /**
     * This map contains the user-defined data stored when
     * {@link MatchCondition}s were executed. This map is not
     * {@code null} but may be empty.
     */
    public final Map<Object, Object> userCache;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new match event based on the given builder
     * configuration.
     * @param builder contains message configuration.
     */
    private MatchEvent(final Builder builder)
    {
        super (builder);

        groups = builder.mGroups;
        userCache = builder.mUserCache;
    } // end of MatchEvent(Builder)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns a string containing each of the groups and the
     * group event.
     * @return textual representation of the match event.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public String toString()
    {
        int ei;
        final Formatter retval = new Formatter();

        retval.format("%s", super.toString());
        retval.format("%n  Capturing groups:");

        for (Map.Entry<String, List<ENotificationMessage>> entry :
                 groups.entrySet())
        {
            retval.format("%n  group[%s]", entry.getKey());

            ei = 0;
            for (ENotificationMessage msg : entry.getValue())
            {
                retval.format("%n    event[%2d]%n%s", ei, msg);
                ++ei;
            }
        }

        retval.format("%n%n  User-defined data:");
        userCache.entrySet()
                 .forEach(
                     entry -> retval.format("%n    %s=%s",
                                            entry.getKey(),
                                            entry.getValue()));

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the notification list associated with the given
     * group name. May return {@code null} if there is not group
     * with the given name.
     * @param name group name.
     * @return group event list.
     */
    public List<ENotificationMessage> group(final String name)
    {
        return (groups.get(name));
    } // end of group(String)

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    /**
     * Returns a new {@code MatchEvent} builder instance.
     * @return {@code MatchEvent} builder instance.
     */
    public static Builder builder()
    {
        return (new Builder());
    } // end of builder()

//---------------------------------------------------------------
// Inner classes.
//

    public static final class Builder
        extends ENotificationMessage.Builder<MatchEvent, Builder>
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        private Map<String, List<ENotificationMessage>> mGroups;
        private Map<Object, Object> mUserCache;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        private Builder()
        {
            super (MatchEvent.class);
        } // end of Builder()

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Builder Method Overrides.
        //

        /**
         * Returns a new {@code MatchEvent} instance based on
         * this builder's configuration.
         * @return {@code MatchEvent} instance.
         */
        @Override
        protected MatchEvent buildImpl()
        {
            return (new MatchEvent(this));
        } // end of buildImpl()

        /**
         * Checks if this builder is correctly configured before
         * attempting to build the target message instance.
         * A valid {@code MatchEvent} requires that both the
         * message groups and user cache be set.
         * @param problems place invalid fields into this list.
         * @return {@code problems} is returned to all method
         * chaining.
         */
        @Override
        protected Validator validate(final Validator problems)
        {
            return (super.validate(problems)
                         .requireNotNull(mGroups, "groups")
                         .requireNotNull(mUserCache, "userCache"));
        } // end of validate(Validator)

        //
        // end of Builder Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the message collection map to the given value.
         * @param groups message collection map. May be empty but
         * not {@code null}.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code groups} is {@code null}.
         */
        public Builder groups(
            final Map<String, List<ENotificationMessage>> groups)
        {
            mGroups =
            Objects.requireNonNull(groups, "groups is null");

            return (this);
        } // end of groups(Map<>)

        /**
         * Sets the user cache to the given value.
         * @param userCache user cache map. May not be
         * {@code null} but may be empty.
         * @return {@code this Builder} instance.
         * @throws NullPointerException
         * if {@code userCache} is {@code null}.
         */
        public Builder userCache(final Map<Object, Object> userCache)
        {
            mUserCache =
                Objects.requireNonNull(
                    userCache, "user cache is null");

            return (this);
        } // end of userCache(Map<>)

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class Builder
} // end of class MatchEvent
