//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2018. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.feed;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import net.sf.eBus.client.EClient;
import net.sf.eBus.client.EFeed;
import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.ESubscribeFeed;
import static net.sf.eBus.client.ESubscribeFeed.FEED_STATUS_METHOD;
import net.sf.eBus.client.ESubscriber;
import net.sf.eBus.client.FeedStatusCallback;
import net.sf.eBus.client.IESubscribeFeed;
import net.sf.eBus.client.NotifyCallback;
import net.sf.eBus.feed.EventPattern.PatternType;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.ENotificationMessage;


/**
 * eBus pattern feeds sit between
 * {@link IESubscribeFeed eBus notification events} and an
 * {@link ESubscriber eBus client}, searching for a
 * client-specified  {@link EventPattern event pattern} in the
 * notification event stream. When a match is found, a
 * {@link MatchEvent} is posted to the client via the configured
 * notification callback, just like any other notification.
 * <p>
 * Follow these steps when using a pattern feed:
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 1:</strong> Implement
 * the {@link ESubscriber} interface.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 2:</strong> Define an
 * {@link EventPattern}.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 3:</strong> Open a
 * pattern feed for the subscriber and pattern defined above.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 4 (optional):</strong>
 * Do not override
 * {@link ESubscriber} interface methods. Instead, set callbacks
 * using {@link #statusCallback(FeedStatusCallback)} and/or
 * {@link #notifyCallback(NotifyCallback)} passing in Java lambda
 * expressions.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 5:</strong> Subscribe
 * to the open pattern feed.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 6:</strong> Wait for an
 * {@link EFeedState#UP up}
 * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed) feed status}.
 * This callback will occur before any {@link MatchEvent}s are
 * delivered. If the feed state is {@link EFeedState#DOWN down},
 * then no matches will be delivered until the feed state comes
 * back up.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 7:</strong> When the
 * pattern feed is up, wait for {@code MatchEvent}s to arrive.
 * </p>
 * <p>
 * <strong style="color:ForestGreen">Step 8:</strong> When the
 * subsceriber is shutting down, {@link #close() close} the feed.
 * </p>
 * <h2>Example use of {@code EPatternFeed}</h2>
 * <pre><code>import net.sf.eBus.client.EFeedState;
import net.sf.eBus.client.EPatternFeed;
import net.sf.eBus.client.ESubscriber;
import net.sf.eBus.messages.EventPattern;
import net.sf.eBus.messages.MatchEvent;

<strong style="color:ForestGreen">Step 1: Implement the ESubscriber interface.</strong>
public final class TradeAlgo implements ESubscriber
{
    // Monitor this stock symbol.
    private final String mSymbol;

    // Store the feed here so it can be used to unsubscribe.
    private EPatternFeed mFeed;

    public TradeAlgo(final String symbol) {
        mSymbol = symbol;
    }

    &#64;Override public void startup() {
        <strong style="color:ForestGreen">Step 2: Define event pattern.</strong>
        final EventPattern pattern = (see {@link EventPattern} to learn how to create a pattern.)

        <strong style="color:ForestGreen">Step 3: Open the pattern feed.</strong>
        mFeed = EPatternFeed.open(this, pattern);

        <strong style="color:ForestGreen">Step 4: ESubscriber interface method overridden.</strong>

        <strong style="color:ForestGreen">Step 5: Subscribe to the pattern feed.</strong>
        mFeed.subscribe();
    }


    <strong style="color:ForestGreen">Step 6: Wait for EFeedState.UP feed status.</strong>
    &#64;Override public void feedStatus(final EFeedState feedState, final IESubscribeFeed feed) {
        // What is the feed state?
        if (feedState == EFeedState.DOWN) {
            // Down. At least one pattern is down. Expect no matches until all
            // subordinate pattern feeds are up.
        } else {
            // Up. Expect to receive match notifications.
        }
    }

    <strong style="color:ForestGreen">Step 7: Wait for matches to arrive.</strong>
    &#64;Override public void notify(final ENotificationMessage msg, final IESubscribeFeed feed) {
        final MatchEvent match = (MatchEvent) msg;

        Match handling code here.
    }

    &#64;Override public void shutdown() {
        <strong style="color:ForestGreen">Step 8: When subscriber is shutting down, retract subscription feed.</strong>
        // mFeed.unsubscribe() is not necessary since close() will unsubscribe.
        if (mFeed != null) {
            mFeed.close();
            mFeed = null;
        }
    }
}</code></pre>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public abstract class EPatternFeed
    extends EFeed
    implements ESubscriber,
               IESubscribeFeed
{
//---------------------------------------------------------------
// Inner classes.
//

    /**
     * A match frame tracks an in-progress match. When a frame
     * contains a complete pattern match the frame is used to
     * generate the {@link MatchEvent} posted to the subscriber.
     * Match frames are kept until either an event violates the
     * pattern or the match completes.
     * <p>
     * Base class for {@code EOrderPatternFeed.MatchFrame} and
     * {@code EUnorderedPatternFeed.MatchFrame}.
     * </p>
     */
    protected static abstract class AbstractMatchFrame
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * Match frame created by this pattern. Used for logging
         * purposes.
         */
        protected final String mPatternName;

        /**
         * Contains all events matching the pattern so far.
         * Since these events are repeatedly referenced by the
         * matching algorithm, this list is kept for quick
         * reference.
         */
        protected final List<ENotificationMessage> mAllEvents;

        /**
         * Cache for storing interim, user-defined data
         * calculated while performing a {@link MatchCondition}.
         */
        protected final Map<Object, Object> mUserCache;

        /**
         * {@code true} if this match frame is for an exclusive
         * pattern. That means events may not be used for
         * multiple matches within the same pattern.
         */
        protected final boolean mIsExclusive;

        /**
         * Set to {@code true} when a match frame is 1) used for
         * an exclusive pattern, 2) the pattern is completely
         * matched, and 3) this frame contains common events with
         * the completely matched frame. Otherwise, will be
         * {@code false}. Marking a match frame as defunct allows
         * for "lazy removal" from the frames queue.
         */
        protected boolean mIsDefunct;

        /**
         * Every match frame has a unique identifier. Used for
         * logging purposes.
         */
        private final int mFrameId;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Creates a new match frame for the given pattern and
         * exclusivity flag.
         * @param patternName pattern name.
         * @param isExclusive {@code true} if the pattern is
         * exclusive.
         */
        protected AbstractMatchFrame(final String patternName,
                                     final boolean isExclusive)
        {
            mPatternName = patternName;
            mAllEvents = new ArrayList<>();
            mUserCache = new HashMap<>();
            mIsExclusive = isExclusive;
            mIsDefunct = false;
            mFrameId = nextFrameIdentifier();
        } // end of AbstractMatchFrame(String, boolean)

        /**
         * Copies the given frame data into this frame.
         * @param frame copy this frame.
         */
        protected AbstractMatchFrame(final AbstractMatchFrame frame)
        {
            mPatternName = frame.mPatternName;
            mIsExclusive = frame.mIsExclusive;
            mAllEvents = new ArrayList<>(frame.mAllEvents);
            mUserCache = new HashMap<>(frame.mUserCache);
            mIsDefunct = false;
            mFrameId = nextFrameIdentifier();
        } // end of AbstractMatchFrame(AbstractMatchFrame)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Abstract Method Declarations.
        //

        /**
         * Returns a <em>read-only copy</em> of the group map.
         * This is necessary since the returned map is passed to
         * the client-defined match condition and the client is
         * not allowed to modify this map.
         * @return group map read-only copy.
         */
        protected abstract Map<String, List<ENotificationMessage>> groupMap();

        //
        // end of Abstract Method Declarations.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        @Override
        public String toString()
        {
            int i = 0;
            final Formatter retval = new Formatter();

            retval.format(
                "%s %d%nEvents:", mPatternName, mFrameId);

            for (ENotificationMessage event : mAllEvents)
            {
                retval.format("%n  [%2d] %s", i, event);
                ++i;
            }

            return (retval.toString());
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the pattern name.
         * @return pattern name.
         */
        public final String patternName()
        {
            return (mPatternName);
        } // end of patternName()

        /**
         * Returns the unique match frame identifier.
         * @return match frame identifier.
         */
        public final int frameId()
        {
            return (mFrameId);
        } // end of frameId()

        /**
         * Returns {@code true} if this is an exclusive feed;
         * {@code false} otherwise.
         * @return {@code true} if this is an exclusive feed;
         */
        public final boolean isExclusive()
        {
            return (mIsExclusive);
        } // end of isExclusive()

        /**
         * Returns the "all events" collection as an unmodifiable
         * list.
         * @return unmodifiable all events list.
         */
        public final List<ENotificationMessage> allEvents()
        {
            return (Collections.unmodifiableList(mAllEvents));
        } // end of allEvents();

        /**
         * Returns the user-defined match data cache.
         * @return user match data cache.
         */
        public final Map<Object, Object> userCache()
        {
            return (mUserCache);
        } // end of userCache()

        /**
         * Returns {@code true} if this match frame is defunct
         * due to a prior exclusive matching and, therefore, may
         * no longer be used.
         * @return {@code true} if defunct.
         */
        public final boolean isDefunct()
        {
            return (mIsDefunct);
        } // end of isDefunct()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Marks this match frame as defunct. Next time this
         * frame is removed from the frame queue, it will be
         * ignored.
         */
        public final void setDefunct()
        {
            mIsDefunct = true;
            return;
        } // end of setDefunct()

        //
        // end of Set Methods.
        //-------------------------------------------------------
    } // end of class AbstractMatchFrame

//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Logging subsystem interface.
     */
    protected static final Logger sLogger =
        Logger.getLogger(EPatternFeed.class.getName());

    /**
     * Used to generate unique match frame identifiers.
     */
    private static int sNextFrameId = 0;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * Pattern name used as the {@link MatchEvent} subject.
     */
    protected final String mPatternName;

    /**
     * Publish match events on this message key. The
     * {@link EMessageKey#subject() message key subject} is
     * {@link #mPatternName}.
     */
    protected final EMessageKey mPubKey;

    /**
     * Collected matching events must satisfy this until
     * predicate. The collected events list and latest event are
     * the predicate parameters. This condition may be based on
     * any event field but will most likely use
     * {@link net.sf.eBus.messages.EMessage#timestamp}.
     */
    protected final BiPredicate<List<ENotificationMessage>, ENotificationMessage> mUntil;

    /**
     * Set to {@code true} if an event may be used for only one
     * match. If this is the case, then all extant match frames
     * are cleared when a single match completes.
     * <p>
     * <strong>Note:</strong> exclusivity applies only within a
     * pattern and not between patterns. Even if two patterns are
     * marked exclusive, a single event may still be used to
     * satisfy those two patterns.
     * </p>
     */
    protected final boolean mIsExclusive;

    /**
     * Match event is posted to subscriber if-and-only-if the
     * match event satisfies this condition. While this condition
     * may appear to play the same role as a feed condition,
     * the difference is that if this condition returns
     * {@code true} and the pattern is exclusive, that all
     * related match frames are removed.
     */
    protected final Predicate<MatchEvent> mCondition;

    /**
     * Maps the {@link EFeed#feedId() feed identifier} to the
     * subordinate notify feed. These feeds are defined by the
     * {@link EventPattern#parameters()}.
     */
    protected final List<ESubscribeFeed> mSubFeeds;

    /**
     * Maps an accepted notification event to the match frames
     * in which it appears. This map is used only for exclusive
     * pattern feeds. When a match frame completely matches the
     * pattern, then this map is used to find the match frames
     * referencing the same events.
     * <p>
     * Set to {@code null} for non-exclusive.
     * </p>
     */
    protected final Map<ENotificationMessage, List<AbstractMatchFrame>> mEventFrames;

    /**
     * Bit mask for all subordinate feeds. Bits are mapped to
     * feed identifiers.
     */
    protected long mAllSubFeedsMask;

    /**
     * Bit mask used to determine if all subordinate feeds are
     * up. When a feed is up, its matching bit is set in this
     * bit mask. When this mask equals {@link #mAllSubFeedsMask},
     * then this event feed is up.
     */
    protected long mSubFeedMask;

    /**
     * Feed status callback. If not explicitly set by client,
     * then defaults to
     * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}.
     */
    protected FeedStatusCallback<IESubscribeFeed> mStatusCallback;

    /**
     * Notification message callback. If not explicity set by
     * client, then defaults to
     * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}.
     */
    protected NotifyCallback mNotifyCallback;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new eBus pattern feed for the given subscriber
     * client and event pattern.
     * @param client post match events to this client.
     * @param pattern event pattern.
     */
    protected EPatternFeed(final EClient client,
                           final EventPattern pattern)
    {
        super (client, FeedScope.LOCAL_ONLY);

        mPatternName = pattern.patternName();
        mPubKey =
            new EMessageKey(
                MatchEvent.class, pattern.patternName());
        mUntil = pattern.until();
        mIsExclusive = pattern.isExclusive();
        mCondition = pattern.patternCondition();
        mEventFrames = (mIsExclusive ? new HashMap<>() : null);

        mSubFeeds = new ArrayList<>();
        mSubFeedMask = 0L;
        mAllSubFeedsMask = 0L;
    } // end of EPatternFeed(EClient, EventPattern)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Declarations.
    //

    /**
     * Applies {@code event} to the pattern. If this event
     * completes the pattern, then a {@link MatchEvent} is
     * created, tested with the match condition, and, if the match
     * event satisfies the condition, the match event is
     * forwarded to the subscriber.
     * @param event apply this event to the pattern.
     * @param eventId unique identifier associated with the
     * event type.
     */
    protected abstract void matchEvent(ENotificationMessage event,
                                       int eventId);

    //
    // end of Abstract Method Declarations.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Overrides.
    //

    /**
     * Retracts all in place, subordinate feed subscriptions.
     */
    @Override
    protected final void inactivate()
    {
        // Retract the subscription in case it is in place.
        unsubscribe();

        return;
    } // end of inactivate()

    //
    // end of Abstract Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // IESubscribeFeed Interface Implementation.
    //

    /**
     * Returns a message key with {@link MatchEvent} as the class
     * and the event pattern name as the subject.
     * @return pattern feed key.
     */
    @Override
    public final EMessageKey key()
    {
        return (mPubKey);
    } // end of key()

    /**
     * Puts the subscription feed status callback in place.
     * If {@code cb} is not {@code null}, feed status updates are
     * passed to {@code cb} rather than to
     * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}.
     * The reverse is true if {@code cb} is {@code null}. That
     * is, a {@code null cb} means feed status updates are
     * posted to the
     * {@link ESubscriber#feedStatus(EFeedState, IESubscribeFeed)}
     * override.
     * @param cb feed status update callback. May be {@code null}.
     * @throws IllegalStateException
     * if this feed is either closed or subscribed.
     */
    @Override
    public final void statusCallback(final FeedStatusCallback<IESubscribeFeed> cb)
    {
        if (!mIsActive.get())
        {
            throw (
                new IllegalStateException("feed is inactive"));
        }

        if (mInPlace)
        {
            throw (
                new IllegalStateException(
                    "subscription in place"));
        }

        mStatusCallback = cb;

        return;
    } // end of statusCallback(FeedStatusCallback)

    /**
     * Puts the {@link MatchEvent} notification message callback
     * in place. If {@code cb} is not {@code null}, then
     * notification messages will be passed to {@code cb} rather
     * than
     * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}.
     * A {@code null cb} means that notification messages will be
     * passed to the
     * {@link ESubscriber#notify(ENotificationMessage, IESubscribeFeed)}
     * override.
     * @param cb pass notification messages back to application
     * via this callback.
     * @throws IllegalStateException
     * if this feed is either closed or subscribed.
     */
    @Override
    public final void notifyCallback(final NotifyCallback cb)
    {
        if (!mIsActive.get())
        {
            throw (
                new IllegalStateException("feed is inactive"));
        }

        if (mInPlace)
        {
            throw (
                new IllegalStateException(
                    "subscription in place"));
        }

        mNotifyCallback = cb;

        return;
    } // end of notifyCallback(NotifyCallback)

    /**
     * Activates this event pattern feed by putting the
     * subordinate subscription feeds in place.
     * <p>
     * Nothing is done if already subscribed.
     * </p>
     * @throws IllegalStateException
     * if this feed is closed or if the client did not override
     * {@link ESubscriber} methods nor put the required callbacks
     * in place.
     *
     * @see #unsubscribe()
     * @see #close()
     */
    @Override
    public final void subscribe()
    {
        if (!mIsActive.get())
        {
            throw (
                new IllegalStateException("feed is inactive"));
        }

        // Do nothing if already subscribed.
        if (!mInPlace)
        {
            // If the feed state and/or notify callbacks are
            // not set, then set them to their defaults.
            if (mStatusCallback == null)
            {
                // Did the subscriber override feedStatus?
                if (!isOverridden(FEED_STATUS_METHOD,
                                  EFeedState.class,
                                  IESubscribeFeed.class))
                {
                    // No? Gotta do one or the other.
                    throw (
                        new IllegalStateException(
                            FEED_STATUS_METHOD +
                            " not overridden and statusCallback not set"));
                }

                // Yes. Use the override method.
                mStatusCallback =
                    ((ESubscriber) mEClient.target())::feedStatus;
            }

            if (mNotifyCallback == null)
            {
                // Did the subscriber override notify?
                if (!isOverridden(NOTIFY_METHOD,
                                  ENotificationMessage.class,
                                  IESubscribeFeed.class))
                {
                    // Nope. Not much point in putting this
                    // subscription is place.
                    throw (
                        new IllegalStateException(
                            NOTIFY_METHOD +
                            " not overridden and notifyCallback not set"));
                }

                // Yes. Use the override method.
                mNotifyCallback =
                    ((ESubscriber) mEClient.target())::notify;
            }

            if (sLogger.isLoggable(Level.FINEST))
            {
                final StringBuilder output = new StringBuilder();

                output.append(mPubKey)
                      .append(" subscribing to ")
                      .append(mSubFeeds.size())
                      .append(":");

                mSubFeeds.forEach(
                    feed ->
                        output.append('\n').append(feed));
            }
            else if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format(
                        "%s: subscribing to %d subordinate feeds.",
                        mPubKey,
                        mSubFeeds.size()));
            }

            // Subscribe to subordinate feeds.
            mSubFeeds.stream()
                     .map(
                         feed ->
                         {
                             feed.statusCallback(this::onFeedStateUpdate);
                             feed.notifyCallback(this::onEvent);

                             return (feed);
                         })
                     .forEachOrdered(ESubscribeFeed::subscribe);

            // Feed advertised and subordinate subcriptions are
            // now in place.
            mInPlace = true;
        }

        return;
    } // end of subscribe()

    @Override
    public final void unsubscribe()
    {
        // Is the feed in place?
        if (mInPlace)
        {
            // Yes.
            if (sLogger.isLoggable(Level.FINER))
            {
                sLogger.finer(
                    String.format("%s subscriber %d, feed %d: unsubscribing.",
                        mEClient.location(),
                        mEClient.clientId(),
                        mFeedId));
            }

            // Retract the subscriptions.
            mSubFeeds.forEach(ESubscribeFeed::unsubscribe);
        }

        return;
    } // end of unsubscribe()

    //
    // end of IESubscribeFeed Interface Implementation.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Event Handlers.
    //

    /**
     * Updates the feed bit in the subordinate feed mask. If the
     * feed state is up, then the feed bit is set; otherwise the
     * feed bit is cleared. If the composite subordinate feed
     * mask changes state, then this feed posts an update to its
     * publish state.
     * @param state subordinate feed's new state.
     * @param feed subordinate subscription feed.
     */
    private void onFeedStateUpdate(final EFeedState state,
                                   final IESubscribeFeed feed)
    {
        final long startMask = mSubFeedMask;
        final int feedId = feed.feedId();
        final long feedMask = (1L << feedId);
        boolean postUpdate = false;

        // Up or down?
        if (state == EFeedState.UP)
        {
            // Up. Set the feed's bit.
            mSubFeedMask |= feedMask;
        }
        // Down or unknown. Clear the feed's bit.
        else
        {
            mSubFeedMask &= ~(feedMask);
        }

        // Did the subordinate feed mask change?
        if (startMask == mSubFeedMask)
        {
            // No. no-op.
        }
        // Yes, it did change. Is the feed now completely up?
        else if (mSubFeedMask == mAllSubFeedsMask)
        {
            // Yes. Let the world know that this feed will now
            // start matching events.
            mFeedState = EFeedState.UP;
            postUpdate = true;
        }
        // Was the feed completely up?
        else if (startMask == mAllSubFeedsMask)
        {
            // Yes. Well now it is down.
            mFeedState = EFeedState.DOWN;
            postUpdate = true;
        }

        if (postUpdate)
        {
            EClient.dispatch(
                new StatusTask<>(
                    mFeedState, this, mStatusCallback),
                mEClient.target());
        }

        return;
    } // end of onFeedStateUpdate(EFeedState, ESubscribeFeed)

    /**
     * Runs {@code event} through the state machine if-and-only-if
     * all the subordinate feeds are up.
     * @param event inbound notification event.
     * @param feed event is from this subordinate feed.
     */
    private void onEvent(final ENotificationMessage event,
                         final IESubscribeFeed feed)
    {
        // Are all the subordinate feeds up?
        if (mSubFeedMask == mAllSubFeedsMask)
        {
            // Yes. Go head and run this event through the state
            // machine.
            matchEvent(event, feed.feedId());
        }
        // No, not all the subordinate feeds are up.
        // Ignore events until this happens.

        return;
    } // end of onEvent(ENotificationMessage, ESubscribeFeed)

    //
    // end of Event Handlers.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if this is an exclusive pattern feed.
     * Exclusive match events do not share events between
     * instances of the same pattern feed.
     * <p>
     * Events may be shared between different pattern feeds.
     * </p>
     * @return {@code true} if the match pattern is exclusive.
     */
    public final boolean isExclusive()
    {
        return (mIsExclusive);
    } // end of isExclusive()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Adds a mapping from the given event to the match frame
     * which now contains the event. Note: this method assumes
     * that the caller previously determined this is an exclusive
     * pattern feed.
     * @param event event added to the given match frame.
     * @param mf match frame containing the event.
     */
    protected final void addMapping(final ENotificationMessage event,
                                    final AbstractMatchFrame mf)
    {
        List<AbstractMatchFrame> frames = mEventFrames.get(event);

        if (frames == null)
        {
            frames = new ArrayList<>();
            mEventFrames.put(event, frames);
        }

        frames.add(mf);

        return;
    } // end of addMapping(ENotification, AbstractMatchFrame)

    /**
     * Marks all of the match frames associated with the given
     * events as defunct. Note: this method assumes that the
     * caller previously determined this is an exclusive
     * pattern feed. Also removes each of the listed events from
     * the event-to-match frame map.
     * <p>
     * Marking a match frame as defunct allows frames to be
     * lazily removed from the frames list when encountered
     * during the match process.
     * </p>
     * @param events the events comprising a completely matched,
     * exclusive pattern.
     */
    protected final void markDefunct(final List<ENotificationMessage> events)
    {
        List<AbstractMatchFrame> frames;

        for (ENotificationMessage event : events)
        {
            if ((frames = mEventFrames.remove(event)) != null)
            {
                for (AbstractMatchFrame frame : frames)
                {
                    frame.setDefunct();
                }
            }
        }

        return;
    } // end of markDefunct(List<>)

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    /**
     * Opens either an
     * {@link EOrderedPatternFeed#open(ESubscriber, EventPattern) ordered pattern}
     * or
     * {@link EUnorderedPatternFeed#open(ESubscriber, EventPattern) unordered pattern}
     * based on {@link EventPattern#patternType() pattern type}.
     * @param client forward {@link MatchEvent}s to this
     * subscriber.
     * @param pattern event pattern definition.
     * @return event pattern feed instance.
     */
    public static EPatternFeed open(final ESubscriber client,
                                    final EventPattern pattern)
    {
        Objects.requireNonNull(client, "client is null");

        final PatternType pType =
            Objects.requireNonNull(
                pattern, "pattern is null").patternType();
        final EPatternFeed retval;

        switch (pType)
        {
            case ORDERED:
                retval =
                    EOrderedPatternFeed.openFeed(client, pattern);
                break;

            default:
                retval =
                    EUnorderedPatternFeed.openFeed(client, pattern);
        }

        return (retval);
    } // end of open(ESubscriber, String, EventPattern)

    protected static boolean componentTest(final ENotificationMessage e,
                                           final AbstractMatchFrame mf,
                                           final MatchCondition mc)
    {
        boolean retcode;

        try
        {
            final Map<String, List<ENotificationMessage>> groups =
                mf.groupMap();
            final Map<Object, Object> userCache = mf.userCache();

            retcode = mc.test(e, groups, userCache);
        }
        catch (Throwable tex)
        {
            sLogger.log(Level.WARNING,
                        NOTIFY_METHOD,
                        String.format(
                            "%s match condition exception",
                            mf.mPatternName));

            retcode = false;
        }

        return (retcode);
    } // end of componentTest(ENotificationMessage, MatchFrame)

    /**
     * Returns the next unique match frame identifier.
     * @return match frame identifier.
     */
    private static int nextFrameIdentifier()
    {
        int retval = sNextFrameId;

        ++sNextFrameId;

        return (retval);
    } // end of nextFrameIdentifier()
} // end of class EPatternFeed
