//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2018. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.feed;

import java.io.Serializable;
import java.util.Collections;
import java.util.Formatter;
import java.util.List;
import java.util.Map;
import net.sf.eBus.messages.ELocalOnly;
import net.sf.eBus.messages.ENotificationMessage;


/**
 * A match event reports an event pattern match. Contains the
 * <a href="CapturingGroups">capturing group map</a> and the
 * <a href="UserCache">user-defined data cache</a>.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

@ELocalOnly
public final class MatchEvent
    extends ENotificationMessage
    implements Serializable
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * Serialization version identifier.
     */
    private static final long serialVersionUID = 0x050200L;

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The collected events for each pattern group. Pattern
     * group {@link EventPattern#ALL_EVENTS} contains all matched
     * events. The remaining map keys (if any) are from the
     * user-defined {@link EventPattern}. The groups map and
     * the notification list values are both read-only.
     */
    public final Map<String, List<ENotificationMessage>> groups;

    /**
     * This map contains the user-defined data stored when
     * {@link MatchCondition}s were executed. This map is not
     * {@code null} but may be empty.
     */
    public final Map<Object, Object> userCache;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Create a new match event for the given pattern groups. The
     * timestamp is set the current millisecond epoch time.
     * @param subject subject used to open pattern feed.
     * @param groups matched pattern groups.
     * @param userCache map containing user-defined data. This
     * map is never {@code null} but may be empty.
     */
    public MatchEvent(final String subject,
                      final Map<String, List<ENotificationMessage>> groups,
                      final Map<Object, Object> userCache)
    {
        this (subject,
              System.currentTimeMillis(),
              groups,
              userCache);
    } // end of MatchEvent(String, Map, Map)

    /**
     * Creates a new match event for the given subject and
     * timestamp. This event has an empty groups map.
     * @param subject notification subject.
     * @param timestamp millisecond epoch time for event.
     */
    @SuppressWarnings ("unchecked")
    public MatchEvent(final String subject,
                      final long timestamp)
    {
        this (subject,
              timestamp,
              Collections.emptyMap(),
              Collections.emptyMap());
    } // end of MatchEvent(String, long)

    /**
     * Creates a new match event for the given subject, timestamp,
     * and pattern groups.
     * @param subject notification subject. Set by user when
     * opening {@link EPatternFeed}.
     * @param timestamp millisecond epoch time for event.
     * @param groups pattern groups containing zero or more
     * matched events.
     * @param userCache map containing user-defined data. This
     * map is never {@code null} but may be empty.
     */
    public MatchEvent(final String subject,
                      final long timestamp,
                      final Map<String, List<ENotificationMessage>> groups,
                      final Map<Object, Object> userCache)
    {
        super (subject, timestamp);

        this.groups = groups;
        this.userCache = userCache;
    } // end of MatchEvent(String, long, Map)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Object Method Overrides.
    //

    /**
     * Returns a string containing each of the groups and the
     * group event.
     * @return textual representation of the match event.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public String toString()
    {
        int ei;
        final Formatter retval = new Formatter();

        retval.format("%s", super.toString());
        retval.format("%n  Capturing groups:");

        for (Map.Entry<String, List<ENotificationMessage>> entry :
                 groups.entrySet())
        {
            retval.format("%n  group[%s]", entry.getKey());

            ei = 0;
            for (ENotificationMessage msg : entry.getValue())
            {
                retval.format("%n    event[%2d]%n%s", ei, msg);
                ++ei;
            }
        }

        retval.format("%n%n  User-defined data:");
        userCache.entrySet()
                 .forEach(
                     entry -> retval.format("%n    %s=%s",
                                            entry.getKey(),
                                            entry.getValue()));

        return (retval.toString());
    } // end of toString()

    //
    // end of Object Method Overrides.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns the notification list associated with the given
     * group name. May return {@code null} if there is not group
     * with the given name.
     * @param name group name.
     * @return group event list.
     */
    public List<ENotificationMessage> group(final String name)
    {
        return (groups.get(name));
    } // end of group(String)

    //
    // end of Get Methods.
    //-----------------------------------------------------------
} // end of class MatchEvent
