//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Formatter;
import static net.sf.eBus.messages.type.DataType.STRING_TYPE;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.time.ZonedDateTime} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class ZonedDateTimeType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code ZonedDateTime} data type instance.
     */
    /* package */ ZonedDateTimeType()
    {
        super (ZonedDateTime.class, true, VARIABLE_SIZE, null);
    } // end of ZonedDateTimeType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code ZonedDateTime.toLocalDate} and
     * {@code ZonedDateTime.toLocalTime} values to
     * {@code buffer}.
     * @param o a {@code ZonedDateTime} instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain enough space to store
     * the date and time values.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof ZonedDateTime)
        {
            final ZonedDateTime zdt = (ZonedDateTime) o;

            buffer.putLong((zdt.toLocalDate()).toEpochDay())
                  .putLong((zdt.toLocalTime()).toNanoOfDay());
            STRING_TYPE.serialize(
                (zdt.getZone()).getId(), buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the de-serialized {@code ZonedDateTime} value by
     * converting the encode {@code long ZonedDateTime.ofNanoOfDay}
     * value back to a {@code ZonedDateTime} using
     * {@link ZonedDateTime#of(LocalDate, LocalTime, ZoneId)}.
     * @param buffer contains the serialized {@code ZonedDateTime}.
     * @return the de-serialized {@code ZonedDateTime}.
     * @throws BufferUnderflowException
     * if {@code buffer} does not contain the complete
     * {@code ZonedDateTime} value.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final LocalDate ld =
            LocalDate.ofEpochDay(buffer.getLong());
        final LocalTime lt =
            LocalTime.ofNanoOfDay(buffer.getLong());
        final ZoneId zoneId =
            ZoneId.of((String) STRING_TYPE.deserialize(buffer));

        return (ZonedDateTime.of(ld, lt, zoneId));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a
     * {@code ZonedDateTime} instance to a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code ZonedDateTime} message field
     * name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putLong((%s.toLocalDate()).toEpochDay())%n",
            indent,
            fieldName);
        output.format(
            "%s      .putLong((%s.toLocalTime()).toNanoOfDay());%n",
            indent,
            fieldName);
        STRING_TYPE.createSerializer(
            field,
            String.format(
                "(((%s).getZone()).getId())", fieldName),
            indent,
            output);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a
     * {@code ZonedDateTime} instance from a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code ZonedDateTime} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.ZonedDateTime.of(ld, lt, java.time.ZoneId.of(s)));%n" :
             "%s%s = java.time.ZonedDateTime.of(ld, lt, java.time.ZoneId.of(s));%n");

        output.format("%sfinal java.time.LocalDate ld = java.time.LocalDate.ofEpochDay(buffer.getLong());%n",
                      indent)
              .format("%sfinal java.time.LocalTime lt = java.time.LocalTime.ofNanoOfDay(buffer.getLong());%n",
                      indent)
              .format("%sfinal String s;%n%n", indent);
        STRING_TYPE.createDeserializer(
            field, "s", indent, output, false);
        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class ZonedDateTimeType
