//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.ZoneOffset;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.time.ZoneOffset} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class ZoneOffsetType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A {@link ZoneOffset} requires {@value} bytes for encoding.
     */
    public static final int BYTES = 8;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code ZoneOffset} data type instance.
     */
    /* package */ ZoneOffsetType()
    {
        super (ZoneOffset.class, true, BYTES, null);
    } // end of ZoneOffsetType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code ZoneOffset.getTotalSeconds} value to
     * {@code buffer}.
     * @param o a {@code ZoneOffset} instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain enough space to store
     * the {@code int} total seconds value.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof ZoneOffset)
        {
            buffer.putLong(((ZoneOffset) o).getTotalSeconds());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the de-serialized {@code ZoneOffset} value by
     * converting the encoded
     * {@code int ZoneOffset.getTotalSeconds} value back to a
     * {@code ZoneOffset} using
     * {@link ZoneOffset#ofTotalSeconds(int)}.
     * @param buffer contains the serialized {@code ZoneOffset}.
     * @return the de-serialized {@code ZoneOffset}.
     * @throws BufferUnderflowException
     * if {@code buffer} does not contain the complete
     * {@code int} total seconds value.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (ZoneOffset.ofTotalSeconds(buffer.getInt()));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a {@code ZoneOffset}
     * instance to a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code ZoneOffset} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putInt(%s.getTotalSeconds());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a
     * {@code ZoneOffset} instance from a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code ZoneOffset} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.ZoneOffset.ofTotalSeconds(buffer.getInt()));%n" :
             "%s%s = java.time.ZoneOffset.ofTotalSeconds(buffer.getInt());%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class ZoneOffsetType
