//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.ZoneId;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class ZoneIdType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code ZoneId} data type instance.
     */
    /* package */ ZoneIdType()
    {
        super (ZoneId.class, true, VARIABLE_SIZE, null);
    } // end of ZoneIdType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes this URI instance to the given buffer.
     * @param o a URI instance.
     * @param buffer serialize to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not have available space to store
     * the serialized URI.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof ZoneId)
        {
            STRING_TYPE.serialize(((ZoneId) o).getId(), buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    @Override
    public Object deserialize(ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (
            ZoneId.of((String) STRING_TYPE.deserialize(buffer)));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the URI instance to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        STRING_TYPE.createSerializer(
            field,
            String.format("((%s).getId())", fieldName),
            indent,
            output);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the URI instance
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.ZoneId.of(s));%n" :
             "%s%s = java.time.ZoneId.of(s);%n");

        output.format("%sfinal String s;%n", indent);
        STRING_TYPE.createDeserializer(
            field, "s", indent, output, false);
        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class ZoneIdType
