//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2013, 2014, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.net.URI;
import java.net.URISyntaxException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization of a {@link java.net.URI}
 * instance. Transmits the URI as text.
 *
 * @author crapp
 */

public final class UriType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new URI type instance. Considered an eBus
     * built-in type and an SQL {@code varchar}.
     */
    /* package */ UriType()
    {
        super (URI.class, false, VARIABLE_SIZE, null);
    } // end of UriType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes this URI instance to the given buffer.
     * @param o a URI instance.
     * @param buffer serialize to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not have available space to store
     * the serialized URI.
     * @see #deserialize(ByteBuffer)
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof URI)
        {
            STRING_TYPE.serialize(o.toString(), buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    @Override
    public Object deserialize(ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final String s =
            (String) STRING_TYPE.deserialize(buffer);
        Object retval = null;

        if (!s.isEmpty())
        {
            try
            {
                retval = new URI(s);
            }
            catch (URISyntaxException synex)
            {
                // Ignore and return null.
            }
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the URI instance to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sfinal String s = (%s).toString();%n",
            indent,
            fieldName);
        STRING_TYPE.createSerializer(field, "s", indent, output);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the URI instance
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;
        final String indent2 = indent1 + INDENT;
        final String format0 =
            (useBuilder ?
             "%sbuilder.%s(new java.net.URI(s));%n" :
             "%s%s = new java.net.URI(s);%n");
        final String format1 = (useBuilder ?
                                "%sbuilder.%s(null);%n" :
                                "%s%s = null;%n");

        output.format("%sfinal String s;%n", indent);
        STRING_TYPE.createDeserializer(
            field, "s", indent, output, false);
        output.format("%sif (s.isEmpty()) {%n", indent)
              .format((useBuilder ?
                       "%sbuilder.%s(null);%n" :
                       "%s%s = null;%n"),
                      indent1,
                      fieldName)
              .format("%s} else {%n", indent)
              .format("%stry {%n", indent1)
              .format(format0, indent2, fieldName)
              .format("%s} catch (java.net.URISyntaxException synex) {%n",
                      indent1)
              .format(format1, indent2, fieldName)
              .format("%s}%n", indent1)
              .format("%s}%n", indent);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class UriType
