//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2004 - 2008, 2011, 2013, 2019. Charles W. Rapp.
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.lang.String} objects. A serialized string has two
 * parts: length and byte array where length is the byte array
 * size. The byte array is created by
 * {@link String#getBytes(java.nio.charset.Charset)} using
 * {@link #CHARSET}.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class StringType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates the String type object. Considered a
     * built-in type and an SQL {@code varchar}.
     */
    /* package */ StringType()
    {
        super (String.class, true, VARIABLE_SIZE, null);
    } // end of StringType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes this string to the given {@code buffer}.
     * @param o a {@code String} object.
     * @param buffer write to this buffer.
     * @exception BufferOverflowException
     * if {@code buffer} overflows or the string length exceeds
     * {@link #MAX_STRING_LENGTH}.
     * @see #deserialize(ByteBuffer)
     * @see #serialize(String, int, ByteBuffer)
     */
    @Override
    public void serialize(final Object o, final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof String)
        {
            serialize((String) o, MAX_STRING_LENGTH, buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the deserialized {@code String} in
     * {@code buffer}.
     * @param buffer Deserialize the string object from this
     * {@code ByteBuffer}.
     * @return the deserialized {@code String}.
     * @exception BufferUnderflowException
     * if {@code buffer} does not contain enough bytes for the
     * string.
     *
     * @see #serialize(java.lang.Object, java.nio.ByteBuffer)
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int size = buffer.getShort();
        Object retval = EMPTY_STRING;

        if (size > 0)
        {
            final byte[] b = new byte[size];

            buffer.get(b, 0, size);
            retval = new String(b, CHARSET);
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the string instance
     * to a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        createSerializer(
            field, fieldName, indent, "MAX_STRING_LENGTH", output);
        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the string instance
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;
        final String format =
            (useBuilder ?
             "%sbuilder.%s(new String(b, charset));%n" :
             "%s%s = new String(b, charset);%n");

        output.format("%sfinal java.nio.charset.Charset charset = net.sf.eBus.messages.type.DataType.findCharset(\"%s\");%n",
                      indent,
                      field.charset())
              .format("%sfinal int stringSize = buffer.getShort();%n%n",
                      indent)
              .format("%sif (stringSize == 0) {%n", indent)
              .format((useBuilder ?
                       "%sbuilder.%s(EMPTY_STRING);%n" :
                       "%s%s = EMPTY_STRING;%n"),
                      indent1,
                      fieldName)
              .format("%s} else {%n", indent)
              .format("%sfinal byte[] b = new byte[stringSize];%n",
                      indent1)
              .format("%sbuffer.get(b, 0, stringSize);%n", indent1)
              .format(format, indent1, fieldName)
              .format("%s}%n", indent);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------

    /**
     * Serializes the given string to the buffer, checking if
     * the string length is &le; {@code maxSize}. This method
     * is used by those {@link DataType}s which serialize their
     * objects as text.
     * @param s serialize this string to {@code buffer}.
     * @param maxSize the maximum allowed string length.
     * @param buffer write the string to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} overflows or the string length exceeds
     * {@code maxSize}.
     */
    /* package */ void serialize(final String s,
                                 final int maxSize,
                                 final ByteBuffer buffer)
        throws BufferOverflowException
    {
        final int length = s.length();

        if (length > maxSize)
        {
            final BufferOverflowException bufex =
                new BufferOverflowException();

            bufex.initCause(
                new IllegalArgumentException(
                    String.format(
                        "string length %,d exceeds allowed maximum %,d",
                        length,
                        maxSize)));

            throw (bufex);
        }
        else if (length == 0)
        {
            buffer.putShort((short) 0);
        }
        else
        {
            final byte[] b = s.getBytes(CHARSET);

            // First output the string's length and then the
            // string bytes.
            buffer.putShort((short) b.length)
                  .put(b);
        }

        return;
    } // end of serialize(String, int, ByteBuffer)

    /**
     * Generates the Java code to serialize a {@code String}
     * instance for the given field name and maximum string
     * length constant.
     * @param field message field.
     * @param fieldName the string field name.
     * @param indent indent the code by this amount.
     * @param maxConst the maximum string length constant name.
     * @param output append the code to this formatter.
     */
    /* package */ void createSerializer(final MessageField field,
                                        final String fieldName,
                                        final String indent,
                                        final String maxConst,
                                        final Formatter output)
    {
        final String indent1 = indent + INDENT;

        output.format("%sjava.nio.charset.Charset charset = net.sf.eBus.messages.type.DataType.findCharset(\"%s\");%n",
                      indent,
                      field.charset())
              .format("%sfinal int length = (%s).length();%n%n",
                      indent,
                      fieldName)
              .format("%sif (length > %s) {%n", indent, maxConst)
              .format("%sjava.nio.BufferOverflowException bufex = new java.nio.BufferOverflowException();%n",
                      indent1)
              .format("%sbufex.initCause(new IllegalArgumentException(\"string length exceeds %s\"));%n",
                      indent1,
                      maxConst)
              .format("%sthrow (bufex);%n", indent1)
              .format("%s} else if (length == 0) {%n", indent)
              .format("%sbuffer.putShort((short) 0);%n", indent1)
              .format("%s} else {%n", indent)
              .format("%sfinal byte[] b = %s.getBytes(charset);%n",
                      indent1,
                      fieldName);
        output.format(
            "%sbuffer.putShort((short) b.length)%n", indent1);
        output.format(
            "%s      .put(b);%n", indent1);
        output.format("%s}%n", indent);

        return;
    } // end of createSerializer(String,String,String,Formatter)
} // end of class StringType
