//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.LocalTime;
import java.time.OffsetTime;
import java.time.ZoneOffset;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.time.OffsetTime} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class OffsetTimeType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A {@link OffsetTime} requires {@value} bytes for encoding.
     */
    public static final int BYTES = 12;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code OffsetTime} data type instance.
     */
    public OffsetTimeType()
    {
        super (OffsetTime.class, true, BYTES, null);
    } // end of OffsetTimeType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code OffsetTime} local time and offset
     * values to {@code buffer}.
     * @param o a {@code OffsetTime} instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain enough space to store
     * the {@code OffsetTime} value.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof OffsetTime)
        {
            final OffsetTime ot = (OffsetTime) o;

            buffer.putLong((ot.toLocalTime()).toNanoOfDay())
                  .putInt((ot.getOffset()).getTotalSeconds());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the de-serialized {@code OffsetTime} value by
     * converting the encoded {@code LocalTime} and
     * {@code ZoneOffset} values back to a {@code OffsetTime}
     * using {@link OffsetTime#of(LocalTime, ZoneOffset)}.
     * @param buffer contains the serialized {@code OffsetTime}.
     * @return the de-serialized {@code OffsetTime}.
     * @throws BufferUnderflowException
     * if {@code buffer} does not contain the complete
     * {@code OffsetTime} value.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (
            OffsetTime.of(
                LocalTime.ofNanoOfDay(buffer.getLong()),
                ZoneOffset.ofTotalSeconds(buffer.getInt())));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a {@code OffsetTime}
     * instance to a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code OffsetTime} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putLong((%s.toLocalTime()).toNanoOfDay())%n",
            indent,
            fieldName);
        output.format(
            "%s      .putInt((%s.getOffset()).getTotalSeconds());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a
     * {@code OffsetTime} instance from a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code OffsetTime} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.OffsetTime.of(java.time.LocalTime.ofNanoOfDay(buffer.getLong()), java.time.ZoneOffset.ofTotalSeconds(buffer.getInt())));%n" :
             "%s%s = java.time.OffsetTime.of(java.time.LocalTime.ofNanoOfDay(buffer.getLong()), java.time.ZoneOffset.ofTotalSeconds(buffer.getInt()));%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class OffsetTimeType
