//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.time.OffsetDateTime} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class OffsetDateTimeType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A {@link OffsetDateTime} requires {@value} bytes for
     * encoding.
     */
    public static final int BYTES = 20;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code OffsetDateTime} data type instance.
     */
    /* package */ OffsetDateTimeType()
    {
        super (OffsetDateTime.class, true, BYTES, null);
    } // end of OffsetDateTimeType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code OffsetDateTime.toLocalDate},
     * {@code LocalDateTime.toLocalTime}, and
     * {@code LocalDateTime.getOffset} values to {@code buffer}.
     * @param o an {@code OffsetDateTime} instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain enough space to store
     * the date, time, and offset values.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof OffsetDateTime)
        {
            final OffsetDateTime odt = (OffsetDateTime) o;

            buffer.putLong((odt.toLocalDate()).toEpochDay())
                  .putLong((odt.toLocalTime()).toNanoOfDay())
                  .putInt((odt.getOffset()).getTotalSeconds());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the de-serialized {@code OffsetDateTime} value by
     * converting the encoded local date, local time, and zone
     * offset values back to an {@code OffsetDateTime} using
     * {@link OffsetDateTime#of(LocalDate, LocalTime, ZoneOffset)}.
     * @param buffer contains the serialized
     * {@code OffsetDateTime}.
     * @return the de-serialized {@code OffsetDateTime}.
     * @throws BufferUnderflowException
     * if {@code buffer} does not contain the complete
     * {@code OffsetDateTime} value.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (
            OffsetDateTime.of(
                LocalDate.ofEpochDay(buffer.getLong()),
                LocalTime.ofNanoOfDay(buffer.getLong()),
                ZoneOffset.ofTotalSeconds(buffer.getInt())));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a
     * {@code OffsetDateTime} instance to a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code OffsetDateTime} message field
     * name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putLong((%s.toLocalDate()).toEpochDay())%n",
            indent,
            fieldName);
        output.format(
            "%s      .putLong((%s.toLocalTime()).toNanoOfDay())%n",
            indent,
            fieldName);
        output.format(
            "%s      .putInt((%s.getOffset()).getTotalSeconds());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a
     * {@code OffsetDateTime} instance from a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code OffsetDateTime} message field
     * name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.OffsetDateTime.of(java.time.LocalDate.ofEpochDay(buffer.getLong()), java.time.LocalTime.ofNanoOfDay(buffer.getLong()), java.time.ZoneOffset.ofTotalSeconds(buffer.getInt())));%n" :
             "%s%s = java.time.OffsetDateTime.of(java.time.LocalDate.ofEpochDay(buffer.getLong()), java.time.LocalTime.ofNanoOfDay(buffer.getLong()), java.time.ZoneOffset.ofTotalSeconds(buffer.getInt()));%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class OffsetDateTimeType
