//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.MonthDay;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.time.MonthDay} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class MonthDayType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A {@link MonthDay} requires {@value} bytes for encoding.
     */
    public static final int BYTES = 8;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code MonthDay} data type instance.
     */
    /* package */ MonthDayType()
    {
        super (MonthDay.class, true, BYTES, null);
    } // end of MonthDayType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code MonthDay.getMonthValue} and
     * {@code MonthDay.getDayOfMonth} values to
     * {@code buffer}.
     * @param o a {@code MonthDay} instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain enough space to store
     * the {@code MonthDay} value.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof MonthDay)
        {
            final MonthDay md = (MonthDay) o;

            buffer.putInt(md.getMonthValue())
                  .putInt(md.getDayOfMonth());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the de-serialized {@code MonthDay} value by
     * converting the encoded month and day-of-month values
     * back to a {@code MonthDay} using
     * {@link MonthDay#of(int, int)}.
     * @param buffer contains the serialized {@code MonthDay}.
     * @return the de-serialized {@code MonthDay}.
     * @throws BufferUnderflowException
     * if {@code buffer} does not contain the complete
     * {@code MonthDay} value.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (MonthDay.of(buffer.getInt(), buffer.getInt()));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a {@code MonthDay}
     * instance to a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code MonthDay} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%1$sbuffer.putInt(%2$s.getMonthValue()).putInt(%2$s.getDayOfMonth());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a
     * {@code MonthDay} instance from a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code MonthDay} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.MonthDay.of(buffer.getInt(), buffer.getInt()));%n" :
             "%s%s = java.time.MonthDay.of(buffer.getInt(), buffer.getInt());%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class MonthDayType
