//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013 - 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import com.google.common.base.Strings;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import javax.annotation.Nullable;
import net.sf.eBus.messages.ELocalOnly;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.EReplyInfo;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.messages.EStringInfo;
import net.sf.eBus.messages.InvalidMessageException;
import net.sf.eBus.messages.UnknownMessageException;
import net.sf.eBus.messages.ValidationException;

/**
 * This class provides binary serialization/de-serialization for
 * a unique {@link EMessageObject} sub-class. The serialized
 * binary format is:
 * <ul>
 *   <li>
 *     {@code EMessageObject} class name (serialized as a
 *     {@link StringType String}).
 *   </li>
 *   <li>
 *     4-byte, signed integer field mask (note this restricts
 *     eBus {@link net.sf.eBus.messages.EMessage} and
 *     {@link net.sf.eBus.messages.EField} classes to 31 fields.
 *   </li>
 *   <li>
 *     Each field serialized in the order specified by
 *     field size and field name. Serialization is done based
 *     on the field class.
 *   </li>
 * </ul>
 * This eBus data type is able to serialize both its specified
 * {@code EMessageObject} class or its sub-class. This allows a
 * {@code EMessage} and {@code EField} to use an abstract class
 * extending {@code EField} as a field and initializing that
 * field with a concrete sub-class. This allows the message/field
 * to support new information without changing the message/field
 * class.
 * <p>
 * <strong>Note:</strong> as of eBus 4.4.0, messages are
 * automatically compiled when first used for serialization
 * or de-serialization. This increases the time it takes to
 * send or receive a message the first time. When a message
 * is frequently transmitted, it is recommended that the
 * message class is "compiled" by calling
 * {@link DataType#findType(java.lang.Class)} for that
 * message class. So when a message is sent or received the
 * first time, the message compilation is already done.
 * </p>
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public abstract class MessageType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * {@link EMessageObject} concrete subclass must have a
     * {@code public static} method named {@value} which takes
     * no arguments and returns an object which extends
     * {@link EMessageObject.Builder}.
     */
    public static final String BUILDER_METHOD = "builder";

    /**
     * {@link EMessageObject} subclasses must have an inner
     * class names {@value} <em>if</em> the subclass has any
     * fields.
     */
    public static final String BUILD_INNER_CLASS = "Builder";

    /**
     * The "builder" method must be public and static.
     */
    public static final int PUBLIC_STATIC =
        (Modifier.PUBLIC | Modifier.STATIC);

    /**
     * Message fields must be public and final.
     */
    public static final int PUBLIC_FINAL =
        (Modifier.PUBLIC | Modifier.FINAL);

    /**
     * Ignore the {@value} message field.
     */
    public static final String SUBJECT_FIELD = "subject";

    //-----------------------------------------------------------
    // Statics.
    //

    /**
     * Used to sort message fields by name. This sorting
     * guarantees that message fields will be sorted exactly the
     * same across JVMs.
     */
    private static final Comparator<MessageType.MessageField> sNameComparator =
        (f0, f1) -> (f0.name()).compareTo(f1.name());

    /**
     * Used to sort message fields by length.
     */
    private static final Comparator<MessageType.MessageField> sSizeComparator =
        (f0, f1) ->
        {
            final DataType d0 = f0.dataType();
            final DataType d1 = f1.dataType();
            final int d0Size = d0.size();
            final int d1Size = d1.size();
            int retval;

            // Is either field of variable length?
            // If so then place the variable length field last.
            if (d0Size == DataType.VARIABLE_SIZE)
            {
                retval = 1;
            }
            else if (d1Size == DataType.VARIABLE_SIZE)
            {
                retval = -1;
            }
            // Both fields are fixed size.
            // Do the fields have equal sizes?
            else
            {
                retval = (d1Size - d0Size);
            }

            return (retval);
        };

    //-----------------------------------------------------------
    // Locals.
    //

    /**
     * The message field names and types. The list ordering
     * defines serialization order and, therefore, is
     * significant. That is why a {@code map} cannot be used to
     * match field names to field instances.
     */
    protected final List<MessageField> mFields;

    /**
     * If this a request message type, then this array specifies
     * the reply messages which may be sent in reply to the
     * request class.
     */
    protected final List<Class<? extends EReplyMessage>> mReplyClasses;

    /**
     * {@link EMessage} has a public {@code subject} field but
     * this field is <em>not</em> serialized because it is
     * include in the message key identifier. But it must be
     * included in
     * {@link EMessage#EMessage(java.lang.String, long)}
     * constructor. This problem is resolved setting the
     * subject via {@link #subject(String)} before de-serializing
     * the message.
     */
    protected String mSubject;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new message type instance for the given class,
     * message fields, and constructor method handle.
     * @param clazz the message class instance.
     * @param fields the message fields.
     * @param replies reply classes associated with a request
     * message class. Will be an empty array for non-request
     * message types.
     */
    protected MessageType(final Class<?> clazz,
                          final List<MessageField> fields,
                          final List<Class<? extends EReplyMessage>> replies)
    {
        super (clazz, false, VARIABLE_SIZE, null);

        mFields = fields;
        mReplyClasses = replies;
        mSubject = null;
    } // end of MessageType(...)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Get Methods.
    //

    /**
     * Returns {@code true} if {@code clazz} is listed in the
     * allowed reply message class list and {@code false} if not.
     * In short, returns {@code true} if {@code clazz} message
     * may be sent in reply to this request message type.
     * @param clazz the reply message class.
     * @return {@code true} if {@code clazz} message
     * may be sent in reply to this request message type.
     */
    public boolean isValidReply(final Class<?> clazz)
    {
        final Iterator<Class<? extends EReplyMessage>> cit =
            mReplyClasses.iterator();
        boolean retcode = false;

        while (cit.hasNext() && !retcode)
        {
            retcode = (clazz.equals(cit.next()));
        }

        return (retcode);
    } // end of isValidReply(Class<?>)

    /**
     * Returns the reply classes associated with a request
     * message class. Returns a copy of the reply class list. If
     * this message type is not a request, then returns an empty
     * list.
     * @return reply class list.
     */
    public List<Class<? extends EReplyMessage>> replyTypes()
    {
        return (mReplyClasses);
    } // end of replyTypes()

    /**
     * Returns the message object's field values as an
     * {@code Object} list. The returned list values are in
     * the same order as {@link #fields()}.
     * @param message extract the field values from this message
     * object.
     * @return message field values.
     * @throws Throwable
     * if any sort of error occurs when extracting the values
     * from {@code message}.
     */
    public List<Object> values(final EMessageObject message)
        throws Throwable
    {
        final List<Object> retval =
            new ArrayList<>(mFields.size());

        for (MessageField field : mFields)
        {
            retval.add(
                (field.field()).invokeWithArguments(message));
        }

        return (retval);
    } // end of values(EMessageObject)

    /**
     * Returns the number of fields in this {@code EMessage}
     * type.
     * @return number of message fields.
     */
    public int numberFields()
    {
        return (mFields.size());
    } // end of numberFields()

    /**
     * Returns the message object public fields.
     * @return message object fields.
     */
    public List<MessageField> fields()
    {
        return (mFields);
    } // end of fields()

    //
    // end of Get Methods.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Set Methods.
    //

    /**
     * Sets the message subject to the given value. Must be
     * called prior to deserializing the message.
     * @param subject de-serialized message subject.
     * @throws IllegalArgumentException
     * if {@code subject} is {@code null} or empty.
     */
    public void subject(final String subject)
        throws IllegalArgumentException
    {
        if (Strings.isNullOrEmpty(subject))
        {
            throw (
                new IllegalArgumentException(
                    "subject is null or empty"));
        }

        mSubject = subject;

        return;
    } // end of subject(String)

    //
    // end of Set Methods.
    //-----------------------------------------------------------

    /**
     * Writes the message object fields to the given buffer and
     * returns the resulting field mask.
     * <p>
     * As of eBus v. 4.4.0, all {@code EMessageObject} classes
     * are automatically "compiled" - meaning that an
     * {@code MessageType}-subclass is generated for any given
     * {@code EMessageObject} subclass. The {@code MessageType}
     * subclass overrides this {@code serializeFields} method,
     * meaning this method implementation is never called.
     * Therefore, this method implementation has an empty body
     * and always returns zero.
     * </p>
     * @param o write this {@code EMessageObject}'s fields to
     * {@code buffer}. Note: {@code o} is not {@code null}.
     * @param buffer serialize fields to this buffer.
     * @return resulting field mask.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain sufficient space to
     * store the message.
     */
    protected int serializeFields(final Object o,
                                  final ByteBuffer buffer)
        throws BufferOverflowException
    {
        // This method implementation is always overridden and
        // never called. So return zero.
        return (0);
    } // end of serializeFields(Object, ByteBuffer)

    /**
     * Returns an {@code EMessageObject} de-serialized from the
     * given buffer.
     * <p>
     * As of eBus v. 4.4.0, all {@code EMessageObject} classes
     * are automatically "compiled" - meaning that an
     * {@code MessageType}-subclass is generated for any given
     * {@code EMessageObject} subclass. The {@code MessageType}
     * subclass overrides this {@code deserializeFields} method,
     * meaning this method implementation is never called.
     * Therefore, this method implementation has an empty body
     * and always returns {@code null}.
     * </p>
     * @param fieldMask the message field mask specifying which
     * fields are serialized and which are {@code null}.
     * @param buffer read in the message from this buffer.
     * @return the de-serialized message.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete message.
     * @throws UnknownMessageException
     * if the serialized message is unknown to this eBus.
     * @throws ValidationException
     * if the message is incorrectly serialized.
     */
    @SuppressWarnings ("unchecked")
    protected Object deserializeFields(final int fieldMask,
                                       final ByteBuffer buffer)
        throws BufferUnderflowException,
               UnknownMessageException,
               ValidationException
    {
        // This method implementation is always overridden and
        // never called. So return null.
        return (null);
    } // end of deserializeFields(int, ByteBuffer)

    /**
     * Returns the {@code public final}, non-{@code static}
     * fields (except {@link EMessage#subject}). The returned
     * fields are sorted by field name and field length with
     * longer fields preceding shorter fields. Two fields of the
     * same length are further sorted lexicographically by field
     * name.
     * @param mc message object subclass.
     * @return message fields.
     */
    protected static List<MessageField> findFields(final Class<? extends EMessageObject> mc)
    {
        final boolean messageFlag =
            (EMessage.class.isAssignableFrom(mc));
        final Field[] fields = mc.getFields();
        final int numFields = fields.length;
        final boolean localFlag =
            mc.isAnnotationPresent(ELocalOnly.class);
        int index;
        int modifiers;
        final List<MessageField> retval = new ArrayList<>();

        for (index = 0; index < numFields; ++index)
        {
            // Look for public, final, non-static fields
            // ignoring the subject field.
            modifiers = fields[index].getModifiers();

            // Ignore static and non-static, non-public fields.
            if (Modifier.isStatic(modifiers) ||
                !Modifier.isPublic(modifiers))
            {
                // no-op.
            }
            // This field is a non-static, public filed.
            // Is this field final?
            else if (!Modifier.isFinal(modifiers))
            {
                // No and that is a problem. Non-static message
                // fields must be private.
                throw (
                    new InvalidMessageException(
                        mc,
                        mc.getCanonicalName() +
                        "." +
                        fields[index].getName() +
                        " is public but not final"));
            }
            else if ((!messageFlag ||
                      !SUBJECT_FIELD.equals(
                          fields[index].getName())))
            {
                retval.add(loadField(index, fields[index], mc));
            }
        }

        // If this is a remote-capable message, then validate
        // the fields (there must be 31 or fewer remote-capable
        // fields) before sorting them into serialization order.
        // Remote-capable means that the ELocalOnly is *not*
        // present
        if (!localFlag)
        {
            validateFields(mc, retval);

            // First sort the message fields by name.
            retval.sort(sNameComparator);

            // Next sort the message fields by length.
            retval.sort(sSizeComparator);
        }

        return (retval);
    } // end of findFields(Class)

    /**
     * Puts the reply classes into the given collection, starting
     * with the root class down to the leaf class. This list
     * defines which reply message classes may be sent in reply
     * to the given request message class.
     * @param mc the request message class.
     * @param classes  the class name collection.
     * @throws InvalidMessageException
     * if {@code mc} is missing the required
     * {@link EReplyInfo @EReplyInfo} attribute.
     */
    /* package */ static void replyClasses(final Class<? extends EMessageObject> mc,
                                           final List<Class<? extends EReplyMessage>> classes)
        throws InvalidMessageException
    {
        @SuppressWarnings ("unchecked")
        final Class<? extends EMessageObject> parent =
            (Class<? extends EMessageObject>) mc.getSuperclass();
        final EReplyInfo replyInfo;

        // Does this reply class have the required @EReplyInfo
        // annotation?
        if (!mc.isAnnotationPresent(EReplyInfo.class))
        {
            // No. That is not acceptable.
            throw (
                new InvalidMessageException(
                    mc,
                    String.format(
                        "%s missing @EReplyInfo annotation",
                         mc.getName())));
        }

        replyInfo = mc.getAnnotation(EReplyInfo.class);

        // If the parent is not the topmost super class, put
        // the parent fields into the names list before the
        // child.
        if (parent != null && !parent.equals(EMessage.class))
        {
            replyClasses(parent, classes);
        }

        // Now put the child fields into the collection.
        classes.addAll(
            Arrays.asList(replyInfo.replyMessageClasses()));

        return;
    } // end of replyClass(Class, List)

    /**
     * Loads the specified message field layout.
     * @param index the field index.
     * @param name the field name.
     * @param mc the eBus message class.
     * @return the eBus message field layout.
     * @throws InvalidMessageException
     * if {@code name} is not a valid message field.
     */
    private static MessageField loadField(final int index,
                                          final Field field,
                                          final Class<? extends EMessageObject> mc)
        throws InvalidMessageException
    {
        final MethodHandles.Lookup lookup =
            MethodHandles.publicLookup();
        final Class<?> jType = field.getType();
        final String name = field.getName();
        DataType dataType;
        MethodHandle fh;
        String charset = getCharset(field);
        final boolean optFlag;

        try
        {
            optFlag = jType.isAnnotationPresent(Nullable.class);

            // Note: DataType.findType throws an
            //       IllegalArgumentException if jType is not a
            //       supported eBus type.
            dataType = DataType.findType(jType);
            fh = lookup.findGetter(mc, name, jType);
        }
        catch (NullPointerException |
               IllegalArgumentException argex)
        {
            throw (
                new InvalidMessageException(
                    mc,
                    String.format(
                        "%s.%s invalid type %s",
                            mc.getName(),
                            name,
                            (jType == null ?
                             "(no type)" :
                             jType.getName())),
                    argex));
        }
        catch (NoSuchFieldException |
               SecurityException |
               IllegalAccessException jex)
        {
            throw (
                new InvalidMessageException(
                    mc,
                    String.format(
                        "%s has no field \"%s\"",
                        mc.getName(),
                        name),
                    jex));
        }
        // Let InvalidMessageExceptions fall through.

        return (new MessageField(index,
                                 name,
                                 fh,
                                 dataType,
                                 charset,
                                 optFlag));
    } // end of loadField(int, String, Class)

    /**
     * Returns the character set name associated with this
     * field. If field is not a {@code String} or does not have
     * an {@code EStringInfo} annotation, then returns
     * {@link #CHARSET} name. Otherwise returns
     * {@link EStringInfo#charset()}.
     * @param field return the associated character set for this
     * field.
     * @return character set name.
     */
    private static String getCharset(final Field field)
    {
        final String retval;

        // Is this a string field?
        // If so, is EStringInfo defined?
        if (!(String.class.equals(field.getType())) ||
            !field.isAnnotationPresent(EStringInfo.class))
        {
            // No. So return the default charset name.
            retval = CHARSET.name();
        }
        else
        {
            final EStringInfo sInfo =
                field.getAnnotation(EStringInfo.class);

            retval = sInfo.charset();
        }

        return (retval);
    } // end of getCharset(Class)

    /**
     * Validates that all fields in this remote-capable message
     * are themselves remote-capable and that the field count
     * does not exceed the maximum allowed number.
     * @param mc fields belong to this message class.
     * @param fields validate these message fields.
     * @throws InvalidMessageException
     * if the message field count exceeds the maximum or there is
     * a local-only field.
     */
    private static void validateFields(final Class<? extends EMessageObject> mc,
                                       final List<MessageField> fields)
    {
        if (fields.size() > MAX_FIELDS)
        {
            throw (
                new InvalidMessageException(
                    mc,
                    String.format(
                        "%,d fields exceeds max allowed %,d",
                        fields.size(),
                        MAX_FIELDS)));
        }

        fields.stream()
              .filter(
                  field ->
                          ((field.javaType()).isAnnotationPresent(
                              ELocalOnly.class)))
              .forEachOrdered(
                  field ->
                  {
                      throw (
                          new InvalidMessageException(
                              mc,
                              String.format(
                                  "%s field is local-only",
                          field.name())));
                  });

        return;
    } // end of validateFields(List<>)

//---------------------------------------------------------------
// Inner classes.
//

    /**
     * Stores a field index, field name, class field, eBus data
     * type. Fields are indexed incrementally from zero to the
     * number of message fields - 1. The index is used to
     * maintain field ordering.
     * <p>
     * This class is immutable.
     * </p>
     *
     * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
     */
    public static final class MessageField
    {
    //-----------------------------------------------------------
    // Member data.
    //

        //-------------------------------------------------------
        // Locals.
        //

        /**
         * The field index.
         */
        private final int mIndex;

        /**
         * The field name.
         */
        private final String mName;

        /**
         * The field Java class.
         */
        private final MethodHandle mField;

        /**
         * The field data type.
         */
        private final DataType mDataType;

        /**
         * If {@link mDataType} is {@link StringType}, then this
         * is the character set used to serialize the text.
         * Will be {@code null} otherwise.
         */
        private final String mCharset;

        /**
         * Set to {@code true} if this field may be set to
         * {@code null}.
         */
        private final boolean mIsOptional;

        /**
         * Matching setter field in message builder class. This
         * member is not {@code final} because there is a
         * chicken-and-egg problem between creating the fields
         * and finding/validating the message builder. The latter
         * requires knowing the message fields so the builder
         * can be verified. So the message fields must be
         * collected first, then find the builder, and finally
         * set the message field setter.
         */
        private MethodHandle mSetter;

    //-----------------------------------------------------------
    // Member methods.
    //

        //-------------------------------------------------------
        // Constructors.
        //

        /**
         * Stores the layout field attributes.
         * @param index field index (&ge; zero)
         * @param name the field name.
         * @param field the message field.
         * @param eType eBus field data type.
         * @param charset name of {@code java.nio.charset.Charset}
         * class.
         * @param optFlag {@code true} if field may be set to
         * {@code null}.
         */
        public MessageField(final int index,
                            final String name,
                            final MethodHandle field,
                            final DataType eType,
                            final String charset,
                            final boolean optFlag)
        {
            mIndex = index;
            mName = name;
            mField = field;
            mDataType = eType;
            mCharset = charset;
            mIsOptional = optFlag;
        } // end of MessageField(...)

        //
        // end of Constructors.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Object Method Overrides.
        //

        /**
         * Returns {@code true} if {@code o} is a
         * non-{@code null MessageType.Field} with the same index
         * as {@code this} field.
         * @param o comparison object.
         * @return {@code true} if {@code o} equals {@code this}
         * layout field.
         */
        @Override
        public boolean equals(final Object o)
        {
            boolean retcode = (this == o);

            if (!retcode && o instanceof MessageField)
            {
                retcode = (mIndex == ((MessageField) o).mIndex);
            }

            return (retcode);
        } // end of equals(Object)

        /**
         * Returns the field index as the hash code.
         * @return the field index.
         */
        @Override
        public int hashCode()
        {
            return (mIndex);
        } // end of hashCode()

        /**
         * Returns the string representation of this message
         * field.
         * @return the string representation of this message
         * field.
         */
        @Override
        public String toString()
        {
            return (String.format("[%,d] %s %s",
                                  mIndex,
                                  mName,
                                  mDataType));
        } // end of toString()

        //
        // end of Object Method Overrides.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Get Methods.
        //

        /**
         * Returns the field index.
         * @return the field index.
         */
        public int index()
        {
            return (mIndex);
        } // end of index()

        /**
         * Returns the field name.
         * @return the field name.
         */
        public String name()
        {
            return (mName);
        } // end of name()

        /**
         * Returns the Java class field.
         * @return the Java class field.
         */
        public MethodHandle field()
        {
            return (mField);
        } // end of field()

        /**
         * Returns the Java class associated with this field.
         * @return the field Java class.
         */
        public Class<?> javaType()
        {
            return (mDataType.dataClass());
        } // end of javaType()

        /**
         * Returns the eBus data type.
         * @return the eBus data type.
         */
        public DataType dataType()
        {
            return (mDataType);
        } // end of dataType()

        /**
         * Returns {@code true} if this is an array field and
         * {@code false} otherwise.
         * @return {@code true} if this is an array field and
         * {@code false} otherwise.
         */
        public boolean isArray()
        {
            return (mDataType instanceof ArrayType);
        } // end of isArray()

        /**
         * Returns the associated character set used to serialize
         * a text field if this field is of {@link StringType}.
         * Otherwise returns {@code null}.
         * @return character set if field is string type and
         * {@code null} otherwise.
         */
        public String charset()
        {
            return (mCharset);
        } // end of charset()

        /**
         * Returns {@code true} if this field is option and,
         * therefore, may be set to {@code null}.
         * @return {@code true} if field is optional.
         */
        public boolean isOptional()
        {
            return (mIsOptional);
        } // end of isOptional()

        /**
         * Returns the message builder class setter method for
         * this field.
         * @return builder setter method.
         */
        public MethodHandle setter()
        {
            return (mSetter);
        } // end of setter()

        //
        // end of Get Methods.
        //-------------------------------------------------------

        //-------------------------------------------------------
        // Set Methods.
        //

        /**
         * Sets the builder setter method handle
         * post-construction.
         * @param setter builder setter method handle.
         */
        /* package */ void setter(final MethodHandle setter)
        {
            mSetter = setter;
            return;
        } // end of setter(MethodHandle)

        //
        // end of Get Methods.
        //-------------------------------------------------------
    } // end of class EField
} // end of class MessageType
