//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.EMessageList;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization for
 * {@link EMessageList}. The serialized message list format is:
 * <br>
 * <table border="1" cellpadding="3" cellspacing="3">
 *   <caption>EMessageList Serialization Layout</caption>
 *   <thead>
 *     <tr>
 *       <th>Position</th>
 *       <th>Size (bytes)</th>
 *       <th>Description</th>
 *     </tr>
 *   </thead>
 *   <tbody>
 *     <tr>
 *       <td>0</td>
 *       <td>2</td>
 *       <td>Message list size</td>
 *     </tr>
 *     <tr>
 *       <td>2</td>
 *       <td>m</td>
 *       <td>eBus message key (includes message class and subject)</td>
 *     </tr>
 *     <tr>
 *       <td>2+m</td>
 *       <td>n</td>
 *       <td>Each {@link EMessage} in list</td>
 *     </tr>
 *   </tbody>
 * </table>
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class MessageListType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new message list type instance.
     */
    /* package */ MessageListType()
    {
        super (EMessageList.class, true, VARIABLE_SIZE, null);
    } // end of MessageListType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code EMessageList} instance to
     * {@code buffer}. The serialize format is show in
     * {@code MessageListType} class documentation.
     * @param o a {@code EMessageList} instance.
     * @param buffer write output to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not have sufficient space to store
     * the serialized message list.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o != null)
        {
            final EMessageList ml = (EMessageList) o;
            final EMessageKey key = ml.key();
            final MessageType mt =
                (MessageType)
                    DataType.findType(key.messageClass());

            // Firstly, output the list size.
            buffer.putShort((short) ml.size());

            // Secondly, output the message key.
            KEY_TYPE.serialize(key, buffer);

            // Finally, output each message in turn using the
            // message type.
            ml.forEach(msg -> { mt.serialize(msg, buffer); });

            // Mark the message list as read-only 'cuz once
            // serialized, it cannot be modified.
            ml.setReadOnly();
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the message list instance de-serialized from the
     * given buffer.
     * @param buffer read in the message list instance from this
     * buffer.
     * @return de-serialized message list.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete message list.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int size = buffer.getShort();
        final EMessageKey key =
            (EMessageKey) KEY_TYPE.deserialize(buffer);
        final String subject = key.subject();
        EMessageList retval = null;

        if (size < 0)
        {
            final BufferUnderflowException bufex =
                new BufferUnderflowException();

            bufex.initCause(
                new IllegalArgumentException(
                    "invalid message list size " + size));

            throw (bufex);
        }
        else if (size == 0)
        {
            retval = new EMessageList(key);
        }
        else
        {
            final MessageType mt =
                (MessageType)
                    DataType.findType(key.messageClass());
            int index;

            retval = new EMessageList(size, key);

            for (index = 0; index < size; ++index)
            {
                // Need to set the message type subject before
                // each call because MessageType.deserialize()
                // clears the subject before returning.
                mt.subject(subject);

                retval.add((EMessage) mt.deserialize(buffer));
            }
        }

        retval.setReadOnly();

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a message list to a
     * {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName message list field name.
     * @param indent indent the code by this white space.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%s(net.sf.eBus.messages.type.DataType.findType(net.sf.eBus.messages.EMessageList.class)).serialize(%s, buffer);%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a message list
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName message list field name.
     * @param indent indent code by this white space.
     * @param output write the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s((%s) ((net.sf.eBus.messages.type.MessageListType) net.sf.eBus.messages.type.DataType.findType(net.sf.eBus.messages.EMessageList.class)).deserialize(buffer));%n" :
             "%s%s = (%s) ((net.sf.eBus.messages.type.MessageListType) net.sf.eBus.messages.type.DataType.findType(net.sf.eBus.messages.EMessageList.class)).deserialize(buffer);%n");

        output.format(
            format, indent, fieldName, mClass.getName());

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class MessageListType
