//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2013, 2014, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.EMessage;
import net.sf.eBus.messages.EMessageKey;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * This class serializes/de-serializes a {@link EMessageKey} so
 * that it may be sent via an {@link EMessage}.
 *
 * @see ClassType
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class MessageKeyType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new message key eBus type. This is a "built-in"
     * type for eBus purposes.
     */
    /* package */ MessageKeyType()
    {
        super (EMessageKey.class, true, VARIABLE_SIZE, null);
    } // end of MessageKeyType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the message key instance to the given buffer.
     * Writes nothing if {@code o} is {@code null} or not a
     * {@link EMessageKey} instance.
     * @param o serialize this message key instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain sufficient space to
     * hold the message key.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof EMessageKey)
        {
            final EMessageKey key = (EMessageKey) o;

            CLASS_TYPE.serialize(key.messageClass(), buffer);
            STRING_TYPE.serialize(key.subject(), buffer);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the message key instance de-serialized from the
     * given buffer. May return {@code null}.
     * @param buffer read in the message key from this buffer.
     * @return the message key object.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete message key.
     */
    @Override
    @SuppressWarnings ("unchecked")
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (
            new EMessageKey(
                (Class<? extends EMessage>)
                    CLASS_TYPE.deserialize(buffer),
                (String) STRING_TYPE.deserialize(buffer)));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the message key
     * instance to a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sfinal Class _mc = (%s).messageClass();%n",
            indent,
            fieldName);
        output.format(
            "%sfinal String _subject = (%s).subject();%n",
            indent,
            fieldName);

        // Put each serializer in its own block to prevent
        // local variable name duplication.
        output.format("%s{%n", indent);
        CLASS_TYPE.createSerializer(
            field, "_mc", indent + "  ", output);
        output.format("%s}%n", indent);

        output.format("%s{%n", indent);
        STRING_TYPE.createSerializer(
            field, "_subject", indent + "  ", output);
        output.format("%s}%n", indent);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the
     * from a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized field in this
     * name.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(new net.sf.eBus.messages.EMessageKey(_mc, _subject));%n" :
             "%s%s = new net.sf.eBus.messages.EMessageKey(_mc, _subject);%n");

        output.format("%sClass _mc = null;%n", indent)
              .format("%sString _subject = null;%n", indent)
              .format("%s{%n", indent);

        // Put each serializer in its own block to prevent
        // local variable name duplication.
        CLASS_TYPE.createDeserializer(
            field, "_mc", indent + "  ", output, false);

        output.format("%s}%n", indent)
              .format("%s{%n", indent);

        STRING_TYPE.createDeserializer(
            field, "_subject", indent + "  ", output, false);

        output.format("%s}%n", indent)
              .format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class MessageKeyType
