// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright (C) 2019. Charles W. Rapp.
// All Rights Reserved.

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Formatter;
import java.util.List;
import net.sf.eBus.messages.EMessageObject;
import net.sf.eBus.messages.EReplyMessage;
import net.sf.eBus.messages.ERequestMessage;
import net.sf.eBus.messages.ValidationException;
import static net.sf.eBus.messages.type.MessageType.replyClasses;


/**
 * This type defines {@link net.sf.eBus.messages.EMessageObject}
 * sub-classes which are annotated with
 * {@link net.sf.eBus.messages.ELocalOnly}. This means that the
 * message will not be serialized for remote transmission but
 * instead is used only within a JVM. Because of this, local
 * message fields are not collected. If the underlying type is
 * a request message, then the supported reply message types are
 * collected.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class LocalMessageType
    extends MessageType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    //-----------------------------------------------------------
    // Statics.
    //

    //-----------------------------------------------------------
    // Locals.
    //

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new local message type instance for the given
     * {@code @ELocalOnly} message class and replies (if the
     * message class is an {@code ERequestMessage}). The message
     * fields will be set to an empty list.
     * @param mc message class.
     * @param replies supported reply message classes if
     * {@code mc} is an {@code ERequestMessage} sub-class.
     */
    private LocalMessageType(final Class<?> mc,
                             final List<Class<? extends EReplyMessage>> replies)
    {
        super (mc, Collections.emptyList(), replies);
    } // end of LocalMessageType(Class, List<>)

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // Abstract Method Implementations.
    //

    /**
     * Does nothing since local-only message objects are not
     * serialized or de-serialized.
     * @param o serialize this message object.
     * @param buffer place serialized message object into this
     * buffer.
     * @throws BufferOverflowException
     * is never thrown since no data is written to
     * {@code buffer}.
     * @throws UnsupportedOperationException
     * since this operation is not supported for local-only
     * message objects.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        throw (
            new UnsupportedOperationException(
                "local only objects cannot be serialized"));
    } // end of serialize(Object, ByteBuffer)

    /**
     * Does nothing since local-only message objects are not
     * serialized or de-serialized.
     * @param buffer read on serialized message from this buffer.
     * @return never returns
     * @throws BufferUnderflowException
     * is never thrown since no data is read from {@code buffer}.
     * @throws ValidationException
     * if never thrown since no message object is de-serialized.
     * @throws UnsupportedOperationException
     * since this operation is not supported for local-only
     * message objects.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException,
               ValidationException
    {
        throw (
            new UnsupportedOperationException(
                "local only objects cannot be de-serialized"));
    } // end of deserialize(ByteBuffer)

    /**
     * Does nothing since local-only message objects are not
     * serialized or de-serialized.
     * @param field generate serializer code for this field.
     * @param fieldName serialized message field.
     * @param indent indent generated code by this amount.
     * @param output place generated code into this formatter.
     * @throws UnsupportedOperationException
     * since this operation is not supported for local-only
     * message objects.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        throw (
            new UnsupportedOperationException(
                "local only objects cannot be serialized"));
    } // end of createSerializer(...)

    /**
     * Does nothing since local-only message objects are not
     * serialized or de-serialized.
     * @param field generate de-serializer code for this field.
     * @param fieldName field name.
     * @param indent indent generated code by this amount.
     * @param output place generated code into this formatter.
     * @param useBuilder if {@code true} place de-serialized
     * message object into builder; otherwise into a local
     * variable.
     * @throws UnsupportedOperationException
     * since this operation is not supported for local-only
     * message objects.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        throw (
            new UnsupportedOperationException(
                "local only objects cannot be de-serialized"));
    } // end of createDeserializer(...)

    //
    // end of Abstract Method Implementations.
    //-----------------------------------------------------------

    /**
     * Returns a local message type for the given local-only
     * {@code EMessageObject} sub-class. This message type
     * contains the reply classes list if {@code mc} is an
     * {@code ERequestMessage} sub-class. This message type does
     * <em>not</em> collect the message fields since local-only
     * messages are no serialize/de-serialized.
     * @param mc local-only message class.
     * @return local message type instance.
     */
    /* package */ static LocalMessageType createLocalMessageType(final Class<? extends EMessageObject> mc)
    {
        final List<Class<? extends EReplyMessage>> replyClasses =
            new ArrayList<>();

        // Is this a request message type?
        if ((ERequestMessage.class).isAssignableFrom(mc))
        {
            // Yes, this is a request message type. Retrieve the
            // reply message classes from the @ReplyInfo class
            // annotation.
            replyClasses(mc, replyClasses);
        }

        return (
            new LocalMessageType(
                mc,
                Collections.unmodifiableList(replyClasses)));
    } // end of createLocalMessageType(Class)
} // end of class LocalMessageType
