//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2017, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.time.LocalDate;
import java.time.chrono.ChronoLocalDate;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * Provides binary serialization/de-serialization of
 * {@code java.time.LocalDate} values.
 *
 * @author <a href="mailto:rapp@acm.org">Charles W. Rapp</a>
 */

public final class LocalDateType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

    //-----------------------------------------------------------
    // Constants.
    //

    /**
     * A {@link LocalDate} requires {@value} bytes for encoding.
     */
    public static final int BYTES = 8;

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new {@code LocalDate} data type instance.
     */
    public LocalDateType()
    {
        super (LocalDate.class, true, BYTES, null);
    } // end of LocalDateType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the {@code LocalDate.toEpochDay} value to
     * {@code buffer}.
     * @param o a {@code LocalDate} instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain enough space to store
     * the {@code long} nanoseconds value.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof LocalDate)
        {
            buffer.putLong(((ChronoLocalDate) o).toEpochDay());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the de-serialized {@code LocalDate} value by
     * converting the encode {@code long LocalDate.ofNanos} value
     * back to a {@code LocalDate} using
     * {@link LocalDate#ofEpochDay(long)}.
     * @param buffer contains the serialized {@code LocalDate}.
     * @return the de-serialized {@code LocalDate}.
     * @throws BufferUnderflowException
     * if {@code buffer} does not contain the complete
     * {@code long} nanoseconds value.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        return (LocalDate.ofEpochDay(buffer.getLong()));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize a {@code LocalDate}
     * instance to a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code LocalDate} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sbuffer.putLong(%s.toEpochDay());%n",
            indent,
            fieldName);

        return;
    } // end of createSerializer(...)

    /**
     * Adds the Java code used to de-serialize a
     * {@code LocalDate} instance from a {@code ByteBuffer}.
     * @param field message field.
     * @param fieldName the {@code LocalDate} message field name.
     * @param indent indent code by this many spaces.
     * @param output output code to here.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.time.LocalDate.ofEpochDay(buffer.getLong()));%n" :
             "%sbuilder.%s(java.time.LocalDate.ofEpochDay(buffer.getLong()));%n");

        output.format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class LocalDateType
