//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2014, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * This class serializes and de-serializes an
 * {@link InetSocketAddress} which may contain either an
 * {@link java.net.Inet4Address IPv4} or
 * {@link java.net.Inet6Address IPv6} address.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class InetSocketAddressType
    extends DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new Internet address and port data type
     * serializer.
     */
    /* package */ InetSocketAddressType()
    {
        super (InetSocketAddress.class, true, VARIABLE_SIZE, true);
    } // end of InetSocketAddressType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the Internet address and TCP port to the given
     * buffer. Writes nothing if {@code o} is {@code null} or is
     * not an {@code java.net.InetSocketAddress} instance.
     * @param o serialize this socket address instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain sufficient space to
     * hold the socket address.
     * @see InetAddressType
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof InetSocketAddress)
        {
            final InetSocketAddress address =
                (InetSocketAddress) o;

            ADDRESS_TYPE.serialize(address.getAddress(), buffer);
            buffer.putInt(address.getPort());
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the socket address instance de-serialized from the
     * given buffer.
     * @param buffer read in the Internet address and port from
     * this buffer.
     * @return the socket address object.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete Internet socket
     * address.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final InetAddress addr =
            (InetAddress) ADDRESS_TYPE.deserialize(buffer);
        final int port = buffer.getInt();

        return (new InetSocketAddress(addr, port));
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the socket address to
     * a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sfinal java.net.InetAddress addr = (%s).getAddress();%n",
            indent,
            fieldName);
        output.format("%sfinal int port = (%s).getPort();%n",
                      indent,
                      fieldName);
        ADDRESS_TYPE.createSerializer(field, "addr", indent, output);
        output.format("%sbuffer.putInt(port);%n", indent);

        return;
    } // end of createSerializer(...)

    /**
     * Generates the code to de-serialize the
     * {@code InetSocketAddress} instance from a
     * {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName store the de-serialized address in this
     * field.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} then {@code fieldName}
     * is a builder method name; otherwise a local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String format =
            (useBuilder ?
             "%sbuilder.%s(new java.net.InetSocketAddress(_addr, _port));%n" :
             "%s%s = new java.net.InetSocketAddress(_addr, _port);%n");

        output.format("%sjava.net.InetAddress _addr = null;%n", indent)
              .format("%sfinal int _port;%n", indent);
        ADDRESS_TYPE.createDeserializer(
            field, "_addr", indent, output, false);
        output.format("%s_port = buffer.getInt();%n", indent)
              .format(format, indent, fieldName);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class InetSocketAddressType
