//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later
// version.
//
// This library is distributed in the hope that it will be
// useful, but WITHOUT ANY WARRANTY; without even the implied
// warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
// PURPOSE. See the GNU Lesser General Public License for more
// details.
//
// You should have received a copy of the GNU Lesser General
// Public License along with this library; if not, write to the
//
// Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330,
// Boston, MA
// 02111-1307 USA
//
// The Initial Developer of the Original Code is Charles W. Rapp.
// Portions created by Charles W. Rapp are
// Copyright 2014, 2019. Charles W. Rapp
// All Rights Reserved.
//

package net.sf.eBus.messages.type;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.util.Formatter;
import net.sf.eBus.messages.type.MessageType.MessageField;

/**
 * This class is able to serialize and de-serialize an
 * {@link InetAddress} which may be either an
 * {@link java.net.Inet4Address IPv4} or
 * {@link java.net.Inet6Address IPv6} address.
 *
 * @author <a href="mailto:rapp@acm.org">Charles Rapp</a>
 */

public final class InetAddressType
    extends  DataType
{
//---------------------------------------------------------------
// Member data.
//

//---------------------------------------------------------------
// Member methods.
//

    //-----------------------------------------------------------
    // Constructors.
    //

    /**
     * Creates a new Internet address data type serializer.
     * Supports both IPv4 and IPv6 address types.
     */
    /* package */ InetAddressType()
    {
        super (InetAddress.class, true, VARIABLE_SIZE, true);
    } // end of InetAddressType()

    //
    // end of Constructors.
    //-----------------------------------------------------------

    //-----------------------------------------------------------
    // DataType Abstract Method Overrides.
    //

    /**
     * Serializes the Internet address to the given buffer.
     * Writes nothing if {@code o} is {@code null} or is not an
     * {@code java.net.InetAddress} instance. Serializes the
     * raw address byte array to the buffer.
     * @param o serialize this address instance.
     * @param buffer write to this buffer.
     * @throws BufferOverflowException
     * if {@code buffer} does not contain sufficient space to
     * hold the address bytes.
     */
    @Override
    public void serialize(final Object o,
                          final ByteBuffer buffer)
        throws BufferOverflowException
    {
        if (o instanceof InetAddress)
        {
            final byte[] addr = ((InetAddress) o).getAddress();

            buffer.put((byte) addr.length).put(addr);
        }

        return;
    } // end of serialize(Object, ByteBuffer)

    /**
     * Returns the IPv4 or IPv6 Internet address instance
     * de-serialized from the given buffer. May return
     * {@code null}.
     * @param buffer read in the Internet address from this
     * buffer.
     * @return the Internet address object.
     * @throws BufferUnderflowException
     * if {@code buffer} contains an incomplete Internet address
     * instance.
     */
    @Override
    public Object deserialize(final ByteBuffer buffer)
        throws BufferUnderflowException
    {
        final int addrSize = buffer.get();
        final byte[] addr = new byte[addrSize];
        Object retval = null;

        buffer.get(addr);

        try
        {
            retval = InetAddress.getByAddress(addr);
        }
        catch (UnknownHostException jex)
        {
            // Ignore and return null.
        }

        return (retval);
    } // end of deserialize(ByteBuffer)

    /**
     * Adds the Java code used to serialize the Internet address
     * to a {@link ByteBuffer}.
     * @param field message field.
     * @param fieldName the message field name.
     * @param indent indent the code by this amount.
     * @param output write the code to this formatter.
     */
    @Override
    protected void createSerializer(final MessageField field,
                                    final String fieldName,
                                    final String indent,
                                    final Formatter output)
    {
        output.format(
            "%sfinal byte[] a = (%s).getAddress();%n",
            indent,
            fieldName);

        output.format(
            "%sbuffer.put((byte) a.length).put(a);%n",
            indent);

        return;
    } // end of createSerialize(...)

    /**
     * Generates the class de-serialization code storing the
     * result according to the {@code useBuilder} flag. If
     * {@code true}, then store the result into the message/field
     * builder. Otherwise store the result into the named local
     * variable.
     * @param field message field.
     * @param fieldName store result into this field.
     * @param indent indent the code by this amount.
     * @param output append the code to this formatter.
     * @param useBuilder if {@code true} store de-serialized
     * Internet address into the message/field builder; if
     * {@code false} store result into named local variable.
     */
    @Override
    protected void createDeserializer(final MessageField field,
                                      final String fieldName,
                                      final String indent,
                                      final Formatter output,
                                      final boolean useBuilder)
    {
        final String indent1 = indent + INDENT;
        final String format =
            (useBuilder ?
             "%sbuilder.%s(java.net.InetAddress.getByAddress(_a));%n" :
             "%s%s = java.net.InetAddress.getByAddress(_a);%n");

        output.format("%sfinal int _addrSize = (int) buffer.get();%n",
                      indent)
              .format("%sfinal byte[] _a = new byte[_addrSize];%n%n",
                      indent)
              .format("%sbuffer.get(_a, 0, _addrSize);%n", indent)
              .format("%stry {%n", indent)
              .format(format, indent1, fieldName)
              .format("%s} catch (Exception jex) {}%n", indent);

        return;
    } // end of createDeserializer(...)

    //
    // end of DataType Abstract Method Overrides.
    //-----------------------------------------------------------
} // end of class InetAddressType
